/* erate_evolve.c
 *
 *
 * ER, Fri Feb 29 09:31:10 EST 2008
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>

#include <easel.h>
#include <esl_alphabet.h>
#include <esl_getopts.h>
#include <esl_dmatrix.h>
#include <esl_msa.h>
#include <esl_sqio.h>
#include <esl_tree.h>
#include <esl_vectorops.h>
#include <esl_wuss.h>

#include "erate_evolve.h"

static int add_residue(ESL_RANDOMNESS *r, ERATE *E, char *achar);
static int erate_evolve_root(ESL_RANDOMNESS *r, ERATE *erate, ESL_TREE *T,
			     ESL_MSA *msa, int *ret_idx, int *nidx, int be_verbose);
static int erate_evolve_ascendant_to_descendant(ESL_RANDOMNESS *r, int *ret_idx, int *nidx,
						int p, int d, double time, 
						ERATE *E, ESL_TREE *T, 
						ESL_MSA *msa, int be_verbose);
static int erate_generate_root(ESL_RANDOMNESS *r, ERATE *erate, ESL_MSA *msa, 
			       int *ret_idx, int *nidx); 
static int erate_substiteordelete(ESL_RANDOMNESS *r, int col, int p, int d, 
				  TERATE *TE, ESL_MSA *msa, int be_verbose);
static int erate_insert(ESL_RANDOMNESS *r, int col, int p, int d, 
			ERATE *E, TERATE *TE, ESL_MSA *msa, int be_verbose);
static int erate_substiteordelete_res(ESL_RANDOMNESS *r, double *m, int col, int d,
				      int dim, ESL_MSA *msa, int be_verbose);
static int maxwidth(char **s, int n);
static int actually_write_phylip(FILE *fp, const ESL_MSA *msa, int cpl);

/* Function:  Erate_Create()
 *
 * Purpose:   Creates the evolutionary  model
 *
 * Args:      
 *
 * Returns:   E, allocated here.
 */
ERATE *
Erate_Create(int dim, double ttr, double ins, double del, double bernoulli, double *frq, int isfixlen, char *errbuf)
{
  ERATE *E;
  double frqr, frqy;
  double frqpr, frqpy;
  double aa, bb;
  double tol = 0.0000001;

  E = malloc(sizeof(ERATE));

  E->ttr       = ttr;
  E->ins       = ins;
  E->del       = del;
  E->bernoulli = bernoulli;
  E->isfixlen  = isfixlen;
 
  /* the residue frequencies */
  E->dim = dim;
  E->frq = malloc(sizeof(double) * dim);
  esl_vec_DCopy(frq, dim, E->frq);
  esl_vec_DValidate(E->frq, dim, tol, errbuf);

  /* alpha and beta as a fuction of ttr and the
   * residue frequencies 
   */
  frqr  = E->frq[0] + E->frq[2];
  frqy  = E->frq[1] + E->frq[3];
  frqpr = E->frq[0] * E->frq[2];
  frqpy = E->frq[1] * E->frq[3];

  aa = ttr*frqr*frqy - frqpr - frqpy;
  bb = frqpr/frqr + frqpy/frqy;

  E->alpha = aa / (aa + bb);
  E->beta = 1.0 - E->alpha;

  return E;
}

void   
Erate_Destroy(ERATE *erate)
{
  free(erate->frq);
  free(erate);
}

void   
Erate_Dump(FILE *fp, ERATE *E)
{
  fprintf(fp, "\nEvolutionary model including gaps (F84 substitutions)\n");
  fprintf(fp, "transitions to transversions ratio = %f\n", E->ttr);
  fprintf(fp, "alpha = %f\n", E->alpha);
  fprintf(fp, "beta  = %f\n", E->beta);
  fprintf(fp, "rate insertions = %f\n", E->ins);
  fprintf(fp, "rate deletions  = %f\n", E->del);
  fprintf(fp, "bernoulli probability parameter = %f\n", E->bernoulli);
  fprintf(fp, "residue frequencies  = %f %f %f %f\n", 
	  E->frq[0], E->frq[1], E->frq[2], E->frq[3]);

}


int Erate_Generate_Alignment(ERATE *erate, ESL_TREE *T, ESL_MSA **ret_msa, int be_verbose) 
{
  ESL_RANDOMNESS *r   = NULL;
  ESL_MSA        *msa = NULL; /* alignment of leaf and node sequences */
  int             idx;        /* node index */
  int            *nidx;       /* node index array */
  int             nnode;
  int             n;
  int             status;

  r = esl_randomness_CreateTimeseeded();

  /* indexing of internal nodes */
  idx = 0;
  ESL_ALLOC(nidx, sizeof(int) * T->N-1);
  for(n = 0; n < T->N-1; n ++)
    nidx[n] = -1;

  /* create the alignment */
  nnode = (T->N > 1)? T->N-1 : T->N;
  msa = esl_msa_Create(nnode+T->N, 0);

  status = esl_strdup("whole", -1, &(msa->name));

  /* create the root sequence */
  if (erate_generate_root(r, erate, msa, &idx, nidx) != eslOK)
    esl_fatal("failed to generate root sequence");
  
  /* evolve root */
  if (erate_evolve_root(r, erate, T, msa, &idx, nidx, be_verbose)!= eslOK)
    esl_fatal("failed to evolve root");
  
  *ret_msa = msa;
  
  free(nidx);
  esl_randomness_Destroy(r);
  
  return eslOK;
  
 ERROR:
  if (nidx != NULL) free(nidx);
  if (r    != NULL) esl_randomness_Destroy(r);
  return status;
}


TERATE 
*TErate_Create(ERATE *E, double time)
{
  TERATE  *TE;
  double   term1, term2, term3, term4;
  double   frqr, frqy;
  double   frqar, frqgr;
  double   frqcy, frqty;
  double   sum;
  double   tol = 0.000001;
  int      i, j;

  TE = malloc(sizeof(TERATE));

  TE->dim = E->dim;
  TE->p   = E->bernoulli;

  TE->M = esl_dmatrix_Create(TE->dim, TE->dim);
  
  TE->t = time;

  /* insertions */
  if      (E->del > 0.0 && E->ins > 0.0) TE->xit = E->ins/(E->ins+E->del)*(1.0 - exp(-(E->ins+E->del) * time));
  else if (E->ins > 0.0)                 TE->xit = 1.0 - exp(-(E->ins) * time);
  else if (E->del > 0.0)                 TE->xit = 0.0;
  else                                   TE->xit = 0.0;
  if (TE->xit < 0.0 || TE->xit > 1.0) esl_fatal("bad xit");
  if (TE->xit > 1.0 || TE->xit < 0.0) esl_fatal("bad xit");

  /* deletions */
  if      (E->del > 0.0 && E->ins > 0.0) TE->gammat = E->del/(E->ins+E->del)*(1.0 - exp(-(E->ins+E->del) * time));
  else if (E->del > 0.0)                 TE->gammat = 1.0 - exp(-(E->del) * time);
  else if (E->ins > 0.0)                 TE->gammat = 0.0;
  else                                   TE->gammat = 0.0;

  /* substitutions */
  frqr = E->frq[0] + E->frq[2];
  frqy = E->frq[1] + E->frq[3];

  if (frqr > 0.0) {
    frqar = E->frq[0]/frqr;
    frqgr = E->frq[2]/frqr;
  }
  else {
    frqar = 0.0;
    frqgr = 0.0;
   }

  if (frqy > 0.0) {
    frqcy = E->frq[1]/frqy;
    frqty = E->frq[3]/frqy;
  }
  else {
    frqcy = 0.0;
    frqty = 0.0;
   }

  for (i = 0; i < TE->M->n; i ++) {
    
    sum = 0.0;
    for (j = 0; j < TE->M->m; j ++) {
      if (E->del > 0.0 && E->ins > 0.0) term1 = E->ins/(E->ins+E->del)*E->frq[j];
      else if (E->del > 0.0)            term1 = 0.0;
      else if (E->ins > 0.0)            term1 = E->frq[j];
      else                              term1 = E->frq[j];
      
      if      (i == 0 && j == 0) term2 = frqar - E->frq[0];
      else if (i == 0 && j == 2) term2 = frqgr - E->frq[2];
      else if (i == 2 && j == 0) term2 = frqar - E->frq[0];
      else if (i == 2 && j == 2) term2 = frqgr - E->frq[2];
      else if (i == 1 && j == 1) term2 = frqcy - E->frq[1];
      else if (i == 1 && j == 3) term2 = frqty - E->frq[3];
      else if (i == 3 && j == 1) term2 = frqcy - E->frq[1];
      else if (i == 3 && j == 3) term2 = frqty - E->frq[3];
      else                       term2 = -E->frq[j];
      term2 *= exp(-(E->beta+E->del)* time);
      
      if      (i == 0 && j == 0) term3 = 1.0 - frqar;
      else if (i == 0 && j == 2) term3 = -frqgr;
      else if (i == 2 && j == 0) term3 = -frqar;
      else if (i == 2 && j == 2) term3 = 1.0 - frqgr;
      else if (i == 1 && j == 1) term3 = 1.0 - frqcy;
      else if (i == 1 && j == 3) term3 = -frqty;
      else if (i == 3 && j == 1) term3 = -frqcy;
      else if (i == 3 && j == 3) term3 = 1.0 - frqty;
      else if (i == j)           term3 = 1.0;
      else                       term3 = 0.0;
      term3 *= exp(-(E->alpha+E->beta+E->del)* time);
      
      if (E->del > 0.0 && E->ins > 0.0) term4 = E->del/(E->ins+E->del)*E->frq[j]*exp(-(E->ins+E->del) * time);
      else if (E->del > 0.0)            term4 = E->frq[j]*exp(-(E->del) * time);
      else if (E->ins > 0.0)            term4 = 0.0;
      else                              term4 = 0.0;
      
      if (isnan(term1) || isnan(term2) || isnan(term3) || isnan(term4))
	esl_fatal("F84 with indels has nan terms");
      
       TE->M->mx[i][j] = term1 + term2 + term3 + term4;
       if (TE->M->mx[i][j] < 0.0)  {
	 if (TE->M->mx[i][j] > -tol )
	   TE->M->mx[i][j] = 0.0;
	 else 
	   esl_fatal("F84 with indels has negative substitution probabilities %f", TE->M->mx[i][j]);
       }
       if (TE->M->mx[i][j] > 1.0)
	esl_fatal("F84 with indels has bad substitution probabilities %f", TE->M->mx[i][j]);
      
      sum += TE->M->mx[i][j];
      /* paranoia */
      if (sum-1.0 > tol) esl_fatal("bad conditional model for substitutions, i=%d sum %f", i, sum); 
    }

    /* paranoia */
    if (fabs(sum + TE->gammat-1.0) > tol) 
      esl_fatal("bad conditional model for substitutions/deletions, i=%d sum %f", i, sum+TE->gammat); 
  }


  return TE;
}
void    
TErate_Destroy(TERATE *TE)
{
  esl_dmatrix_Destroy(TE->M);
  free(TE);
}

void    
TErate_Dump(FILE *fp, TERATE *TE)
{
  int i, j;
  
  fprintf(fp, "\nEvolutionary model including gaps (F84 substitutions)\n");
  fprintf(fp, "Functions for divergence time %f\n", TE->t);
  
  fprintf(fp, "Insertions fuction xi(t) %f\n", TE->xit);
  fprintf(fp, "Deletions fuction gamma(t) %f\n", TE->gammat);
  
  fprintf(fp, "SUBSTITUTIONS MATRIX\n");
  for (i = 0; i < TE->M->m; i ++) {
    for (j = 0; j < TE->M->n; j ++) {
      fprintf(fp, "%f ", TE->M->mx[i][j]);
    }
    fprintf(fp, "\n");
  }  
}


/* Write_Phylip():
 * ER, Tue Mar  4 10:22:25 EST 2008 [Janelia]
 *
 * Purpose:   Write an alignment <msa> in Phylip format 
 *            to a stream <fp>, in multiblock format, with
 *            50 residues per line.
 *
 * Returns:   <eslOK> on success.
 *
 * Throws:    <eslEMEM> on allocation failure.
 *
 */
int
Write_Phylip(FILE *fp, const ESL_MSA *msa)
{
  return (actually_write_phylip(fp, msa, 50)); /* 50 char per block */
}


/* local functions */
int
add_residue(ESL_RANDOMNESS *r, ERATE *E, char *achar)
{
  double          frq = 0.0;
  double          x;
  int             n;

  x = esl_random(r);
  for (n = 0; n < E->dim; n++) {
    frq += E->frq[n];
    if (frq > x) break;
  }
  
  switch(n) {
  case 0: 
    *achar = 'A';
    break;
  case 1: 
    *achar = 'C';
    break;
  case 2: 
    *achar = 'G';
    break;
  case 3: 
    *achar = 'T';
    break;
  default: printf("not such residue\n"); return eslFAIL;
  }

  return eslOK;
}

int
erate_evolve_root(ESL_RANDOMNESS *r, ERATE *E, ESL_TREE *T, ESL_MSA *msa, int *ret_idx, int *nidx, int be_verbose)
{
  int    v;       /* index for internal nodes */
  int    dl, dr;
  double ld, rd;

  if (0) {
    printf("\nTHE TREE\n");
    for (v = 0; v < T->N-1; v++) {
      printf("v %d[%s] left %d[%s] right %d[%s] \n", 
	     v, T->nodelabel[v], 
	     T->left[v],  (T->left[v]>0)?  T->nodelabel[T->left[v]]  : T->taxonlabel[-T->left[v]], 
	     T->right[v], (T->right[v]>0)? T->nodelabel[T->right[v]] : T->taxonlabel[-T->right[v]]);
    }
  }

  for (v = 0; v < T->N-1; v++) {
    dl = T->left[v];
    dr = T->right[v];

    ld = T->ld[v];
    rd = T->rd[v];
    
    if (erate_evolve_ascendant_to_descendant(r, ret_idx, nidx, v, dl, ld, E, T, msa, be_verbose) != eslOK)
      esl_fatal("failed to evolve from p=%d to daughther %d after time %f", v, dl, ld);
    if (erate_evolve_ascendant_to_descendant(r, ret_idx, nidx, v, dr, rd, E, T, msa, be_verbose) != eslOK)
      esl_fatal("failed to evolve from p=%d to daughther %d after time %f", v, dr, rd);

    if (be_verbose) esl_msa_Write(stdout, msa, eslMSAFILE_STOCKHOLM);
  }

  return eslOK;
}

int
erate_evolve_ascendant_to_descendant(ESL_RANDOMNESS *r, int *ret_idx, int *nidx,
				     int p, int d, double time, 
				     ERATE *E, ESL_TREE *T, ESL_MSA *msa, 
				     int be_verbose)
{
  TERATE *TE;
  int  u;
  int  idx;
  int  d_idx;
  int  i;
  int  status;

  TE = TErate_Create(E, time);
  if (FALSE) TErate_Dump(stdout, TE);

  idx = *ret_idx;
  /* generate the descendant sequence */
  if (d > 0) {
    d_idx   = idx++;
    nidx[d] = d_idx;
  }
  else {
    d_idx = T->N - 1 - d;
  }

  ESL_ALLOC(msa->sqname[d_idx], sizeof(char) * MAXNAME);
  ESL_ALLOC(msa->aseq[d_idx], sizeof(char) * (msa->alen+1));
  msa->sqlen[d_idx] = msa->alen;

  if (d <= 0) sprintf(msa->sqname[d_idx], "%s", T->taxonlabel[-d]);
  else        sprintf(msa->sqname[d_idx], "v%d", d);

  /* SUBSTITUTIONS/DELETIONS
   * Each residue is either substituted or deleted.
   * Gaps are propagated as gaps.
   */
  for (u = 0; u < msa->alen; u ++) 
    if (erate_substiteordelete(r, u, nidx[p], d_idx, TE, msa, be_verbose) != eslOK)
      { printf("residue did not evolve properly\n"); return eslFAIL; }
  /* esl_msa_Write(stdout, msa, eslMSAFILE_STOCKHOLM); */
  
  /* INSERTIONS
   * Each link puts an insertion, possibly of length zero.
   * A link is defined after a residue, not after a gap.
   * Assign u=-1 for the immortal link insertion.
   */
  if (erate_insert(r, -1, nidx[p], d_idx, E, TE, msa, be_verbose) != eslOK)
    { printf("bad insertion\n"); return eslFAIL; }
  /* esl_msa_Write(stdout, msa, eslMSAFILE_STOCKHOLM); */
  
  for (u = 0; u < msa->alen; u ++) { 
    if (erate_insert(r, u, nidx[p], d_idx, E, TE, msa, be_verbose) != eslOK)
      { printf("bad insertion\n"); return eslFAIL; }
    /* esl_msa_Write(stdout, msa, eslMSAFILE_STOCKHOLM); */
  }
  
  /* check */
  for (i = 0; i < idx; i ++)
    if (msa->alen != strlen(msa->aseq[i]))
      esl_fatal("bad msa alen %d seq %d len %d\n", msa->alen, i, msa->aseq[i]);
  
  *ret_idx = idx;

  TErate_Destroy(TE);
  return eslOK;
  
 ERROR:
  if (TE != NULL) TErate_Destroy(TE);
  return status;
}

int 
erate_generate_root(ESL_RANDOMNESS *r, ERATE *E, ESL_MSA *msa, int *ret_idx, int *nidx) 
{
  double          pdf;
  double          fac;
  double          p;
  double          x;
  int             idx;
  int             L, l;
  int             status;

  idx     = *ret_idx;
  nidx[0] = idx++;

  p = E->bernoulli;

  ESL_ALLOC(msa->sqname[nidx[0]], sizeof(char) * MAXNAME);
  sprintf(msa->sqname[nidx[0]], "v%d", nidx[0]);
 
  if (E->isfixlen == TRUE) { 
    /* fixed root length */ 
    L = (int)(p / (1.0 - p));
  }
  else { 
    /* sample the root length */    
    L = 0;
    pdf = (1.0 - p);
    fac = pdf;
    
    x = esl_random(r);
    while (pdf < x) {
      L ++;
      fac *= p;
      pdf += fac;
    }
  }
  
  /* set the root length */
  msa->sqlen[nidx[0]] = L;
  msa->alen           = msa->sqlen[nidx[0]];

  /* generate the root sequence */
  ESL_ALLOC(msa->aseq[nidx[0]], sizeof(char) * (msa->alen+1));

  for (l = 0; l < L; l++) 
    if (add_residue(r, E, &(msa->aseq[nidx[0]][l])) != eslOK) return eslFAIL;
  msa->aseq[nidx[0]][L] = '\0';

  *ret_idx = idx;
  return eslOK;

 ERROR:
  return status;
}

int
erate_insert(ESL_RANDOMNESS *r, int col, int p, int d, ERATE *E, TERATE *TE, ESL_MSA *msa, int be_verbose)
{
  void   *pt;
  double  q;
  double  pdf;
  double  fac;
  double  x;
  int     newalen;
  int     l;    /* length of the sampled insertion */
  int     n;
  int     i;
  int     status;

  /* no link after a gap in the ascendant sequence */
  if (col >= 0) if (msa->aseq[p][col] == '-') return eslOK;

  /* the geometric factor for insertions */
  q = TE->xit;
  if (q > 1.0 || q < 0.0) esl_fatal("bad q");
  if (isnan(q)) 
    esl_fatal("insertions have nan terms");

  l = 0;
  pdf = (1.0 - q);
  fac = pdf;

  x = esl_random(r);
  while (pdf < x) {
    l ++;
    fac *= q;
    pdf += fac;
  }

  /* Extend the alignment by l columns after 'col'. 
     This also works for the immmortal link for which col = -1
   */
  newalen = msa->alen + l;
  for (i = 0; i < msa->nseq; i ++) {
    if (msa->aseq[i] != NULL) 
      ESL_RALLOC(msa->aseq[i], pt, sizeof(char) * (newalen+1));
    
    for (n = msa->alen-1; n > col; n--)
      msa->aseq[i][n+l] = msa->aseq[i][n];

    for (n = 1; n <= l; n ++)
      if (i == d) add_residue(r, E, &(msa->aseq[i][col+n]));
      else        msa->aseq[i][col+n] = '-';

    msa->sqlen[i] = newalen;
    msa->aseq[i][newalen] = '\0';
  }
  msa->alen = newalen; 

  if (l > 0) printf("\nInsertion of %d residues after col %d seq %d->%d. newalen %d\n", l, col, p, d, newalen);
 
  return eslOK;

 ERROR:
  return status;
}

int
erate_substiteordelete(ESL_RANDOMNESS *r, int col, int p, int d, TERATE *TE, ESL_MSA *msa, int be_verbose)
{
  if (msa->aseq[p][col] == '-') {
    msa->aseq[d][col] = '-'; 
    return eslOK;
  }

  switch(msa->aseq[p][col]) {
  case 'A': 
    erate_substiteordelete_res(r, TE->M->mx[0], col, d, TE->dim, msa, be_verbose);
    break;
  case 'C': 
    erate_substiteordelete_res(r, TE->M->mx[1], col, d, TE->dim, msa, be_verbose);
    break;
  case 'G': 
    erate_substiteordelete_res(r, TE->M->mx[2], col, d, TE->dim, msa, be_verbose);
    break;
  case 'T': 
    erate_substiteordelete_res(r, TE->M->mx[3], col, d, TE->dim, msa, be_verbose);
    break;
  case '-': 
    msa->aseq[d][col] = '-'; 
    break;
  default: esl_fatal("seq %d col %d what is this character? %c", p, col, msa->aseq[p][col]);
  }

  return eslOK;
}

int 
erate_substiteordelete_res(ESL_RANDOMNESS *r, double *M, int col, int d, int dim, ESL_MSA *msa, int be_verbose)
{
  double  pdf = 0.0;
  double  x;
  int     des;

  x = esl_random(r);
 
  for (des = 0; des < dim; des++) {
    pdf += M[des];
    if (pdf > x) break;
  }
  
  switch(des) {
  case 0: 
    msa->aseq[d][col] = 'A';
    break;
  case 1: 
    msa->aseq[d][col] = 'C';
    break;
  case 2: 
    msa->aseq[d][col] = 'G';
    break;
  case 3: 
    msa->aseq[d][col] = 'T';
    break;
  case 4: /* delete */
    msa->aseq[d][col] = '-';
    break;
  default: printf("not such case!"); return eslFAIL;
  }

  return eslOK;  
}

/* maxwidth()
 * Return the length of the longest string in 
 * an array of strings.
 */
static int
maxwidth(char **s, int n)
{
  int max, i, len;
  
  max = 0;
  for (i = 0; i < n; i++)
    if (s[i] != NULL)
      {
	len = strlen(s[i]);
	if (len > max) max = len;
      }
  return max;
}

/* actually_write_phylip()
 * ER, Tue Mar  4 10:26:44 EST 2008 [Janelia]
 *
 * Write an alignment in Phylip format to an open file. This is the
 * function that actually does the work. The API's Write_Phylip()
 * is a wrapper.
 *
 * Args:     fp    - file that's open for writing
 *           msa   - alignment to write        
 *           cpl   - characters to write per line in alignment block
 *
 * Returns:  eslOK on success.
 * 
 * Throws:   eslEMEM on allocation failure.
 */
static int
actually_write_phylip(FILE *fp, const ESL_MSA *msa, int cpl)
{
  int  maxname;		/* maximum name length     */
  int  margin;        	/* total left margin width */
  int  i;
  char *buf = NULL;
  int  currpos;
  int  acpl;            /* actual number of character per line */
  int  status;
  
  /* Figure out how much space we need for name 
   * to keep the alignment in register.
   *
   * The left margin of an alignment is 10 in phylip format.
   * seqnames have to be at most 10 characters long.
   *
   */
  maxname = maxwidth(msa->sqname, msa->nseq);
  margin  = 10;
  if (maxname > margin) esl_fatal("names are too long for phylip format\n");

  /* Allocate a tmp buffer to hold sequence chunks in
   */
  ESL_ALLOC(buf, sizeof(char) * (cpl+1));

  /* Phylip first line
   */
  fprintf(fp, " %d %" PRId64 "\n", msa->nseq, msa->alen);

  /* Alignment section:
   */
  for (currpos = 0; currpos < msa->alen; currpos += cpl)
    {
      acpl = (msa->alen - currpos > cpl)? cpl : msa->alen - currpos;

      if (currpos > 0) fprintf(fp, "\n");
      for (i = 0; i < msa->nseq; i++)
	{
#ifdef eslAUGMENT_ALPHABET
	  if (msa->flags & eslMSA_DIGITAL)
	    esl_abc_TextizeN(msa->abc, msa->ax[i] + currpos + 1, acpl, buf);
	  else
	    strncpy(buf, msa->aseq[i] + currpos, acpl);
#else
	  strncpy(buf, msa->aseq[i] + currpos, acpl);
#endif
	  
	  buf[acpl] = '\0';	      
	  if (currpos == 0) fprintf(fp, "%-*s %s\n", margin, msa->sqname[i], buf);
	  else              fprintf(fp, "%-*s %s\n", margin, "", buf);
	}
      
    }
  fprintf(fp, "\n");
  free(buf);
  return eslOK;
  
 ERROR:
  if (buf != NULL) free(buf);
  return status;
}


