/*
 * $Id: mxparser.c,v 1.13 2000/04/17 16:20:24 dirk Exp $
 */

/* #define DEBUG */

/* includes */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef HAVE_STDIO_H
#include <stdio.h>
#endif
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif


#include "defs.h"
#include "mxerror.h"
#include "global.h"
#include "mxparser.h"
#include "mygen.h"


/* extern vars */
extern char yytext[];
extern FILE * yyin;
extern int pc;
extern tagvalue_t pp[100];

/* global vars */
char *absfilename;

/* prototypes */
tagvalue_t *find_tag(const char *str);
int test_typ(const tagvalue_t *tv, const int typ);

int yyparse (void);
int yylex(void);
int yyerror (char * s);
void print_tree(node *n);
void print_subtree(node *n);
void repointer(node *father, node *son);


int
yyerror (char *s) {

  int first = TRUE;
  int i;

  MXDEBUG(DB_DEBUG-1 MXDELIM "-> yyerror");

  for(i=0;i<MAX_INCLUDE_DEPTH;i++) {
    if(filenames[i+1]!= NULL) {
      if(first == TRUE) {
	fprintf(stderr, "In file included from %s:%d:\n",
		filenames[i], line_num[i]);
	first = FALSE;
      }
      else
	fprintf(stderr, "                 from %s:%d:\n",
		filenames[i], line_num[i]);
    }
    else
      break;
  }

  fprintf(stderr, "%s:%d: %s\n",
	  filenames[i], line_num[i], s);

  exit(EXIT_FAILURE);
  return 0; /* never gets here */
}

/*
int
main(int argc, char *argv[]) {


  mxDebugLevel=DB_DEBUG;

  AddParmInt("Int", 5);
  AddParmIntVec("IntVec", NULL, 0);
  AddParmIntMat("IntMat", NULL, 0, 0);
  AddParmString("String", "Hello world!");
  AddParmDouble("Double", 12.3);
  AddParmFloat("Float", 12.3);
  AddParmDoubleVec("FloatVec", NULL, 0);
  AddParmDoubleMat("DoubleMat", NULL, 0, 0);
  AddParmBool("Boolean", FALSE);
  AddParmTree("Tree", NULL);


  MXDEBUG(DB_DEBUG-1 MXDELIM "before PrintAllParms");
  PrintAllParms();
  MXDEBUG(DB_DEBUG-1 MXDELIM "before ParseCmd");
  ParseCmd(argc, argv);
  MXDEBUG(DB_DEBUG-1 MXDELIM "before 2. PrintAllParms");
  PrintAllParms();
  return 0;
}
*/

int
ParseFile(const char *file) {

  int i;
  FILE *stream = NULL;

  stream = fopen(file, "r");
  if(stream) {

  MXDEBUG(DB_DEBUG-1 MXDELIM "beginning to parse...");
  
    /* initialize line_numbering */
    for(i=0;i<MAX_INCLUDE_DEPTH;i++)
      line_num[i] = 1;
    
    yyin = stream;
    yyparse();
  }
  else
    return 1;

  /* success */
  yyin = NULL;
  fclose(stream);

  return 0;
}


#define PC_USAGE "Usage: %s -v -h -I <include path> <inputfile> | -"

int
ParseCmd(int argc, char *argv[]) {

  int c, i;
  FILE *stream = NULL;

  opterr = 0; /* tell getopt to be quiet */

  /* check options */
  while ((c = getopt(argc, argv, "vhI:d:")) != EOF)
    switch (c) {
    case 'v':
      printf(PACKAGE " Version: " VERSION "\n");
      exit(EXIT_SUCCESS);
      break;
    case 'h':
      printf("Most of the following parameters may be set in the input file.\n");
      printf("(Some are computed automatically.)\n");
      printf("Please consult the documentation for details.\n");
      ListAllParms();
      printf("\n" PC_USAGE "\n", argv[0]);
      printf("Options:\n");
      printf("\t-v : Display package name and version number.\n");
      printf("\t-h : Display this help message.\n");
      printf("\t-I <include path> : Include path for input files.\n");
      printf("Arguments:\n");
      printf("\t<input file> : File containing parameter list.\n");
      printf("\t- : Parameter list is supplied on the standard input stream.\n");
      exit(EXIT_SUCCESS);
      break;
    case 'I':
      include_path = strdup(optarg);
      break;
    case 'd':
      mxDebugLevel=atoi(optarg);
      break;
    default:
      mxError(DO_EXIT, PC_USAGE, argv[0]);
    }
  

  if(optind >= argc)
    mxError(DO_EXIT, PC_USAGE, argv[0]);
  else
    if(!strcmp(argv[optind], "-")) { /* check cmdline args for stdin "-" */
      stream = stdin;
      filenames[0] = "-stdin-";
    }
    else {
      absfilename = mypathfind(include_path, argv[optind], "rfs");
      if(!absfilename) /* if its not in the path, assume it's in CWD */
	absfilename = argv[optind];
      stream = fopen(absfilename, "r"); /* it's a filename */
      if(!stream)
	mxError(DO_EXIT, "Unable to open file: %s", argv[optind]);
      else
	filenames[0] = strdup(argv[optind]); /* it's open, so proceed */
    }

  /* default values have been set as initialisers with AddParm */
  /* so first parse the stream */ 
 
  /* we've found a stream, so parse it */
  MXDEBUG(DB_DEBUG-1 MXDELIM "beginning to parse...");
  
  /* initialize line_numbering */
  for(i=0;i<MAX_INCLUDE_DEPTH;i++)
    line_num[i] = 1;
  
  yyin = stream;
  yyparse();
  
  
  /* then read command line args, so they take precedence */
  /* has yet to come */
  
  return 0;
}

tagvalue_t *
find_tag(const char *str) {

  int i;
  
  for(i=0;i<pc;i++)
    if(!strcmp(pp[i].tag, str))
      return &pp[i];
  
  return NULL;
}

int
test_typ(const tagvalue_t *tv, const int typ) {

  return tv->typ==typ?1:0;
}


#define GENERIC_TOP \
\
  tagvalue_t *t; \
\
  if((t = find_tag(tag))) { \
    mxError(NO_EXIT, "tag: <%s> not unique", t->tag); \
    return NULL; \
  } \
  pp[pc].tag = tag; \


#define GENERIC_BOTTOM return &pp[pc++];
  

tagvalue_t *
AddParmBool(const char *tag, int value) {

  GENERIC_TOP;

  pp[pc].typ = T_BOOL;
  pp[pc].dim[0] = 1;
  pp[pc].dim[1] = 0;
  pp[pc].value.i = value;

  GENERIC_BOTTOM;
}

tagvalue_t *
AddParmInt(const char *tag, int value) {

  GENERIC_TOP;

  pp[pc].typ = T_INT;
  pp[pc].dim[0] = 1;
  pp[pc].dim[1] = 0;
  pp[pc].value.i = value;

  GENERIC_BOTTOM;
}

tagvalue_t *
AddParmIntVec(const char *tag, int *value, int dim1) {

  GENERIC_TOP;

  pp[pc].typ = T_INT_V;
  pp[pc].dim[0] = dim1;
  pp[pc].dim[1] = 0;
  pp[pc].value.iv = value;

  GENERIC_BOTTOM;
}

tagvalue_t *
AddParmIntMat(const char *tag, int **value, int dim1, int dim2) {

  GENERIC_TOP;

  pp[pc].typ = T_INT_M;
  pp[pc].dim[0] = dim1;
  pp[pc].dim[1] = dim2;
  pp[pc].value.im = value;

  GENERIC_BOTTOM;
}

tagvalue_t *
AddParmFloat(const char *tag, float value) {

  GENERIC_TOP;

  pp[pc].typ = T_FLOAT;
  pp[pc].dim[0] = 1;
  pp[pc].dim[1] = 0;
  pp[pc].value.f = value;

  GENERIC_BOTTOM;
}

tagvalue_t *
AddParmFloatVec(const char *tag, float *value, int dim1) {

  GENERIC_TOP;

  pp[pc].typ = T_FLOAT_V;
  pp[pc].dim[0] = dim1;
  pp[pc].dim[1] = 0;
  pp[pc].value.fv = value;

  GENERIC_BOTTOM;
}

tagvalue_t *
AddParmFloatMat(const char *tag, float **value, int dim1, int dim2) {

  GENERIC_TOP;

  pp[pc].typ = T_FLOAT_M;
  pp[pc].dim[0] = dim1;
  pp[pc].dim[1] = dim2;
  pp[pc].value.fm = value;

  GENERIC_BOTTOM;
}
tagvalue_t *
AddParmDouble(const char *tag, double value) {

  GENERIC_TOP;

  pp[pc].typ = T_DOUBLE;
  pp[pc].dim[0] = 1;
  pp[pc].dim[1] = 0;
  pp[pc].value.d = value;

  GENERIC_BOTTOM;
}

tagvalue_t *
AddParmDoubleVec(const char *tag, double *value, int dim1) {

  GENERIC_TOP;

  pp[pc].typ = T_DOUBLE_V;
  pp[pc].dim[0] = dim1;
  pp[pc].dim[1] = 0;
  pp[pc].value.dv = value;

  GENERIC_BOTTOM;
}

tagvalue_t *
AddParmDoubleMat(const char *tag, double **value, int dim1, int dim2) {

  GENERIC_TOP;

  pp[pc].typ = T_DOUBLE_M;
  pp[pc].dim[0] = dim1;
  pp[pc].dim[1] = dim2;
  pp[pc].value.dm = value;

  GENERIC_BOTTOM;
}

tagvalue_t *
AddParmString(const char *tag, char *value) {

  GENERIC_TOP;

  pp[pc].typ = T_STRING;
  pp[pc].dim[0] = value ? strlen(value) : 0;
  pp[pc].dim[1] = 0;
  pp[pc].value.s = value;

  GENERIC_BOTTOM;
}

tagvalue_t *
AddParmTree(const char *tag, node *value) {

  GENERIC_TOP;

  pp[pc].typ = T_TREE;
  pp[pc].dim[0] = 0;
  pp[pc].dim[1] = 0;
  pp[pc].value.t = value;

  GENERIC_BOTTOM;
}

tagvalue_t *
PrintParm(FILE *fptr, tagvalue_t *tv) {

  int j,k;

  switch(tv->typ) {
  case T_INT: 
    fprintf(fptr, "%s = %d;\n", tv->tag, tv->value.i);
    break;
  case T_BOOL:
    if(tv->value.i)
      fprintf(fptr, "%s = True;\n", tv->tag);
    else
      fprintf(fptr, "%s = False;\n", tv->tag);
    break;
  case T_INT_V:
    if(tv->dim[0] > 0) {
      fprintf(fptr, "%s = { %d", tv->tag, tv->value.iv[0]);
      for(j=1;j<tv->dim[0];j++)
	fprintf(fptr, ", %d", tv->value.iv[j]);
      fprintf(fptr, " };\n");
    }
    break;
  case T_INT_M:
    if(tv->dim[0] > 0) {
      fprintf(fptr, "%s =\n{\n", tv->tag);
      for(j=0;j<tv->dim[0];j++) {
	fprintf(fptr, " { %d", tv->value.im[j][0]);
	for(k=0;k<tv->dim[1];k++)    
	  fprintf(fptr, ", %d", tv->value.im[j][k]);
	fprintf(fptr, " },\n");
      }
      fprintf(fptr, "};\n");
    }
    break;
  case T_FLOAT:
    fprintf(fptr, "%s = %.3f;\n", tv->tag, tv->value.f);
    break;
  case T_DOUBLE:
    fprintf(fptr, "%s = %f;\n", tv->tag, tv->value.d);
    break;
  case T_FLOAT_V:
    if(tv->dim[0] > 0) {
      fprintf(fptr, "%s = { %.3f", tv->tag, tv->value.fv[0]);
      for(j=1;j<tv->dim[0];j++)
	fprintf(fptr, ", %.3f", tv->value.fv[j]);
      fprintf(fptr, " };\n");
    }
    break;
  case T_DOUBLE_V:
    if(tv->dim[0] > 0) {
      fprintf(fptr, "%s = { %f", tv->tag, tv->value.dv[0]);
      for(j=1;j<tv->dim[0];j++)
	fprintf(fptr, ", %f", tv->value.dv[j]);
      fprintf(fptr, " };\n");
    }
    break;
  case T_FLOAT_M:
    if(tv->dim[0] > 0) {
      fprintf(fptr, "%s =\n{\n", tv->tag);
      for(j=0;j<tv->dim[0];j++) {
	fprintf(fptr, " { %.3f", tv->value.fm[j][0]);
	for(k=0;k<tv->dim[1];k++)    
	  fprintf(fptr, ", %.3f", tv->value.fm[j][k]);
	fprintf(fptr, " },\n");
      }
      fprintf(fptr, "};\n");
    }
    break;
  case T_DOUBLE_M:
    if(tv->dim[0] > 0) {
      fprintf(fptr, "%s =\n{\n", tv->tag);
      for(j=0;j<tv->dim[0];j++) {
	fprintf(fptr, " { %f", tv->value.dm[j][0]);
	for(k=0;k<tv->dim[1];k++)    
	  fprintf(fptr, ", %f", tv->value.dm[j][k]);
	fprintf(fptr, " },\n");
      }
      fprintf(fptr, "};\n");
    }
    break;
  case T_STRING:
    if(!tv->value.s)
      fprintf(fptr, "%s = \"\";\n", tv->tag);
    else
      fprintf(fptr, "%s = \"%s\";\n", tv->tag, tv->value.s);
    break;
  case T_TREE:
    fprintf(fptr, "%s = ", tv->tag);
    print_tree(tv->value.t);
    break;
  default:
    mxError(NO_EXIT, "unknown parameter type");
    return NULL;
  } 

  return tv;
}


int
PrintAllParms(FILE *fptr) {
  
  int i;
  
  for(i=0;i<pc;i++)
    PrintParm(fptr, &pp[i]);

  return pc;
}


void
ListParm(tagvalue_t *tv) {

  switch(tv->typ) {
  case T_INT: 
    printf("%s : integer\n", tv->tag);
    break;
  case T_BOOL:
      printf("%s : boolean\n", tv->tag);
    break;
  case T_INT_V:
    printf("%s : integer vector\n", tv->tag);
    break;
  case T_INT_M:
    printf("%s : integer matrix\n", tv->tag);
    break;
  case T_FLOAT:
    printf("%s : float\n", tv->tag);
    break;
  case T_DOUBLE:
    printf("%s : double\n", tv->tag);
    break;
  case T_FLOAT_V:
    printf("%s : float vector\n", tv->tag);
    break;
  case T_DOUBLE_V:
    printf("%s : double vector\n", tv->tag);
    break;
  case T_FLOAT_M:
    printf("%s : float matrix\n", tv->tag);
    break;
  case T_DOUBLE_M:
    printf("%s : double matrix\n", tv->tag);
    break;
  case T_STRING:
    printf("%s : string\n", tv->tag);
    break;
  case T_TREE:
    printf("%s : tree\n", tv->tag);
    break;
  default:
    mxError(NO_EXIT, "unknown parameter type");
  } 
}


void
ListAllParms(void) {
  
  int i;
  
  for(i=0;i<pc;i++)
    ListParm(&pp[i]);

}


void
print_tree(node *n) {

  printf("(");
  if(n != NULL)
    print_subtree(n->childs);
  printf(");\n");
}

void
print_subtree(node *n) {

  for(;n!=NULL;n=n->brother) {
    printf("%s:%.3f",  n->Name, n->distance);
    if((n->brother != NULL) && (n->childs == NULL))
      printf(", ");
    if(n->childs) {
      printf(", (");
      print_subtree(n->childs);
      printf(")");
      if(n->brother != NULL)
	printf(", ");
    }
  }
}

void
repointer(node *father, node *son) {

    for(;son!=NULL;son=son->brother) {
      son->father = father;

      if(son->childs)
	repointer(son, son->childs);
    }
}
