/*
 * $Id:$ 
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

/* the following is for debugging
#define DEBUG
#define DEBUG_OUTPUT 100
*/

#ifdef HAVE_STDIO_H
#include <stdio.h>
#endif
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif

#include "defs.h"
#include "global.h"
#include "mxerror.h"
#include "mxparser.h"
#include "lslist.h"
/* --------------------------------------------------------------
   External Variables
   -------------------------------------------------------------- */

extern int AliFormat;

extern FILE
  * OutputStream,
  * SeqStream,
  * AliStream,
  * TreeStream
;

/* --------------------------------------------------------------
   Prototypes of local functions
   -------------------------------------------------------------- */

void WriteSequences(FILE *fptr, node *root, int header);
void WriteSubSequences(FILE *fptr, node *root);
void WriteSequence(FILE *fptr, node *n);
void WriteAlignment(FILE *fptr, node *root, int header);
void WriteTree(FILE *fptr, node *root, int header);
void WriteSubtree(FILE *fptr, node *n, int depth);
int  InsertionsLen(node *root);
void AllocateAlignments(node *root, int len);
void Substitute(node *Node, editop_t *eop);
void Delete(node *Node, delop_t *delop);
void InsertGaps(node *root, node *Node, int aliLen, int aliInspos, int aliInslen);
void Insert(node *Node, insop_t *insop,node *Tree);
void ComputeAlignment(node *root, node *Tree, int len);
void CreateAlignmentMatrix(node *root, char **ali, char **names, int *num);

/* --------------------------------------------------------------
   Write a single sequence to fptr
   -------------------------------------------------------------- */
void
WriteSequence(FILE *fptr, node *n) {

  extern tagvalue_t *InputType;
  int x,j;

  fprintf(fptr,"%c%s\n",CHOOSE_TAG(InputType->value.i),n->Name);
  
  for(x=0; x<n->SequenceLen; x+=SequenceOutputLen->value.i) {
    for(j=x; j<x+SequenceOutputLen->value.i && j<n->SequenceLen; ++j)
      fputc(n->sequence[j],fptr);
    fputc('\n',fptr);
  }
  fputc('\n',fptr);

  return;
}

/* ============================================================== 
   Write sequences (depth first). root must not be NULL.
   -------------------------------------------------------------- */
void
WriteSubSequences(FILE *fptr, node *root) {

  node *n;

  if(root->selected)
    WriteSequence(fptr,root);
  for(n=root->childs; n!=NULL; n=n->brother)
    WriteSubSequences(fptr,n);

  return;
}

/* --------------------------------------------------------------
   Write the sequences to fptr
   -------------------------------------------------------------- */
void
WriteSequences(FILE *fptr, node *root, int header) {

  if(header)
    fprintf(fptr,"Sequences:\n");
  
  WriteSubSequences(fptr, root);
  
  return;
}

/* ************************************************************** */

/* --------------------------------------------------------------
   compute sum of all insertion lengths.
   -------------------------------------------------------------- */
int 
InsertionsLen(node *root){

  editop_u *e;
  node *n;
  int ilen;

  ilen = 0;
  for(e=root->edits; e!=NULL; e=(editop_u*)e->editop.next)
    if(e->editop.type == I_OP)
      ilen += strlen(e->insop.insert_sequence);
  for(n=root->childs; n!=NULL; n=n->brother) {
    ilen += InsertionsLen(n);
  }
  return(ilen);
}

/* --------------------------------------------------------------
   Allocate memory for all alignments (depth first)
   -------------------------------------------------------------- */
void
AllocateAlignments(node *root, int len){

  node *n;

  root->alignment = (char*)malloc(sizeof(char)*len);
  root->alignment[0] = '\0'; /* initialize empty for easy comparison */

  for(n=root->childs; n!=NULL; n=n->brother)
    AllocateAlignments(n,len);

  return;
}

/* --------------------------------------------------------------
   Perform a single substitution
   -------------------------------------------------------------- */
void
Substitute(node *Node, editop_t *eop){

  int i,pos;

 /* walk behind the relevant position
    (otherwise gaps immediately before eop->position cause errors) */
  for(i=0,pos=0; pos<=eop->position; ++i)
    if(Node->alignment[i] != GAP_CHARACTER)
      ++pos;
  Node->alignment[i-1] = eop->new_value;

  return;
}

/* --------------------------------------------------------------
   Perform a single deletion
   -------------------------------------------------------------- */
void
Delete(node *Node, delop_t *delop){

  int pos,i;

  for(i=0,pos=0; pos<delop->position; ++i)
    if(Node->alignment[i] != GAP_CHARACTER)
      ++pos;
  for(pos=0; pos<delop->length; ++i)
    if(Node->alignment[i] != GAP_CHARACTER) {
      Node->alignment[i] = GAP_CHARACTER;
      ++pos;
    }
  return;
}

/* --------------------------------------------------------------
   Insert gaps (by insertion) in subtree except if root==Node
   -------------------------------------------------------------- */
void
InsertGaps(node *root, node *Node, int aliLen, int aliInspos, int aliInslen){

  int j;
  node *n;

  if(aliLen <= 0)
    mxError(DO_EXIT,"Impossible alignment length");

  if(root!=Node && root->alignment[0]!='\0') {
  /* removed +1 from j=aliLen+1 in next line */
    for(j=aliLen; j>=aliInspos; --j) /* shift right part (incl. '\0') */
      root->alignment[j+aliInslen] = root->alignment[j];
    for(j=0; j<aliInslen; ++j) /* insert gaps */
      root->alignment[aliInspos+j] = GAP_CHARACTER;
    for(n=root->childs; n!=NULL; n=n->brother)
      InsertGaps(n,Node,aliLen,aliInspos,aliInslen);
  }
  return;
}

/* --------------------------------------------------------------
   Perform a single insertion
   -------------------------------------------------------------- */
void
Insert(node *Node, insop_t *insop,node *Tree){

  int pos,i,j, aliLen, insLen;


  /* at this node: insert letters */
  aliLen = strlen(Node->alignment); /* determine last character */
  insLen = strlen(insop->insert_sequence); /* determine insertion length */
  for(i=0,pos=0; pos<insop->position; ++i)
    if(Node->alignment[i] != GAP_CHARACTER)
      ++pos;
  /* removed +1 from j=aliLen+1 in next line */
  for(j=aliLen; j>=i; --j) /* shift right part to the right (incl. '\0') */
    Node->alignment[j+insLen] = Node->alignment[j];
  for(j=0; j<insLen; ++j) /* insert sequence */
    Node->alignment[i+j] = insop->insert_sequence[j];

  /* insert gaps at all other nodes in the tree where already sequences are */
  InsertGaps(Tree,Node,aliLen,i,insLen);

  return;
}

/* --------------------------------------------------------------
   Compute alignment (depth first)
   -------------------------------------------------------------- */
void
ComputeAlignment(node *root, node *Tree, int len){

  node *n;
  editop_u *e;

  for(n=root->childs; n!=NULL; n=n->brother) {
    strncpy(n->alignment,root->alignment,len);
    for(e=n->edits; e!=NULL; e=(editop_u*)e->editop.next) {
      switch(e->editop.type) {
        case E_OP: Substitute(n,&e->editop);
                   break;
        case D_OP: Delete(n,&e->delop);
                   break;
        case I_OP: Insert(n,&e->insop,Tree);
                   break;
        default:   mxError(DO_EXIT,"Illegal edit operation: %d\n",
                           e->editop.type);
      }
    }
    ComputeAlignment(n,Tree,len);
  }
  return;
}

/* --------------------------------------------------------------
   Copy alignments into matrix (depth first)
   -------------------------------------------------------------- */
void
CreateAlignmentMatrix(node *root, char **ali, char **names, int *num){

  node *n;

  if(root->selected || AlignmentWithAncestors->value.i) {
    ali[*num] = root->alignment;
    names[*num] = root->Name;
    ++(*num);
  }
  for(n=root->childs; n!=NULL; n=n->brother)
    CreateAlignmentMatrix(n,ali,names,num);

  return;
}

/* ============================================================== 
   Compute and write alignment (root must be non-NULL)
   -------------------------------------------------------------- */
void
WriteAlignment(FILE *fptr, node *root, int header) {

  extern tagvalue_t *SequenceNum;
  int maxAliLen, aliLen, num;
  char **names,**ali;
  int x,i,j,j2;
  BOOL onlygaps;
  int firstround;

  /* compute maximal alignment length and allocate memory for alignments
     in the tree */
  maxAliLen = root->SequenceLen + InsertionsLen(root) + 1; /* terminal '\0' */
  AllocateAlignments(root,maxAliLen);

  /* copy root sequence to alignment */
  strncpy(root->alignment,root->sequence,maxAliLen);

  /* compute alignment along the edges of the tree */
  ComputeAlignment(root,root,maxAliLen);
  aliLen = strlen(root->alignment);

  /* allocate memory for alignment matrix */
  ali = (char**)malloc(sizeof(char*)*(2*SequenceNum->value.i));
  names = (char**)malloc(sizeof(char*)*(2*SequenceNum->value.i));

  /* copy sequence names and alignment into matrix (depth first) */
  num = 0;
  CreateAlignmentMatrix(root,ali,names,&num);

  /* remove columns consisting of gaps only
     (which can occur by insertions at inner nodes which are later deleted) */
  for(j=0; j<aliLen; ) { /* do not check terminal '\0' */
    for(onlygaps=TRUE,i=0; i<num; ++i) {
      onlygaps &= ali[i][j]==GAP_CHARACTER;
    }
    if(onlygaps) { /* shift alignment one position left */
      for(j2=j; j2<aliLen; ++j2) /* terminal '\0' is also copied */
        for(i=0; i<num; ++i)
          ali[i][j2] = ali[i][j2+1];
      --aliLen;
    }
    else
      ++j;
  }

  /* write alignment to fptr */

  /* print header if wanted (for single file output) */
  if(header)
    fprintf(fptr,"Alignment:\n");

  if(AliFormat == ALI_FASTA) {
    for(i=0; i<num; ++i) {
      fprintf(fptr,">%s\n",names[i]);
      for(x=0; x<aliLen; x+=SequenceOutputLen->value.i) {
        for(j=x; j<x+SequenceOutputLen->value.i && j<aliLen; ++j) {
          fprintf(fptr,"%c",ali[i][j]);
        }
        fputc('\n',fptr);
      }
    }
    fputc('\n',fptr);
  }

  /* AliFormat == ALI_PHYLIP */
  else {
    firstround = TRUE;
    fprintf(fptr," %d %d \n", num, aliLen);
    for(x=0; x<aliLen; x+=SequenceOutputLen->value.i) {
      for(i=0; i<num; ++i) {
        if(firstround) {
          fprintf(fptr,"%-10s  ",names[i]);
        }
        else {
          fprintf(fptr,"%10s  ","");
        }
        for(j=x; j<x+SequenceOutputLen->value.i && j<aliLen; ++j) {
          fprintf(fptr,"%c",ali[i][j]);
        }
        fputc('\n',fptr);
      }
      fputc('\n',fptr);
      firstround = FALSE;
    }
  }

  free(ali);
  free(names);

  return;
}

/* -------------------------------------------------------------- 
   Write subtree of relatedness tree to fptr
   -------------------------------------------------------------- */
void
WriteSubtree(FILE *fptr, node *n, int depth){

  for( ; n!=NULL; n=n->brother) {
    if(n->childs != NULL) {
      fprintf(fptr,"\n%*s(",depth,"");
      WriteSubtree(fptr,n->childs,depth+1);
      fprintf(fptr,"\n%*s)",depth,"");
    }
    else {
      fprintf(fptr,"\n%*s",depth,"");
    }
    if(n->selected || (TreeWithAncestors->value.i ? n->relevant : FALSE)) {
      fprintf(fptr,"%s%s%s%s",n->Name,
              TreeWithSequences->value.i ? "[" : "",
              TreeWithSequences->value.i ?
            (TreeSequencesWithGaps->value.i ? n->alignment : n->sequence) : "",
              TreeWithSequences->value.i ? "]" : "");
    }
    if(n->distance > 0) {
      fprintf(fptr,":%g",n->distance);
    }
    if(n->brother != NULL) {
      fprintf(fptr,",");
    }
  }

  return;
}

/* ============================================================== 
   Write relatedness tree (root must be non-NULL)
   -------------------------------------------------------------- */
void
WriteTree(FILE *fptr, node *root, int header){

  if(header)
    fprintf(fptr,"PHYLIP tree:");
  WriteSubtree(fptr,root,0);
  fprintf(fptr,";\n");

  return;
}

/* -------------------------------------------------------------- 
   Write sequences, alignment, and relatedness tree
   -------------------------------------------------------------- */
void
DoOutput(void) {

  
  /* write sequences */
  MXDEBUG(DEBUG_OUTPUT MXDELIM "*** WriteSequences ***");
  if(StdOut->value.i == TRUE)
    WriteSequences(stdout,TheTree->value.t, TRUE);
  if(OutputStream != NULL)
    WriteSequences(OutputStream,TheTree->value.t, TRUE);
  if(SeqStream!= NULL)
    WriteSequences(SeqStream,TheTree->value.t, FALSE);

  /* write alignment */
  MXDEBUG(DEBUG_OUTPUT_LEVEL MXDELIM "*** WriteAlignment ***");
  if(StdOut->value.i == TRUE)
    WriteAlignment(stdout,TheTree->value.t, TRUE);
  if(OutputStream != NULL)
    WriteAlignment(OutputStream,TheTree->value.t, TRUE);
  if(AliStream!= NULL)
    WriteAlignment(AliStream,TheTree->value.t, FALSE);

  /* write relatedness tree */
  MXDEBUG(DEBUG_OUTPUT_LEVEL MXDELIM "*** WriteTree ***");
  if(StdOut->value.i == TRUE)
    WriteTree(stdout,TheTree->value.t, TRUE);
  if(OutputStream != NULL)
    WriteTree(OutputStream,TheTree->value.t, TRUE);
  if(TreeStream!= NULL)
    WriteTree(TreeStream,TheTree->value.t, FALSE);

  return;
}

void
WriteSpacer(int count) {
  /* write spacer */
  MXDEBUG(DEBUG_OUTPUT_LEVEL MXDELIM "*** WriteSpacer ***");
  if(StdOut->value.i == TRUE)
    fprintf(stdout,"\n--- run %d ---\n\n", count);
  if(OutputStream != NULL)
    fprintf(OutputStream,"\n--- run %d ---\n\n", count);
  if(AliStream!= NULL)
    fprintf(AliStream,"\n--- run %d ---\n\n", count);

  return;
}
