/*
 * $Id:$ 
 */


#define DEBUG 
#define DEBUG_ROSE 9
#define DEBUG_SOFT 20
#define BASES      4   /* number of bases (A,C,G,T) */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

/* includes */
#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif
#ifdef HAVE_STDIO_H
#include <stdio.h>
#endif
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_TIME_H
#include <time.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_CTYPE_H
#include <ctype.h>
#endif
#ifdef HAVE_MATH_H
#include <math.h>
#endif

#include "defs.h"
#include "global.h"
#include "mxerror.h"
#include "mxparser.h"
#include "tree.h"
#include "output.h"
#include "evolve.h"

/* local prototypes */
void ProcessConfig(void);
void CreateRootSequence(void);
void CreateMutationProbabilities(void);
void CreateTheTree(void);
int CreateSequence(tagvalue_t *, tagvalue_t *);
int CreateMutationProbability(tagvalue_t *, tagvalue_t *);
void InitArgs(void);
void InitArgs_er(void);
void InitVars(void);
int prepare_matrix(void);
int prepare_func(tagvalue_t *, tagvalue_t *);
int prepare_vector(tagvalue_t *, tagvalue_t *);
int check_model();
int build_matrix(double time);
int build_matrix_er(double time);
int transponse_matrix(double **m);
void copy_matrix(double **m1, double **m2);
char * string_toupper(char *);
FILE * open_file(char *basename, char* suffix, char * mode);

/* global variables */

extern int errno;

FILE
  * OutputStream = NULL,
  * SeqStream = NULL,
  * AliStream = NULL,
  * TreeStream = NULL
;

int AliFormat;

tagvalue_t
  * SeedVal,
  * SequenceLen,
  * SequenceNum,
  * StdOut,           
  * OutputFilename,   
  * OutputFilebase,   
  * SequenceSuffix,   
  * AlignmentSuffix,  
  * AlignmentFormat,  
  * AlignmentWithAncestors,
  * TreeSuffix,       
  * SequenceOutputLen,
  * SequenceOutputLen,
  * InputFilename,
  * InputStream,
  * InputType,
  * Relatedness,
  * ChooseFromLeaves,
  * TreeWithSequences,
  * TreeSequencesWithGaps,
  * TreeWithAncestors,
  * TheTree,
  * TheSequence,
  * ThePAMMatrix,
  * TheMatrix,
  * TheAlphabet,
  * TheFrequencies,
  * TheFreq,
  * TheInsFunc,
  * TheDelFunc,
  * TheInsertionFunction,
  * TheDeletionFunction,
  * TheInsertThreshold,
  * TheDeleteThreshold,
  * TheMutationProbability,
  * TheDNAmodel,
  * MeanSubstitution,
  * TransitionBias,
  * TTratio,
  * NumberOfRuns,
  * SimCoding;
  
/* **************************************************************
   main 
   ER, Mon Oct  9 17:07:05 EDT 2006
   uses FillTree_er() instead of FillTree()
   ************************************************************** */
int 
main (int argc, char* argv[]){

  int runs;

  /* Set to TRUE if you need Error Code and Linenumber Info
     on Debug or Error */
  mxDebugInfo = FALSE;

  /* Declare known parameters, and initialize primitive types
     to default values, others to NULL */
  InitArgs_er();
  
  /* parse command line and config file */
  ParseCmd(argc, argv);
  
  /* process config */
  ProcessConfig();

  printf("\nSimCoding %d\n", SimCoding->value.i);

  for(runs=1; runs<=(NumberOfRuns->value.i); runs++) {
    
    /* new sequence for each run (if not fixed) */
    CreateRootSequence();
    
    /* new mutation probabilities for each run (if not fixed) */
    CreateMutationProbabilities();
    
    /* new tree for each run (if not fixed) */
    CreateTheTree();
    
    /* fill the tree with sequences */
    /* ER:
     * if SimCoding == TRUE we simulate indels as in a coding region.
     * Indels can only appear at multiple of 3 positions (maintaining the frame)
     * and the actual indel distributions only allow 3,6,9, insertions at the time.
     */
    FillTree_er(SimCoding->value.i);
    
    /* write output */
    if(NumberOfRuns->value.i > 1) {
      WriteSpacer(runs);
    }
    DoOutput(); 

  }
  /* all done successfully */
  return EXIT_SUCCESS;
}

void
ProcessConfig(void) {

  if((StdOut->value.i == FALSE)
     && (OutputFilename->value.s == NULL)
     && (OutputFilebase->value.s == NULL))
    mxError(DO_EXIT, "output stream not specified");


  /* What format should the alignment output be? */
  if(!strcmp(AlignmentFormat->value.s,TAG_FASTA)) {
    /* Fasta Format */
    AliFormat = ALI_FASTA;
    /* if the suffix wasn't explicitly set... */
    if(AlignmentSuffix->value.s == NULL)
      AlignmentSuffix->value.s = SUF_FASTA;
  }
  else {
    /* Phylip Format */
    AliFormat = ALI_PHYLIP;
    /* if the suffix wasn't explicitly set... */
    if(AlignmentSuffix->value.s == NULL)
      AlignmentSuffix->value.s = SUF_PHYLIP;
  }

  /* set sequence suffix if not explicitly given */
  if(SequenceSuffix->value.s == NULL)
    SequenceSuffix->value.s = ".fas";

  /* set tree suffix if not explicitly given */
  if(TreeSuffix->value.s == NULL)
    TreeSuffix->value.s = strdup(".tree");
  
  /* Write output to a single file? */
  if(OutputFilename->value.s != NULL)
    OutputStream = open_file(OutputFilename->value.s, NULL, "w");
  
  /* Write output to three separate files? */
  if(OutputFilebase->value.s != NULL) {    
    SeqStream = open_file(OutputFilebase->value.s,
                          SequenceSuffix->value.s, "w");
    AliStream = open_file(OutputFilebase->value.s,
                          AlignmentSuffix->value.s, "w");
    TreeStream = open_file(OutputFilebase->value.s,
                           TreeSuffix->value.s, "w");
  }
  
  /* convert the alphabet to upper case */
  if(!string_toupper(TheAlphabet->value.s))
    mxError(DO_EXIT, "TheAlphabet not specified");

  /* Initialize (accumulate) ThePAMMatrix -> TheMatrix */
  if(!prepare_matrix())
    mxError(DO_EXIT, "ThePAMMatrix not specified resp. no model declared");

  /* Initialize (accumulate) indel functions */
  if(!prepare_func(TheInsFunc, TheInsertionFunction))
    mxError(DO_EXIT, "TheInsFunc not specified");
  if(!prepare_func(TheDelFunc, TheDeletionFunction))
    mxError(DO_EXIT, "TheDelFunc not specified");

  /* Initialize (accumulate) frequency vector */
  if(!prepare_vector(TheFreq, TheFrequencies))
    mxError(DO_EXIT, "TheFreq not specified");

  /* if no PAMMatrix exists, but a correct model is given, fill matrix */
  if(ThePAMMatrix->value.dm == NULL) 
    build_matrix_er(1.0);

  /* seed the random functions 
     if global variable SeedVal is set use it
     instead of time of day*/
  if (SeedVal->value.i == NO_SEED)
    SeedVal->value.i = getpid()*time(NULL);
  MXDEBUG(2 MXDELIM "Using %d as Seed\n" MXDELIM SeedVal->value.i);
  srand48(SeedVal->value.i);

  if (SimCoding->value.s == TRUE) SimCoding->value.i = TRUE;

  return;
}

/* Create the root sequence if not fixed. */
void 
CreateRootSequence(void){

  static int first = TRUE;
  static int create = TRUE;

  /* check if we're here the first time and a root sequence is given */
  if((first == TRUE) && (TheSequence->value.s != NULL)) {

    /* then we do not need to create root sequences ever */
    create = FALSE;

    /* convert the sequence to upper case */
    string_toupper(TheSequence->value.s);

    /* TheSequence takes precedence over length parameter */
    SequenceLen->value.i = TheSequence->dim[0];
  }
  
  /* next time round is never the first */
  first = FALSE;
    
  /* Create the root sequence if needed */
  if (create == TRUE) {
    if(!CreateSequence(TheSequence, SequenceLen))
      mxError(DO_EXIT, "Failed to create TheSequence");
  }

  return;
}

/* Create the mutation probability vector if not fixed. */
void 
CreateMutationProbabilities(void){

  static int first = TRUE;
  static int create = TRUE;

   /* check if we're here the first time and a
      Mutation Probability Vector is given */
  if((first == TRUE) && (TheMutationProbability->value.dv != NULL)) {
    /* then we do not need to create the vector ever */
    create = FALSE;
    /* check for same size of Mutation Probability Vector and Sequence */
    if(TheMutationProbability->dim[0] != SequenceLen->value.i)
      mxError(DO_EXIT,
	      "Length of TheMutationProbability differs from SequenceLen");
  }

  /* next time round is never the first */
  first = FALSE;

  /* Create the Mutation Probability Vector if needed */
  if(create == TRUE) {
    if(!CreateMutationProbability(TheMutationProbability, SequenceLen))
      mxError(DO_EXIT, "Failed to create TheMutationProbability");
  }
  
  return;
}


/* Create the mutation guide tree if not fixed. */
void
CreateTheTree(void){

  static int first = TRUE;
  static int create = TRUE;
  static node *GivenTree = NULL;

  /* check if we're here the first time and TheTree is given */
  if((first == TRUE) && (TheTree->value.t != NULL)) {
    /* then we do not need to create the tree ever */
    create = FALSE;
    /* but we have to prune it each time. That is why we make a copy here. */
    GivenTree = CopyEmptyTree(TheTree->value.t);
  }

  /* if not first time, remove the old tree */
  if(first == FALSE) {
    RemoveTree(TheTree->value.t);
  }

  /* next time round is never the first */
  first = FALSE;

  /* create the new tree */
  if (create == TRUE) {

    /* if needed create a new tree */
    TheTree->value.t = CreateTree();
  }
  else {

    /* otherwise copy the stored initial tree into TheTree before pruning */
    TheTree->value.t = CopyEmptyTree(GivenTree);
  }

  /* remove unneccessary leaves and internal nodes */
  TheTree->value.t = PruneTree(TheTree->value.t);

  return;
}


/* Create a random sequence of length len
   allocate memory for the sequence 
   returns TRUE on success, FALSE on failure */
int
CreateSequence(tagvalue_t *seq, tagvalue_t *len) {
  
  int i;

  /* free the mem if there was a sequence */
  if(seq->value.s != NULL)
    free(seq->value.s);

  /* alloc the memory needed first */
  seq->value.s = malloc((1+ len->value.i) * sizeof(char));
  if(seq->value.s == NULL)
    return FALSE;
  
  /* fill sequence with random characters from correct alphabet */
  for (i=0; i< len->value.i; i++) 
    seq->value.s[i] = get_random_char(ZUFALL, 0);
  
  /* place null char at end of char string */
  seq->value.s[len->value.i] = '\0';

  /* set length of string */
  seq->dim[0] = len->value.i;

  return TRUE;
}

/* Create a mutation probability vector of length len all 1.0
   allocate memory for the sequence 
   returns TRUE on success, FALSE on failure */
int
CreateMutationProbability(tagvalue_t *vec, tagvalue_t *len) {
  
  int i;

  /* free the mem if there was a vector */
  if(vec->value.dv != NULL)
    free(vec->value.dv);

  /* alloc the memory needed first */
  vec->value.dv = malloc(len->value.i * sizeof(double));
  if(vec->value.dv == NULL)
    return FALSE;
  
  /* fill sequence with random characters from correct alphabet */
  for (i=0; i< len->value.i; i++) 
    vec->value.dv[i] = 1.0;

  vec->dim[0] = len->value.i;

  return TRUE;
}

/* Initialize the global parameters known for config
   Values can be addressed as:       
   SequenceLen->value.i, etc. */
void
InitArgs(void) {

  SeedVal           = AddParmInt("SeedVal", NO_SEED);
  SequenceLen       = AddParmInt("SequenceLen", 100);
  SequenceNum       = AddParmInt("SequenceNum", 10);

  StdOut            = AddParmBool("StdOut", TRUE);
  OutputFilename    = AddParmString("OutputFilename", NULL);
  OutputFilebase    = AddParmString("OutputFilebase", NULL);
  SequenceSuffix    = AddParmString("SequenceSuffix", NULL);
  AlignmentSuffix   = AddParmString("AlignmentSuffix", NULL);
  AlignmentFormat   = AddParmString("AlignmentFormat", TAG_PHYLIP);
  AlignmentWithAncestors = AddParmBool("AlignmentWithAncestors", FALSE);
  TreeSuffix        = AddParmString("TreeSuffix", NULL);
  SequenceOutputLen = AddParmInt("SequenceOutputLen", 60);

  InputType         = AddParmInt("InputType", PROTEIN);
  Relatedness       = AddParmInt("Relatedness", 1);
  ChooseFromLeaves  = AddParmBool("ChooseFromLeaves", TRUE);
  TreeWithSequences = AddParmBool("TreeWithSequences", FALSE);
  TreeSequencesWithGaps = AddParmBool("TreeSequencesWithGaps", FALSE);
  TreeWithAncestors = AddParmBool("TreeWithAncestors", FALSE);
  TheTree           = AddParmTree("TheTree", NULL);
  TheSequence       = AddParmString("TheSequence", NULL);
  ThePAMMatrix      = AddParmDoubleMat("ThePAMMatrix", NULL, 0, 0);
  TheMatrix         = AddParmDoubleMat("TheMatrix", NULL, 0, 0);
  TheAlphabet       = AddParmString("TheAlphabet", NULL);
  TheFreq           = AddParmDoubleVec("TheFreq", NULL, 0);
  TheFrequencies    = AddParmDoubleVec("TheFrequencies", NULL, 0);
  TheInsertThreshold  = AddParmDouble("TheInsertThreshold", 0.03);
  TheDeleteThreshold  = AddParmDouble("TheDeleteThreshold", 0.03);
  TheMutationProbability = AddParmDoubleVec("TheMutationProbability", NULL, 0);
  /*accumulated */
  TheInsertionFunction = AddParmDoubleVec("TheInsertionFunction", NULL, 0);
  TheDeletionFunction  = AddParmDoubleVec("TheDeletionFunction", NULL, 0);
  /* normal */
  TheInsFunc        = AddParmDoubleVec("TheInsFunc", NULL, 0);
  TheDelFunc        = AddParmDoubleVec("TheDelFunc", NULL, 0);  
  TheDNAmodel       = AddParmString("TheDNAmodel", NULL);
  MeanSubstitution  = AddParmDouble("MeanSubstitution", 0.01342302);
  TransitionBias    = AddParmDouble("TransitionBias", 1.0);
  TTratio           = AddParmDouble("TTratio", 0.0);      
  NumberOfRuns      = AddParmInt("NumberOfRuns", 1);
}

void
InitArgs_er(void) {

  SeedVal           = AddParmInt("SeedVal", NO_SEED);
  SequenceLen       = AddParmInt("SequenceLen", 100);
  SequenceNum       = AddParmInt("SequenceNum", 10);

  StdOut            = AddParmBool("StdOut", TRUE);
  OutputFilename    = AddParmString("OutputFilename", NULL);
  OutputFilebase    = AddParmString("OutputFilebase", NULL);
  SequenceSuffix    = AddParmString("SequenceSuffix", NULL);
  AlignmentSuffix   = AddParmString("AlignmentSuffix", NULL);
  AlignmentFormat   = AddParmString("AlignmentFormat", TAG_PHYLIP);
  AlignmentWithAncestors = AddParmBool("AlignmentWithAncestors", FALSE);
  TreeSuffix        = AddParmString("TreeSuffix", NULL);
  SequenceOutputLen = AddParmInt("SequenceOutputLen", 60);

  InputType         = AddParmInt("InputType", PROTEIN);
  Relatedness       = AddParmInt("Relatedness", 1);
  ChooseFromLeaves  = AddParmBool("ChooseFromLeaves", TRUE);
  TreeWithSequences = AddParmBool("TreeWithSequences", FALSE);
  TreeSequencesWithGaps = AddParmBool("TreeSequencesWithGaps", FALSE);
  TreeWithAncestors = AddParmBool("TreeWithAncestors", FALSE);
  TheTree           = AddParmTree("TheTree", NULL);
  TheSequence       = AddParmString("TheSequence", NULL);
  ThePAMMatrix      = AddParmDoubleMat("ThePAMMatrix", NULL, 0, 0);
  TheMatrix         = AddParmDoubleMat("TheMatrix", NULL, 0, 0);
  TheAlphabet       = AddParmString("TheAlphabet", NULL);
  TheFreq           = AddParmDoubleVec("TheFreq", NULL, 0);
  TheFrequencies    = AddParmDoubleVec("TheFrequencies", NULL, 0);
  TheInsertThreshold  = AddParmDouble("TheInsertThreshold", 0.03);
  TheDeleteThreshold  = AddParmDouble("TheDeleteThreshold", 0.03);
  TheMutationProbability = AddParmDoubleVec("TheMutationProbability", NULL, 0);
  /*accumulated */
  TheInsertionFunction = AddParmDoubleVec("TheInsertionFunction", NULL, 0);
  TheDeletionFunction  = AddParmDoubleVec("TheDeletionFunction", NULL, 0);
  /* normal */
  TheInsFunc        = AddParmDoubleVec("TheInsFunc", NULL, 0);
  TheDelFunc        = AddParmDoubleVec("TheDelFunc", NULL, 0);  
  TheDNAmodel       = AddParmString("TheDNAmodel", NULL);
  MeanSubstitution  = AddParmDouble("MeanSubstitution", 0.01342302);
  TransitionBias    = AddParmDouble("TransitionBias", 1.0);
  TTratio           = AddParmDouble("TTratio", 0.0);      
  NumberOfRuns      = AddParmInt("NumberOfRuns", 1);

  SimCoding         = AddParmBool("SimCoding", FALSE);
}

/* accumulates the values of ThePAMMatrix in TheMatrix
   size of both matrices is MAX_ALPH
   TheMatrix is allocated in function
   elem [0][i] of TheMatrix has total value
   return TRUE on success, FALSE on failure */
int
prepare_matrix(void) {

  int i,j;

  /* sanity check if PAMMatrix is there; if not, check for model */
  if(ThePAMMatrix->value.dm == NULL) {
    if ((check_model()) != TRUE) 
      return FALSE;
    return TRUE;
  }

  /* allocate memory for the accumulated PAM matrix */
  if(TheMatrix->value.dm == NULL) {
    TheMatrix->value.dm = (double **)malloc(MAX_ALPH * sizeof(double *));
    if(TheMatrix->value.dm == NULL)
      mxError(DO_EXIT, "failed to calloc");
    for(i=0;i<MAX_ALPH;i++) {
      TheMatrix->value.dm[i] = (double *)malloc(MAX_ALPH * sizeof(double));
      if(TheMatrix->value.dm == NULL)
	mxError(DO_EXIT, "failed to calloc");
    }
  }

  /* accumulate the matrix */
  for(j=0;j<TheAlphabet->dim[0];j++) {
    TheMatrix->value.dm[(int)TheAlphabet->value.s[0]][(int)TheAlphabet->value.s[j]] =
      ThePAMMatrix->value.dm[0][j];
    for(i=1;i<TheAlphabet->dim[0];i++) {
      TheMatrix->value.dm[(int)TheAlphabet->value.s[i]][(int)TheAlphabet->value.s[j]] = 
      TheMatrix->value.dm[(int)TheAlphabet->value.s[i-1]][(int)TheAlphabet->value.s[j]] +
	ThePAMMatrix->value.dm[i][j]; 
      
    }
  }

  /* fill the TOTAL column */
  for(j=0;j<TheAlphabet->dim[0];j++)
    TheMatrix->value.dm[TOTAL][(int)TheAlphabet->value.s[j]] = 
      TheMatrix->value.dm[(int)TheAlphabet->value.s[(int)TheAlphabet->dim[0]-1]][(int)TheAlphabet->value.s[j]];

  /* debug printing */

  MXDEBUG(DEBUG_ROSE MXDELIM "The cumulative Matrix\n");
  
#ifdef DEBUG
  for(i=0;i<TheAlphabet->dim[0];i++) {
    for(j=0;j<TheAlphabet->dim[0];j++) {
#endif
      MXDEBUG(DEBUG_ROSE MXDELIM "%f " MXDELIM TheMatrix->value.dm[(int)TheAlphabet->value.s[i]][(int)TheAlphabet->value.s[j]]);
#ifdef DEBUG      
    }
#endif
    MXDEBUG(DEBUG_ROSE MXDELIM "\n");
#ifdef DEBUG
  }
#endif
  MXDEBUG(DEBUG_ROSE MXDELIM "\n");

#ifdef DEBUG
  for(j=0;j<TheAlphabet->dim[0];j++)
#endif
    MXDEBUG(DEBUG_ROSE MXDELIM "%f " MXDELIM TheMatrix->value.dm[TOTAL][(int)TheAlphabet->value.s[j]]);
  MXDEBUG(DEBUG_ROSE MXDELIM "\n\n");

  return TRUE;
}

/* accumulates the values of array orig in array prep 
   len is length of array orig
   array prep is of length len+1
   allocated in function
   elem [0] of prep has total value
   return TRUE on success, FALSE on failure */

int
prepare_func(tagvalue_t *orig, tagvalue_t *prep) {

  int i;
  
  /* sanity check if orig is valid */
  if(orig->value.dv == NULL)
    return FALSE;
  /*    mxError(DO_EXIT, "array not specified");*/
  
  /* allocate size+1 doubles stretch of memory */
  prep->value.dv = (double *)malloc((orig->dim[0]+1)*sizeof(double));
  if(prep->value.dv == NULL)
    mxError(DO_EXIT, "failed to malloc");

  /* set the length of the vector */
  prep->dim[0] = orig->dim[0]+1;
  
  /* set initial value of accum array, shifted by 1 becode [0] is used
     for total value */
  prep->value.dv[1] = orig->value.dv[0];

  /* do the accumulation */
  for(i=2;i<prep->dim[0]; i++)
    prep->value.dv[i] = prep->value.dv[i-1] + orig->value.dv[i-1];

  /* set the total value */
  prep->value.dv[TOTAL] = prep->value.dv[orig->dim[0]];

  /* debug printing */

  /* len undeclared... ???
#ifdef DEBUG
  for(i=0;i<(len->value.i+1);i++)
#endif
  MXDEBUG(DEBUG_SOFT MXDELIM "%f " MXDELIM prep->value.dv[i]);
  MXDEBUG(DEBUG_SOFT MXDELIM "\n");
  */

  return TRUE;
}



int
prepare_vector(tagvalue_t *orig, tagvalue_t *prep) {

  int i;
  
  /* sanity check if orig is valid */
  if(orig->value.dv == NULL)
    return FALSE;
  /*    mxError(DO_EXIT, "vector not specified");*/

  prep->value.dv = (double *)calloc(MAX_ALPH, sizeof(double));
  if(prep->value.dv == NULL)
    mxError(DO_EXIT, "failed to calloc");
  
  /* set initial value of accum array, shifted by 1 becode [0] is used
     for total value */
  prep->value.dv[(int)TheAlphabet->value.s[0]] = orig->value.dv[0]; 

  /* do the accumulation */
  for(i=1;i<(TheAlphabet->dim[0]); i++)
    prep->value.dv[(int)TheAlphabet->value.s[i]] =
      prep->value.dv[(int)TheAlphabet->value.s[i-1]] + orig->value.dv[i];
  
  /* set the total value */
  prep->value.dv[TOTAL] =
    prep->value.dv[(int)TheAlphabet->value.s[TheAlphabet->dim[0]-1]];

  prep->dim[0] = TheAlphabet->dim[0];

#ifdef DEBUG
  for(i=0;i<TheAlphabet->dim[0]; i++)
#endif
    MXDEBUG(DEBUG_SOFT MXDELIM "%f " MXDELIM prep->value.dv[(int)TheAlphabet->value.s[i]]);
  MXDEBUG(DEBUG_SOFT MXDELIM "TOTAL: %f\n" MXDELIM prep->value.dv[TOTAL]);
  
  return TRUE;
}

/*--------------------------------------------------------------------/
/ Aux. function to check whether a keyword for a model (jukes-cantor,
/ f81,...) was given if PAMMatrix wasn't specified
/--------------------------------------------------------------------*/
int check_model(){

  if (TheDNAmodel->value.s == NULL) return FALSE;
  if ((strncmp(TheDNAmodel->value.s,"JC",2) == 0) ||
      (strncmp(TheDNAmodel->value.s,"HKY",3) == 0) ||
      (strncmp(TheDNAmodel->value.s,"F81",3) == 0) ||
      (strncmp(TheDNAmodel->value.s,"F84",3) == 0) ||
      (strncmp(TheDNAmodel->value.s,"K2P",3) == 0))
    return TRUE;
  return FALSE;
}

/*--------------------------------------------------------------------/
/ Build the matrix by keyword if no PAMMatrix was specified.
/ Possible models: "JC":   Jukes-Cantor-Model
/                  "HKY":  Hasegawa 1985
/                  "F81":  Felsenstein 1981
/                  "F84":  Felsenstein 1984
/                  "K2P":  Kimura-Model
/
/ see 'Molecular Systematics', page 437/438 for details
/
/ time-value by default is 1.0
/--------------------------------------------------------------------*/
int build_matrix(double t){
  int i,j;
  double mutate, stay, transition, transversion, total, PI, pihelp;
  double pi[MAX_ALPH];
  double A = 0.0;

  if(ThePAMMatrix->value.dm == NULL) {
    ThePAMMatrix->value.dm = (double **)malloc(BASES * sizeof(double *));
    if(ThePAMMatrix->value.dm == NULL)
      mxError(DO_EXIT, "failed to calloc");
    for(i = 0;i < BASES;i++) {
      ThePAMMatrix->value.dm[i] = (double *)malloc(BASES * sizeof(double));
      if(ThePAMMatrix->value.dm == NULL)
	mxError(DO_EXIT, "failed to calloc");
    } 
  }

  /* get the correct normalized frequencies of the nuclear bases first */
  for(i = 0;i < (TheAlphabet->dim[0]); i++) {
    pi[(int)TheAlphabet->value.s[i]] = 
      TheFrequencies->value.dv[(int)TheAlphabet->value.s[i]];
    if (i > 0) {
      pi[(int)TheAlphabet->value.s[i]] -= 
	TheFrequencies->value.dv[(int)TheAlphabet->value.s[i-1]];
    }
  }

  total = 0.0;
  for (i = 0; i < (TheAlphabet->dim[0]); i++) {
    total += pi[(int)TheAlphabet->value.s[i]];
  }
  
  if ( total  != 1.0) {
    for (i = 0; i < (TheAlphabet->dim[0]); i++) {
      pi[(int)TheAlphabet->value.s[i]] = pi[(int)TheAlphabet->value.s[i]] / total;
      MXDEBUG(DEBUG_ROSE MXDELIM "PI[%c] = %f\n" MXDELIM TheAlphabet->value.s[i] MXDELIM pi[(int)TheAlphabet->value.s[i]]);
    }
  }

  /* Jukes-Cantor-Model */
  if (strncmp(TheDNAmodel->value.s,"JC",2) == 0) {
    mutate = 0.25 - 0.25 * exp(-(MeanSubstitution->value.d) * t);
    stay   = 0.25 + 0.75 * exp(-(MeanSubstitution->value.d) * t);
    
    for (i = 0; i < BASES; i++) {
      for (j = 0; j < BASES ; j++) {
	ThePAMMatrix->value.dm[i][j] =  mutate;
      }
    }
    
    for (i = 0; i < BASES; i++) {
      ThePAMMatrix->value.dm[i][i] = stay;
    }
  }

  /* Kimura-Model */
  else if (strncmp(TheDNAmodel->value.s,"K2P",3) == 0) {
    stay = 0.25 + 0.25 * exp(-(MeanSubstitution->value.d) * t) + 0.5 * 
      exp(-(MeanSubstitution->value.d) * t* 
	  ((TransitionBias->value.d + 1)/2.0));
    transition = 0.25 + 0.25 * exp(-(MeanSubstitution->value.d) * t) - 0.5 * 
      exp(-(MeanSubstitution->value.d) * t* 
	  ((TransitionBias->value.d + 1)/2.0));
    transversion = 0.25 - 0.25 * exp(-(MeanSubstitution->value.d) * t);

    for (i = 0; i < (TheAlphabet->dim[0]); i++) {
      for (j = 0; j < (TheAlphabet->dim[0]) ; j++) {
	if (i == j) ThePAMMatrix->value.dm[i][i] = stay;
	else {
	  if (TheAlphabet->value.s[i] == 'A') {
	    if (TheAlphabet->value.s[j] == 'G')
	      ThePAMMatrix->value.dm[i][j] = transition;
	    else 
	      ThePAMMatrix->value.dm[i][j] = transversion;
	  }
	  else if (TheAlphabet->value.s[i] == 'C') {
	    if (TheAlphabet->value.s[j] == 'T')
	      ThePAMMatrix->value.dm[i][j] = transition;
	    else 
	      ThePAMMatrix->value.dm[i][j] = transversion;
	  }
	  else if (TheAlphabet->value.s[i] == 'G') {
	    if (TheAlphabet->value.s[j] == 'A')
	      ThePAMMatrix->value.dm[i][j] = transition;
	    else 
	      ThePAMMatrix->value.dm[i][j] = transversion;
	  }
	  else if (TheAlphabet->value.s[i] == 'T') {
	    if (TheAlphabet->value.s[j] == 'C')
	      ThePAMMatrix->value.dm[i][j] = transition;
	    else 
	      ThePAMMatrix->value.dm[i][j] = transversion;
	  }
	}
      }
    }
  }
	

  /* Felsenstein 1981 */
  else if (strncmp(TheDNAmodel->value.s,"F81",3) == 0) {
    for (i = 0; i < (TheAlphabet->dim[0]); i++) {
      for (j = 0; j < (TheAlphabet->dim[0]); j++) {
	ThePAMMatrix->value.dm[i][j] = pi[(int)TheAlphabet->value.s[j]] *
	  (1 - exp(-(MeanSubstitution->value.d) * t));
      }
      ThePAMMatrix->value.dm[i][i] = pi[(int)TheAlphabet->value.s[i]] +
	(1 - pi[(int)TheAlphabet->value.s[i]]) * 
	exp(-(MeanSubstitution->value.d) * t);
    }
  }
	
  /* Felsenstein 1984 */
  else if (strncmp(TheDNAmodel->value.s,"F84",3) == 0) {
    A = TTratio->value.d + 1.0;
    for (i = 0; i < (TheAlphabet->dim[0]); i++) {
      for (j = 0; j < (TheAlphabet->dim[0]); j++) {
	if ((TheAlphabet->value.s[j] == 'A') || 
	    (TheAlphabet->value.s[j] == 'T')) {
	  PI = pi['A'] + pi['T'];
	}
	else {
	  PI = pi['C'] + pi['G'];
	}

	pihelp = pi[(int)TheAlphabet->value.s[j]];
	if (i == j) {
	  ThePAMMatrix->value.dm[i][j] = pihelp + pihelp *
	    (1/PI - 1.0) * exp(-(MeanSubstitution->value.d) * t) +
	    ((PI-pihelp)/PI) * exp(-(MeanSubstitution->value.d) * t * A);
	}

	else {    
	  transition   = pihelp + pihelp * (1.0/PI - 1.0) * 
	    exp(-(MeanSubstitution->value.d) * t) -
	    (pihelp/PI) * exp(-(MeanSubstitution->value.d) * t * A);
	  transversion = pihelp * (1 - exp(-(MeanSubstitution->value.d) * t));
	  if (TheAlphabet->value.s[i] == 'A') {
	    if (TheAlphabet->value.s[j] == 'G')
	      ThePAMMatrix->value.dm[i][j] = transition;
	    else 
	      ThePAMMatrix->value.dm[i][j] = transversion;
	  }
	  else if (TheAlphabet->value.s[i] == 'C') {
	    if (TheAlphabet->value.s[j] == 'T')
	      ThePAMMatrix->value.dm[i][j] = transition;
	    else 
	      ThePAMMatrix->value.dm[i][j] = transversion;
	  }
	  else if (TheAlphabet->value.s[i] == 'G') {
	    if (TheAlphabet->value.s[j] == 'A')
	      ThePAMMatrix->value.dm[i][j] = transition;
	    else 
	      ThePAMMatrix->value.dm[i][j] = transversion;
	  }
	  else if (TheAlphabet->value.s[i] == 'T') {
	    if (TheAlphabet->value.s[j] == 'C')
	      ThePAMMatrix->value.dm[i][j] = transition;
	    else 
	      ThePAMMatrix->value.dm[i][j] = transversion;
	  }
	}
      }
    }
  }

  /* Hasegawa 1985 */
  else if (strncmp(TheDNAmodel->value.s,"HKY",3) == 0) {
    for (i = 0; i < (TheAlphabet->dim[0]); i++) {
      for (j = 0; j < (TheAlphabet->dim[0]); j++) {
	if ((TheAlphabet->value.s[j] == 'A') || 
	    (TheAlphabet->value.s[j] == 'T')) {
	  PI = pi['A'] + pi['T'];
	}
	else {
	  PI = pi['C'] + pi['G'];
	}
	A      = 1 + PI * (TransitionBias->value.d - 1.0);
	pihelp = pi[(int)TheAlphabet->value.s[j]];
	if (i == j) {
	  ThePAMMatrix->value.dm[i][j] = pihelp + pihelp *
	    (1/PI - 1.0) * exp(-(MeanSubstitution->value.d) * t) +
	    ((PI-pihelp)/PI) * exp(-(MeanSubstitution->value.d) * t * A);
	}

	else {    
	  transition   = pihelp + pihelp * (1.0/PI - 1.0) * 
	    exp(-(MeanSubstitution->value.d) * t) -
	    (pihelp/PI) * exp(-(MeanSubstitution->value.d) * t * A);
	  transversion = pihelp * (1 - exp(-(MeanSubstitution->value.d) * t));
	  if (TheAlphabet->value.s[i] == 'A') {
	    if (TheAlphabet->value.s[j] == 'G')
	      ThePAMMatrix->value.dm[i][j] = transition;
	    else 
	      ThePAMMatrix->value.dm[i][j] = transversion;
	  }
	  else if (TheAlphabet->value.s[i] == 'C') {
	    if (TheAlphabet->value.s[j] == 'T')
	      ThePAMMatrix->value.dm[i][j] = transition;
	    else 
	      ThePAMMatrix->value.dm[i][j] = transversion;
	  }
	  else if (TheAlphabet->value.s[i] == 'G') {
	    if (TheAlphabet->value.s[j] == 'A')
	      ThePAMMatrix->value.dm[i][j] = transition;
	    else 
	      ThePAMMatrix->value.dm[i][j] = transversion;
	  }
	  else if (TheAlphabet->value.s[i] == 'T') {
	    if (TheAlphabet->value.s[j] == 'C')
	      ThePAMMatrix->value.dm[i][j] = transition;
	    else 
	      ThePAMMatrix->value.dm[i][j] = transversion;
	  }
	}
      }
    }
  }

  /* debug printing */

  MXDEBUG(DEBUG_ROSE MXDELIM "The PAM Matrix\n");
  
#ifdef DEBUG
  for(i = 0;i < BASES;i++) {
    for(j = 0;j < BASES;j++) {
#endif
      MXDEBUG(DEBUG_ROSE MXDELIM "%f " MXDELIM ThePAMMatrix->value.dm[i][j]);
#ifdef DEBUG      
    }
#endif
    MXDEBUG(DEBUG_ROSE MXDELIM "\n");
#ifdef DEBUG
  }
#endif
  MXDEBUG(DEBUG_ROSE MXDELIM "\n");

  /* swap columns and rows in order to adhere to internal convention:
     In Moritz/Hillis they use (i,j) as (from,to) while in rose we use
     (to,from) as in Dayhoff's paper. */
  transponse_matrix(ThePAMMatrix->value.dm);
  prepare_matrix();

  return 0;

}

/*--------------------------------------------------------------------/
/ ER, Tue Oct  3 14:52:58 EDT 2006
/ modification of build_matrix(). 
/ there is an error in the calcualtion of teh F84 and HKY matrices
/ the quantity PI should be : p_a+p_g  (for a or g)
/                             p_c+p_t  (for c or t)
/
/
/
/ Build the matrix by keyword if no PAMMatrix was specified.
/ Possible models: "JC":   Jukes-Cantor-Model
/                  "HKY":  Hasegawa 1985
/                  "F81":  Felsenstein 1981
/                  "F84":  Felsenstein 1984
/                  "K2P":  Kimura-Model
/
/ see 'Molecular Systematics', page 437/438 for details
/
/ time-value by default is 1.0
/--------------------------------------------------------------------*/
int build_matrix_er(double t){
  int i,j;
  double mutate, stay, transition, transversion, total, PI, pihelp;
  double pi[MAX_ALPH];
  double A = 0.0;

  if(ThePAMMatrix->value.dm == NULL) {
    ThePAMMatrix->value.dm = (double **)malloc(BASES * sizeof(double *));
    if(ThePAMMatrix->value.dm == NULL)
      mxError(DO_EXIT, "failed to calloc");
    for(i = 0;i < BASES;i++) {
      ThePAMMatrix->value.dm[i] = (double *)malloc(BASES * sizeof(double));
      if(ThePAMMatrix->value.dm == NULL)
	mxError(DO_EXIT, "failed to calloc");
    } 
  }

  /* get the correct normalized frequencies of the nuclear bases first */
  for(i = 0;i < (TheAlphabet->dim[0]); i++) {
    pi[(int)TheAlphabet->value.s[i]] = 
      TheFrequencies->value.dv[(int)TheAlphabet->value.s[i]];
    if (i > 0) {
      pi[(int)TheAlphabet->value.s[i]] -= 
	TheFrequencies->value.dv[(int)TheAlphabet->value.s[i-1]];
    }
  }

  total = 0.0;
  for (i = 0; i < (TheAlphabet->dim[0]); i++) {
    total += pi[(int)TheAlphabet->value.s[i]];
  }
  
  if ( total  != 1.0) {
    for (i = 0; i < (TheAlphabet->dim[0]); i++) {
      pi[(int)TheAlphabet->value.s[i]] = pi[(int)TheAlphabet->value.s[i]] / total;
      MXDEBUG(DEBUG_ROSE MXDELIM "PI[%c] = %f\n" MXDELIM TheAlphabet->value.s[i] MXDELIM pi[(int)TheAlphabet->value.s[i]]);
    }
  }

  /* Jukes-Cantor-Model */
  if (strncmp(TheDNAmodel->value.s,"JC",2) == 0) {
    mutate = 0.25 - 0.25 * exp(-(MeanSubstitution->value.d) * t);
    stay   = 0.25 + 0.75 * exp(-(MeanSubstitution->value.d) * t);
    
    for (i = 0; i < BASES; i++) {
      for (j = 0; j < BASES ; j++) {
	ThePAMMatrix->value.dm[i][j] =  mutate;
      }
    }
    
    for (i = 0; i < BASES; i++) {
      ThePAMMatrix->value.dm[i][i] = stay;
    }
  }

  /* Kimura-Model */
  else if (strncmp(TheDNAmodel->value.s,"K2P",3) == 0) {
    stay = 0.25 + 0.25 * exp(-(MeanSubstitution->value.d) * t) + 0.5 * 
      exp(-(MeanSubstitution->value.d) * t* 
	  ((TransitionBias->value.d + 1)/2.0));
    transition = 0.25 + 0.25 * exp(-(MeanSubstitution->value.d) * t) - 0.5 * 
      exp(-(MeanSubstitution->value.d) * t* 
	  ((TransitionBias->value.d + 1)/2.0));
    transversion = 0.25 - 0.25 * exp(-(MeanSubstitution->value.d) * t);

    for (i = 0; i < (TheAlphabet->dim[0]); i++) {
      for (j = 0; j < (TheAlphabet->dim[0]) ; j++) {
	if (i == j) ThePAMMatrix->value.dm[i][i] = stay;
	else {
	  if (TheAlphabet->value.s[i] == 'A') {
	    if (TheAlphabet->value.s[j] == 'G')
	      ThePAMMatrix->value.dm[i][j] = transition;
	    else 
	      ThePAMMatrix->value.dm[i][j] = transversion;
	  }
	  else if (TheAlphabet->value.s[i] == 'C') {
	    if (TheAlphabet->value.s[j] == 'T')
	      ThePAMMatrix->value.dm[i][j] = transition;
	    else 
	      ThePAMMatrix->value.dm[i][j] = transversion;
	  }
	  else if (TheAlphabet->value.s[i] == 'G') {
	    if (TheAlphabet->value.s[j] == 'A')
	      ThePAMMatrix->value.dm[i][j] = transition;
	    else 
	      ThePAMMatrix->value.dm[i][j] = transversion;
	  }
	  else if (TheAlphabet->value.s[i] == 'T') {
	    if (TheAlphabet->value.s[j] == 'C')
	      ThePAMMatrix->value.dm[i][j] = transition;
	    else 
	      ThePAMMatrix->value.dm[i][j] = transversion;
	  }
	}
      }
    }
  }
	

  /* Felsenstein 1981 */
  else if (strncmp(TheDNAmodel->value.s,"F81",3) == 0) {
    for (i = 0; i < (TheAlphabet->dim[0]); i++) {
      for (j = 0; j < (TheAlphabet->dim[0]); j++) {
	ThePAMMatrix->value.dm[i][j] = pi[(int)TheAlphabet->value.s[j]] *
	  (1 - exp(-(MeanSubstitution->value.d) * t));
      }
      ThePAMMatrix->value.dm[i][i] = pi[(int)TheAlphabet->value.s[i]] +
	(1 - pi[(int)TheAlphabet->value.s[i]]) * 
	exp(-(MeanSubstitution->value.d) * t);
    }
  }
	
  /* Felsenstein 1984 */
  else if (strncmp(TheDNAmodel->value.s,"F84",3) == 0) {
    A = TTratio->value.d + 1.0;
    for (i = 0; i < (TheAlphabet->dim[0]); i++) {
      for (j = 0; j < (TheAlphabet->dim[0]); j++) {
	if ((TheAlphabet->value.s[j] == 'A') || 
	    (TheAlphabet->value.s[j] == 'G')) {
	  PI = pi['A'] + pi['G'];
	}
	else {
	  PI = pi['C'] + pi['T'];
	}

	pihelp = pi[(int)TheAlphabet->value.s[j]];
	if (i == j) {
	  ThePAMMatrix->value.dm[i][j] = pihelp + pihelp *
	    (1/PI - 1.0) * exp(-(MeanSubstitution->value.d) * t) +
	    ((PI-pihelp)/PI) * exp(-(MeanSubstitution->value.d) * t * A);
	}

	else {    
	  transition   = pihelp + pihelp * (1.0/PI - 1.0) * 
	    exp(-(MeanSubstitution->value.d) * t) -
	    (pihelp/PI) * exp(-(MeanSubstitution->value.d) * t * A);
	  transversion = pihelp * (1 - exp(-(MeanSubstitution->value.d) * t));
	  if (TheAlphabet->value.s[i] == 'A') {
	    if (TheAlphabet->value.s[j] == 'G')
	      ThePAMMatrix->value.dm[i][j] = transition;
	    else 
	      ThePAMMatrix->value.dm[i][j] = transversion;
	  }
	  else if (TheAlphabet->value.s[i] == 'C') {
	    if (TheAlphabet->value.s[j] == 'T')
	      ThePAMMatrix->value.dm[i][j] = transition;
	    else 
	      ThePAMMatrix->value.dm[i][j] = transversion;
	  }
	  else if (TheAlphabet->value.s[i] == 'G') {
	    if (TheAlphabet->value.s[j] == 'A')
	      ThePAMMatrix->value.dm[i][j] = transition;
	    else 
	      ThePAMMatrix->value.dm[i][j] = transversion;
	  }
	  else if (TheAlphabet->value.s[i] == 'T') {
	    if (TheAlphabet->value.s[j] == 'C')
	      ThePAMMatrix->value.dm[i][j] = transition;
	    else 
	      ThePAMMatrix->value.dm[i][j] = transversion;
	  }
	}
      }
    }
  }

  /* Hasegawa 1985 */
  else if (strncmp(TheDNAmodel->value.s,"HKY",3) == 0) {
    for (i = 0; i < (TheAlphabet->dim[0]); i++) {
      for (j = 0; j < (TheAlphabet->dim[0]); j++) {
	if ((TheAlphabet->value.s[j] == 'A') || 
	    (TheAlphabet->value.s[j] == 'G')) {
	  PI = pi['A'] + pi['G'];
	}
	else {
	  PI = pi['C'] + pi['T'];
	}
	A      = 1 + PI * (TransitionBias->value.d - 1.0);
	pihelp = pi[(int)TheAlphabet->value.s[j]];
	if (i == j) {
	  ThePAMMatrix->value.dm[i][j] = pihelp + pihelp *
	    (1/PI - 1.0) * exp(-(MeanSubstitution->value.d) * t) +
	    ((PI-pihelp)/PI) * exp(-(MeanSubstitution->value.d) * t * A);
	}

	else {    
	  transition   = pihelp + pihelp * (1.0/PI - 1.0) * 
	    exp(-(MeanSubstitution->value.d) * t) -
	    (pihelp/PI) * exp(-(MeanSubstitution->value.d) * t * A);
	  transversion = pihelp * (1 - exp(-(MeanSubstitution->value.d) * t));
	  if (TheAlphabet->value.s[i] == 'A') {
	    if (TheAlphabet->value.s[j] == 'G')
	      ThePAMMatrix->value.dm[i][j] = transition;
	    else 
	      ThePAMMatrix->value.dm[i][j] = transversion;
	  }
	  else if (TheAlphabet->value.s[i] == 'C') {
	    if (TheAlphabet->value.s[j] == 'T')
	      ThePAMMatrix->value.dm[i][j] = transition;
	    else 
	      ThePAMMatrix->value.dm[i][j] = transversion;
	  }
	  else if (TheAlphabet->value.s[i] == 'G') {
	    if (TheAlphabet->value.s[j] == 'A')
	      ThePAMMatrix->value.dm[i][j] = transition;
	    else 
	      ThePAMMatrix->value.dm[i][j] = transversion;
	  }
	  else if (TheAlphabet->value.s[i] == 'T') {
	    if (TheAlphabet->value.s[j] == 'C')
	      ThePAMMatrix->value.dm[i][j] = transition;
	    else 
	      ThePAMMatrix->value.dm[i][j] = transversion;
	  }
	}
      }
    }
  }

  /* debug printing */

  MXDEBUG(DEBUG_ROSE MXDELIM "The PAM Matrix\n");
  
#ifdef DEBUG
  for(i = 0;i < BASES;i++) {
    for(j = 0;j < BASES;j++) {
#endif
      MXDEBUG(DEBUG_ROSE MXDELIM "%f " MXDELIM ThePAMMatrix->value.dm[i][j]);
#ifdef DEBUG      
    }
#endif
    MXDEBUG(DEBUG_ROSE MXDELIM "\n");
#ifdef DEBUG
  }
#endif
  MXDEBUG(DEBUG_ROSE MXDELIM "\n");

  /* swap columns and rows in order to adhere to internal convention:
     In Moritz/Hillis they use (i,j) as (from,to) while in rose we use
     (to,from) as in Dayhoff's paper. */
  transponse_matrix(ThePAMMatrix->value.dm);
  prepare_matrix();

  return 0;

}


/* ----------------------------------------------------------
   Transponses a matrix. Needed for the DNA-Models
   ---------------------------------------------------------- */
int transponse_matrix(double **m){

double **copy;
int i,j;

 copy = (double **)malloc((TheAlphabet->dim[0]) * sizeof(double *));
 if(copy == NULL)
   mxError(DO_EXIT, "failed to calloc");
 for(i = 0;i < (TheAlphabet->dim[0]);i++) {
   copy[i] = (double *)malloc((TheAlphabet->dim[0]) * sizeof(double));
   if(copy[i] == NULL)
     mxError(DO_EXIT, "failed to calloc");
 } 
 
 copy_matrix(copy,m);
 for (i=0;i< (TheAlphabet->dim[0]); i++) {
   for(j=0;j< (TheAlphabet->dim[0]); j++) {
     m[i][j] = copy[j][i];
   }
 }

 /* free the allocated memory */

 for(i=0;i<(TheAlphabet->dim[0]);i++) {
   free(copy[i]);
 }

 free(copy);

 return 0;


}


char *
string_toupper(char *str) {

  int i;

  if(str != NULL)
    for(i=0;i<strlen(str);i++)
      str[i] = toupper(str[i]);

  return str;
}

FILE *
open_file(char *basename, char* suffix, char * mode) {

  FILE *fptr;
  char *name;
  char empty[1] = "";

  if(suffix == NULL)
    suffix = empty;

  name =(char *)malloc(sizeof(char)*(1 + strlen(basename) + strlen(suffix)));
  if(name == NULL)
    mxError(DO_EXIT, "unable to malloc");
  sprintf(name,  "%s%s",   basename, suffix);
  fptr = fopen(name, mode);
  if(fptr == NULL) {
    mxError(DO_EXIT, "%s : \"%s\"", strerror(errno), name);
  }
  
  return fptr;
}
