/*
 * $Id: tree.c,v 1.29 2000/04/17 16:22:35 dirk Exp $
 */

/*#define DEBUG*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

/* debug constants */
#define DEBUG_TREE 15
#define DEBUG_PRUNING 15
#define DEBUG_REMOVE 15
#define DEBUG_COPY 15


/* includes */
#ifdef HAVE_STDIO_H
#include <stdio.h>
#endif
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_MATH_H
#include <math.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif

#include "defs.h"
#include "global.h"
#include "mxerror.h"
#include "mxparser.h"
#include "lslist.h"
#include "evolve.h" /* get_random_int() */

/* -------------------------------------------------------------- 
   Create two child nodes and fill their structs.
   -------------------------------------------------------------- */
node*
CreateTwoChilds(node *father, int *number, int depth, int edge_length){

  node *child1, *child2;

  child1 = (node*)malloc(sizeof(node));
  child2 = (node*)malloc(sizeof(node));

  MXDEBUG(DEBUG_TREE MXDELIM "creating node no. %d at depth %d" MXDELIM
          *number MXDELIM depth);

  child1->edits = NULL;
  child1->number = (*number)++;
  child1->distance = edge_length;

  child1->Name = (char*)malloc(sizeof(char)*(strlen(SEQUENCE_IDENTIFIER)+
                                             (int)log10(child1->number)+2));
  sprintf(child1->Name,"%s%d",SEQUENCE_IDENTIFIER,child1->number);
  child1->selected = FALSE;
  child1->relevant = FALSE;
  child1->sequence = NULL;
  child1->SequenceLen = 0;
  child1->alignment = NULL;

  child1->father = father;
  child1->brother = child2;
  if(depth > 0)
    child1->childs = CreateTwoChilds(child1,number,depth-1,edge_length);
  else
    child1->childs = NULL;

  MXDEBUG(DEBUG_TREE MXDELIM "creating node no. %d at depth %d" MXDELIM
          *number MXDELIM depth);

  child2->edits = NULL;
  child2->number = (*number)++;
  child2->distance = edge_length;
  child2->Name = (char*)malloc(sizeof(char)*(strlen(SEQUENCE_IDENTIFIER)+
                                             (int)log10(child2->number)+2));
  sprintf(child2->Name,"%s%d",SEQUENCE_IDENTIFIER,child2->number);
  child2->selected = FALSE;
  child2->relevant = FALSE;
  child2->sequence = NULL;
  child2->SequenceLen = 0;
  child2->alignment = NULL;

  child2->father = father;
  child2->brother = NULL;
  if(depth > 0)
    child2->childs = CreateTwoChilds(child2,number,depth-1,edge_length);
  else
    child2->childs = NULL;

  return(child1);
}

/* -------------------------------------------------------------- 
   Create a binary tree that fits the Options.Relatedness and number of
   sequences we read in the option struct.
   -------------------------------------------------------------- */
node* 
CreateTree(){

  extern tagvalue_t *Relatedness;
  extern tagvalue_t *ChooseFromLeaves;
  extern tagvalue_t *SequenceNum;
  node *root;
  
  int tree_depth,       /* depth of the binary tree */
    edge_length,        /* edge length to guarantee average Relatedness:
                           = Relatedness / (2 * (depth-2)), see paper */
    number_of_nodes,    /* maximal number of nodes: 2^(depth+1)-1 */
    number_of_leaves,   /* maximal number of leaves: 2^depth */
    max_sequence_num,   /* maximal number of sequences (depends on Options) */
    number=0;           /* number of the actually created node */
  
  /* 2^x mit x = \lceil log(n)/log(2)\rceil. */

  MXDEBUG(DEBUG_TREE MXDELIM "*** CreateTree ***");

  tree_depth = ceil(log(SequenceNum->value.i + 1) / log(2));

  if(tree_depth < 3)
    tree_depth = 3;

  MXDEBUG(DEBUG_TREE MXDELIM "SequenceNum=%d" MXDELIM SequenceNum->value.i);
  MXDEBUG(DEBUG_TREE MXDELIM "tree_depth=%d" MXDELIM tree_depth);

  if(ChooseFromLeaves->value.i) {
    edge_length = ROUND((float)Relatedness->value.i/(2*(tree_depth-1)));
  }
  else {
    edge_length = ROUND((float)Relatedness->value.i/(2*(tree_depth-2)));
  }

  number_of_nodes = (int)pow(2,tree_depth+1) - 1;
  number_of_leaves = (int)pow(2,tree_depth);
  max_sequence_num = ChooseFromLeaves->value.i ? number_of_leaves
                                               : number_of_nodes;

  /* test for reasonability */
  if(edge_length < 0) {
    mxError(DO_EXIT,"Relatedness %d is too small (cannot be resolved)!",
            Relatedness->value.i);
  }
  if(SequenceNum->value.i > max_sequence_num) {
    mxError(DO_EXIT,"Not more than %d sequences allowed!",max_sequence_num);
  }

  MXDEBUG(DEBUG_TREE MXDELIM "number_of_nodes=%d" MXDELIM number_of_nodes);
  MXDEBUG(DEBUG_TREE MXDELIM "number_of_leaves=%d" MXDELIM number_of_leaves);
  MXDEBUG(DEBUG_TREE MXDELIM "edge_length=%d" MXDELIM edge_length);

  MXDEBUG(DEBUG_TREE MXDELIM "creating root node no. %d in level %d"
	  MXDELIM number MXDELIM 0);


  root = (node*)malloc(sizeof(node));
  root->edits = NULL;
  root->number = number++;
  root->distance = 0;
  root->Name = malloc(sizeof(char)*(strlen(SEQUENCE_IDENTIFIER)+2));
  sprintf(root->Name,"%s0",SEQUENCE_IDENTIFIER);
  root->selected = FALSE;
  root->relevant = FALSE;
  root->sequence = NULL;
  root->SequenceLen = 0;
  root->alignment = NULL;

  root->father = NULL;
  root->brother = NULL;
  root->childs = CreateTwoChilds(root,&number,tree_depth-1,edge_length);

  return(root);
}

/******************************************************************/

/* --------------------------------------------------------------
   Prepare nodes: clear the "selected" and "relevant" flags
   and count nodes (leaves).
   "root" must not be NULL !!!
   -------------------------------------------------------------- */
int 
prepare_nodes(node *root){

  extern tagvalue_t *ChooseFromLeaves;
  int num;
  node *n;

  /* tag all nodes as not selected and not relevant */
  root->selected = FALSE;
  root->relevant = FALSE;

  /* count nodes */
  if( (!ChooseFromLeaves->value.i) || /* all nodes */
      (root->childs == NULL) || /* leaf                      [only one child */
      (root->father==NULL && root->childs->brother==NULL) ) /* root with     */
    num = 1;
  else
    num = 0;
  MXDEBUG(DEBUG_PRUNING MXDELIM "prepared node no. %d (%sselected)"
	  MXDELIM root->number MXDELIM
          num==1 ? "" : "not ");

  for(n=root->childs; n!=NULL; n=n->brother)
    num += prepare_nodes(n);

  return(num);
}

/* --------------------------------------------------------------
   Insert nodes (leaves) into nodelist (depth first)
   -------------------------------------------------------------- */
void 
insert_nodes(node *root, node **nodelist, int *num){

  extern tagvalue_t *ChooseFromLeaves;
  node *n;

  if( (!ChooseFromLeaves->value.i) || /* all nodes */
      (root->childs == NULL) || /* leaf                      [only one child */
      (root->father==NULL && root->childs->brother==NULL) ) { /* root with   */
    nodelist[(*num)++] = root;
    MXDEBUG(DEBUG_PRUNING MXDELIM "node no. %d added to nodelist position %d"
	    MXDELIM root->number MXDELIM (*num)-1);
  }

  for(n=root->childs; n!=NULL; n=n->brother)
    insert_nodes(n,nodelist,num);

  return;
}

/* --------------------------------------------------------------
   Remove a single node and free the memory
   -------------------------------------------------------------- */
void 
remove_node(node *n){

  lsDelList((LSlist_t*)n->edits);
  if(n->Name != NULL)
    free(n->Name);
  if(n->sequence != NULL)
    free(n->sequence);
  if(n->alignment != NULL)
    free(n->alignment);
  free(n);

  return;
}

/* --------------------------------------------------------------
   Remove the subtree including root.
   -------------------------------------------------------------- */
void 
remove_subtree(node *root){

  node *n, *n_next;

  for(n=root->childs; n!=NULL; n=n_next) {
    n_next = n->brother;
    remove_subtree(n);
  }
  remove_node(root);

  return;
}

/* --------------------------------------------------------------
   Recursively copy a tree.
   The copy is incomplete as only part of the node contents are
   copied, specifically edits, sequence, alignment, InsertionFunction,
   DeletionFunction, and MutationProbability are set to NULL.
   -------------------------------------------------------------- */
node *
copy_empty_node(node *n,node *father) {

  node *new_n=NULL,*c,*new_c,*firstchild,*previous=NULL;

  MXDEBUG(DEBUG_PRUNING MXDELIM "*** CopyTree ***");

  new_n = malloc(sizeof(node));
  if(new_n == NULL) {
    mxError(DO_EXIT, "malloc failed in copy_empty_node");
  }

  for(firstchild=NULL,c=n->childs; c!=NULL; c=c->brother) {
    new_c = copy_empty_node(c,new_n);
    if(firstchild == NULL) {
      firstchild = new_c;
    }
    else {
      previous->brother = new_c;
    }
    previous = new_c;
  }

  new_n->father = father;
  new_n-> brother = NULL; /* might be reset later */
  new_n->childs = firstchild;

  new_n->edits = NULL; /* !!! this makes it an incomplete copy! */

  new_n->number = n->number;
  new_n->distance = n->distance;

  new_n->Name = strdup(n->Name);
  new_n->selected = n->selected;
  new_n->relevant = n->relevant;

  new_n->sequence = NULL; /* !!! */
  new_n->SequenceLen = n->SequenceLen;
  new_n->alignment = NULL; /* !!! */

  new_n->InsertThreshold = n->InsertThreshold;
  new_n->DeleteThreshold = n->DeleteThreshold;

  new_n->InsertionFunction = NULL; /* !!! */
  new_n->DeletionFunction = NULL; /* !!! */

  new_n->MutationProbability = NULL; /* !!! */
  new_n->MutProbAlloc = n->MutProbAlloc;

  return new_n;
}

/* --------------------------------------------------------------
   remove all non-branching nodes with selected==FALSE entry
   from the tree and release their memory.
   precondition: the root must be relevant.
   -------------------------------------------------------------- */
void
compress_tree(node *root){
  node *n, *n_next, **nptr;

  /* compress the subtrees */
  for(n=root->childs; n!=NULL; n=n_next)
    if(n->relevant) {
      n_next = n->brother;
      compress_tree(n);
    }
    else { /* remove this subtree */
      MXDEBUG(DEBUG_PRUNING MXDELIM "removing node no. %d with subtree"
	      MXDELIM n->number);
      for(nptr=&n->father->childs; *nptr!=n; nptr=&(*nptr)->brother)
        /* nothing */ ;
      *nptr = n->brother;
      n_next = n->brother;
      remove_subtree(n);
    }

  /* non-branching nodes get removed if they are not the root of the tree */
  if((root->childs!=NULL) && (root->childs->brother==NULL) &&
     (root->father!=NULL) && (!root->selected)) {
    MXDEBUG(DEBUG_PRUNING MXDELIM "removing non-branching node no. %d"
	    MXDELIM root->number);
    root->childs->distance += root->distance;
    root->childs->brother = root->brother;
    for(nptr=&root->father->childs; *nptr!=root; nptr=&(*nptr)->brother)
      /* nothing */ ;
    *nptr = root->childs;
    root->childs->father = root->father;
    remove_node(root);
  }

  return;
    
} /* compress_tree() */

/* ==============================================================
   Select required number of nodes (see Options.SequenceNum)
   and delete the rest of the tree.
   -------------------------------------------------------------- */
node *
PruneTree(node *root) {

  extern tagvalue_t *SequenceNum;
  int i,r, nodeNum, num=0;
  node **nodelist, *n;

  MXDEBUG(DEBUG_PRUNING MXDELIM "*** PruneTree ***");

  /* set the "selected" and "relevant" flags of all nodes to FALSE
   * and count the nodes in the tree (possibly only leaves) */
  nodeNum = prepare_nodes(root);
  MXDEBUG(DEBUG_PRUNING MXDELIM "nodeNum=%d" MXDELIM nodeNum);
  if(SequenceNum->value.i > nodeNum) {
    mxError(DO_EXIT,"The tree contains only %d nodes!",nodeNum);
  }

  /* create list of nodes (possibly only leaves) */
  nodelist = (node**)malloc(sizeof(node*)*nodeNum);
  insert_nodes(root,nodelist,&num);

  /* select the correct number of nodes */
  for(i=0; i<SequenceNum->value.i; ) {
    r = get_random_int(nodeNum);
    MXDEBUG(DEBUG_PRUNING MXDELIM "random choice: r=%d" MXDELIM r);
    if(!nodelist[r]->selected) {
      nodelist[r]->selected = TRUE;
      ++i;
      MXDEBUG(DEBUG_PRUNING MXDELIM"node no. %d selected (i=%d)" MXDELIM
	      nodelist[r]->number MXDELIM i);
    }
  }

  /* tag the relevant part of the tree */
  root->relevant = TRUE;
  MXDEBUG(DEBUG_PRUNING MXDELIM "node no. %d is relevant" MXDELIM 0);
  for(i=0; i<nodeNum; ++i) {
    if((n=nodelist[i])->selected) {
      for( ; !(n->relevant); n=n->father) {
        n->relevant = TRUE;
        MXDEBUG(DEBUG_PRUNING MXDELIM "node no. %d is relevant" MXDELIM
		n->number);
      }
    }
  }

  /* compress the tree */
  compress_tree(root);

  /* free memory */
  free(nodelist);

  return(root);
}

/* ==============================================================
   Remove a tree.
   -------------------------------------------------------------- */
void
RemoveTree(node *root) {

  MXDEBUG(DEBUG_REMOVE MXDELIM "*** RemoveTree ***");

  remove_subtree(root);

  return;
}

/* ==============================================================
   Copy an empty tree.
   -------------------------------------------------------------- */
node *
CopyEmptyTree(node *root) {

  node *new_root;

  MXDEBUG(DEBUG_COPY MXDELIM "*** CopyTree ***");

  new_root = copy_empty_node(root,NULL);

  return new_root;
}

