#!/usr/bin/perl -w
#compare_histograms.pl

use strict;
use Class::Struct;

use vars qw ($opt_A $opt_B $opt_C $opt_b $opt_e $opt_f $opt_i $opt_o $opt_s $opt_x $opt_y $opt_w $opt_z);  # required if strict used
use Getopt::Std;
use constant GNUPLOT => '/usr/bin/gnuplot';

getopts ('ABCbefiosxywz');

# Print a helpful message if the user provides no input file.
if (!@ARGV) {
        print "usage:  compare_histograms.pl [options] N dir1 dir2 ... dirN\n\n";
        print "options:\n";
	print "-A           :  branch length optimization using dnaml method\n";
	print "-B           :  branch length optimization using cg\n";
	print "-C           :  all paramameter optimization\n";
	print "-b           :  uses 2 programs in the same msa\n";
        print "-e           :  use dnaml-erate    [ default dnaml]\n";
        print "-f           :  add curve fits\n";
        print "-o           :  use s3 programs: dnaml and dnaml-erate and dnaml-erate optimizing all rate paramters in the same msa\n";
        print "-s           :  seeplots    [ default do not]\n";
        print "-x           :  dnaml    \n";
        print "-y           :  dnal-erate [ first dnamle-erate score]\n";
        print "-w           :  dnaml versus dnal-erate \n";
        print "-z           :  dnaml versus dnal-erate.4 \n";
       exit;
}

my $ndir = shift;
my @dir;
for (my $n = 0; $n < $ndir; $n ++) { $dir[$n] = shift; }
for (my $n = 0; $n < $ndir; $n ++) { if ($dir[$n] =~ /^(.+)\/$/) { $dir[$n] = $1; } }

my $evince = "evince";

#eratedir = ERATEDIR (shell env variable)
my $eratedir =  $ENV{'ERATEDIR'};
my $testdir;
my $goutdir;
my $outdir;
if ($ndir == 1) {
    $goutdir = $dir[0];
    $outdir  = $goutdir;
}
else {
    if ($dir[0] =~ /^($eratedir\/test[^\/]+\/.+)\/[^\/]+$/) {
	$testdir = "$1";
	$goutdir = "$testdir/histograms";
	system("mkdir $goutdir\n");
    } 
    if (!$goutdir) { print "NO outdir:\n"; die; }
    print "TESTDIR:\n$testdir\n";
    print "GOUTDIR:\n$goutdir\n";


    my $name;
    my @name;
    for (my $n = 0; $n < $ndir; $n ++) { if ($dir[$n] =~ /^$testdir\/([^\/]+)$/ ) { $name[$n] = $1; } }
    $name = "$name[0]";
    for (my $n = 1; $n < $ndir; $n ++) { $name .= "_$name[$n]"; }
    $outdir = "$goutdir/$name";
    system("mkdir $outdir\n");
}

for (my $n = 1; $n < $ndir; $n ++) { 

    if ($dir[$n] =~ /^$testdir\/[^\/]+$/) 
    {
	my $dir = "$testdir/histograms";
	if ($dir =~ /^$goutdir$/) { next; }
	else { print "unmatched files\n"; die; }
    }
    else { print "bad dir:\n$dir[$n]\n"; die;}
}

my $seeplots = 0;
if ($opt_s) { $seeplots = 1; }

my $dnamlx = "dnaml";
my $dnamle = "";
my $dnamlo = "";

if ($opt_e) { $dnamlx = "dnaml-erate"; }

if ($opt_b || $opt_o) {
    $dnamle = "dnaml-erate";
    if    ($opt_A) { $dnamle .= "-default_bopt"; } # branch opt only a la dnaml
    elsif ($opt_B) { $dnamle .= "-cg_bopt"; }      # branch opt only using cg
}
my $dnamlxe = "$dnamlx\_$dnamle";
my $dnamlxo = "$dnamlx\_$dnamlo";

if ($opt_o ) {
    $dnamlo = "dnaml-erate";
    if (!$opt_C) { $dnamlo .= "-4";}
}

my @hh_fo;
my @hh_msalen;
my @hh_sqleng;
my @hh_sqlena;
my @hp_fo;
my @hp_msalen;
my @hp_sqleng;
my @hp_sqlena;

for (my $n = 0; $n < $ndir; $n ++) {
    $hh_fo[$n]     = get_file($dir[$n], "histogram_fo");
    $hh_msalen[$n] = get_file($dir[$n], "histogram_msalen");
    $hh_sqleng[$n] = get_file($dir[$n], "histogram_sqleng");
    $hh_sqlena[$n] = get_file($dir[$n], "histogram_sqlena");
    $hp_fo[$n]     = get_file($dir[$n], "paramhisto_fo");
    $hp_msalen[$n] = get_file($dir[$n], "paramhisto_msalen");
    $hp_sqleng[$n] = get_file($dir[$n], "paramhisto_sqleng");
    $hp_sqlena[$n] = get_file($dir[$n], "paramhisto_sqlena");
}

calculate_all_chi_square($ndir, 
			 \@hh_msalen, \@hh_sqleng, \@hh_sqlena,
			 \@hp_msalen, \@hp_sqleng, \@hp_sqlena);

calculate_all_meantp($ndir, 
		     \@hh_msalen, \@hh_sqleng, \@hh_sqlena);

calculate_all_maxtime_meanabl($ndir, 
			      \@hh_msalen, \@hh_sqleng, \@hh_sqlena,
			      \@hp_msalen, \@hp_sqleng, \@hp_sqlena);

if ($opt_i || $ndir == 1) {
    plot_individualy($ndir, 
		     \@hh_fo, \@hh_msalen, \@hh_sqleng, \@hh_sqlena,
		     \@hp_fo, \@hp_msalen, \@hp_sqleng, \@hp_sqlena);
}

plot_jointly($ndir, 
	     \@hh_fo, \@hh_msalen, \@hh_sqleng, \@hh_sqlena,
	     \@hp_fo, \@hp_msalen, \@hp_sqleng, \@hp_sqlena);



################
#### subroutines
################
sub accumulate_averages {
    my ($val, $meanval_ref, $meansquareval_ref) = @_;

    $$meanval_ref       += $val;
    $$meansquareval_ref += $val*$val;
}

sub calculate_averages {
    my ($meanval_ref, $stdval_ref, $number) = @_;

    my $mean = $$meanval_ref;
    my $std  = $$stdval_ref;

    if ($number > 1) {
	$mean /= $number;
	
	$std -= $mean*$mean*$number;
	$std /= ($number-1);
	if ($std < 0. && $std> -0.00001) { $std = 0.0; }
	$std  = sqrt($std);
    }
    elsif ($number == 1) {
	$mean /= $number;
	$std   = 0.0;
    }
    else {
	$mean = 0.0;
	$std  = 0.0;
    }

    $$meanval_ref = $mean;
    $$stdval_ref  = $std;

}

sub calculate_all_chi_square {
    my ($ndir,
	$hh_msalen_ref, $hh_sqleng_ref, $hh_sqlena_ref,
	$hp_msalen_ref, $hp_sqleng_ref, $hp_sqlena_ref) = @_;

    my $N = 5000;
    my $k = 1/5;

    my @binused;
    init_histo_array(2.0*$N, $k,\@binused);

    my @tpa;
    my @tpb;
    my @tpc;
    init_histo_array(2.0*$N, $k,\@tpa);
    init_histo_array(2.0*$N, $k,\@tpb);
    init_histo_array(2.0*$N, $k,\@tpc);

    my @d1a;
    my @d1b;
    my @d1c;
    init_histo_array(2.0*$N, $k,\@d1a);
    init_histo_array(2.0*$N, $k,\@d1b);
    init_histo_array(2.0*$N, $k,\@d1c);

    my @d2a;
    my @d2b;
    my @d2c;
    init_histo_array(2.0*$N, $k,\@d2a);
    init_histo_array(2.0*$N, $k,\@d2b);
    init_histo_array(2.0*$N, $k,\@d2c);

    my @title;
    my @key;

    my $xlabel_msalen = "ALIGNMENT LENGTH";
    for (my $n = 0; $n < $ndir; $n ++) {
	calculate_chi_square($hh_msalen_ref->[$n], $hp_msalen_ref->[$n], $xlabel_msalen,
			     \@title, \@key,
			     $n, 
			     $N, $k,
			     \@binused,
			     \@tpa, \@tpb, \@tpc,  
			     \@d1a, \@d1b, \@d1c, 
			     \@d2a, \@d2b, \@d2c);
    }
    compare_by_model($outdir, $hh_msalen_ref, $ndir, \@title, \@key,
		     $N, $k,
		     \@binused,
		     \@tpa, \@tpb, \@tpc, 
		     \@d1a, \@d1b, \@d1c, 
		     \@d2a, \@d2b, \@d2c);
    
    my $xlabel_sqleng = "GEOMETRIC MEAN SEQUENCES LENGTH";
    for (my $n = 0; $n < $ndir; $n ++) {
	calculate_chi_square($hh_sqleng_ref->[$n], $hp_sqleng_ref->[$n], $xlabel_sqleng,
			     \@title, \@key,
			     $n,  
			     $N, $k,
			     \@binused,
			     \@tpa, \@tpb, \@tpc, 
			     \@d1a, \@d1b, \@d1c, 
			     \@d2a, \@d2b, \@d2c);
    }
    compare_by_model($outdir, $hh_sqleng_ref, $ndir, \@title, \@key,
		     $N, $k,
		     \@binused,
		     \@tpa, \@tpb, \@tpc, 
		     \@d1a, \@d1b, \@d1c, 
		     \@d2a, \@d2b, \@d2c);
    
    my $xlabel_sqlena = "ARITHMETIC MEAN SEQUENCES LENGTH";
    for (my $n = 0; $n < $ndir; $n ++) {
	calculate_chi_square($hh_sqlena_ref->[$n], $hp_sqlena_ref->[$n], $xlabel_sqlena,
			     \@title, \@key,
			     $n, 
			     $N, $k,
			     \@binused,
			     \@tpa, \@tpb, \@tpc, 
			     \@d1a, \@d1b, \@d1c, 
			     \@d2a, \@d2b, \@d2c);
    }
    compare_by_model($outdir, $hh_sqlena_ref, $ndir, \@title, \@key,
		     $N, $k,
		     \@binused,
		     \@tpa, \@tpb, \@tpc, 
		     \@d1a, \@d1b, \@d1c, 
		     \@d2a, \@d2b, \@d2c);

}

sub calculate_all_maxtime_meanabl {
    my ($ndir, 
	$hh_msalen_ref, $hh_sqleng_ref, $hh_sqlena_ref,
	$hp_msalen_ref, $hp_sqleng_ref, $hp_sqlena_ref) = @_;

    my $xlabel_msalen = "ALIGNMENT LENGTH";
    for (my $n = 0; $n < $ndir; $n ++) {
	calculate_maxtime_meanabl($hh_msalen_ref->[$n], $hp_msalen_ref->[$n], $xlabel_msalen);
    }   

    my $xlabel_sqleng = "GEOMETRIC MEAN SEQUENCES LENGTH";
    for (my $n = 0; $n < $ndir; $n ++) {
	calculate_maxtime_meanabl($hh_sqleng_ref->[$n], $hp_sqleng_ref->[$n], $xlabel_sqleng);
    }   

    my $xlabel_sqlena = "ARITHMETIC MEAN SEQUENCES LENGTH";
    for (my $n = 0; $n < $ndir; $n ++) {
	calculate_maxtime_meanabl($hh_sqlena_ref->[$n], $hp_sqlena_ref->[$n], $xlabel_sqlena);
    }   
    
}

sub calculate_all_meantp {
    my ($ndir, 
	$hh_msalen_ref, $hh_sqleng_ref, $hh_sqlena_ref) = @_;

    my $xlabel_msalen = "ALIGNMENT LENGTH";
    for (my $n = 0; $n < $ndir; $n ++) {
	calculate_meantp($hh_msalen_ref->[$n], $xlabel_msalen);
    }   

    my $xlabel_sqleng = "GEOMETRIC MEAN SEQUENCES LENGTH";
    for (my $n = 0; $n < $ndir; $n ++) {
	calculate_meantp($hh_sqleng_ref->[$n], $xlabel_sqleng);
    }   

    my $xlabel_sqlena = "ARITHMETIC MEAN SEQUENCES LENGTH";
    for (my $n = 0; $n < $ndir; $n ++) {
	calculate_meantp($hh_sqlena_ref->[$n], $xlabel_sqlena);
    }   
    
}

sub calculate_chi_square {
    my ($hh_file, $hp_file, $xlabel,
	$title_ref, $key_ref,
	$n, 
	$N, $k,
	$binused_ref,
	$tpa_ref, $tpb_ref, $tpc_ref,   
	$d1a_ref, $d1b_ref, $d1c_ref,  
	$d2a_ref, $d2b_ref, $d2c_ref) = @_;

    my $title;
    my $key;
    my $hh_dgf = 0;
    my $hp_dgf = 0;

    my $chi2_tp   = 0;
    my $chi2_d1   = 0;
    my $chi2_d2   = 0;
    my $chi2_like = 0;
    my $chi2_abl  = 0;

    my $chi4_tp   = 0;
    my $chi4_d1   = 0;
    my $chi4_d2   = 0;
    my $chi4_like = 0;
    my $chi4_abl  = 0;

    my $area_diff2_tp   = 0;
    my $area_diff2_d1   = 0;
    my $area_diff2_d2   = 0;

    my $area_diff4_tp   = 0;
    my $area_diff4_d1   = 0;
    my $area_diff4_d2   = 0;

    my $pmaxdiff2_tp = 0; 
    my $avediff2_tp = 0;
    my $avediff2_d1 = 0;
    my $avediff2_d2 = 0;

    my $stddiff2_tp = 0;
    my $stddiff2_d1 = 0;
    my $stddiff2_d2 = 0;

    my $avediff4_tp = 0;
    my $avediff4_d1 = 0;
    my $avediff4_d2 = 0;

    my $stddiff4_tp = 0;
    my $stddiff4_d1 = 0;
    my $stddiff4_d2 = 0;

    my $maxdiff2_tp = 0; 
    my $maxdiff2_d1 = 0;
    my $maxdiff2_d2 = 0; 
    my $mindiff2_tp = 0; 
    my $mindiff2_d1 = 0; 
    my $mindiff2_d2 = 0; 

    my $maxdiff2_tp_at = 0;
    my $maxdiff2_d1_at = 0;
    my $maxdiff2_d2_at = 0;
    my $mindiff2_tp_at = 0;
    my $mindiff2_d1_at = 0;
    my $mindiff2_d2_at = 0;
    
    my $maxdiff4_tp = 0; 
    my $maxdiff4_d1 = 0; 
    my $maxdiff4_d2 = 0;
    my $mindiff4_tp = 0;
    my $mindiff4_d1 = 0;
    my $mindiff4_d2 = 0;
    
    my $maxdiff4_tp_at = 0;
    my $maxdiff4_d1_at = 0;
    my $maxdiff4_d2_at = 0;
    my $mindiff4_tp_at = 0;
    my $mindiff4_d1_at = 0;
    my $mindiff4_d2_at = 0;

    my $target_tp = 200;
    my $diff2_tp_target;
    my $diff4_tp_target;

    my $target_d1 = 200;
    my $diff2_d1_target;
    my $diff4_d1_target;

    my $target_d2 = 200;
    my $diff2_d2_target;
    my $diff4_d2_target;

    my $outfile = "$hh_file.chi";

    parse_histo_header($hh_file, \$title, \$key);
    $title_ref->[$n] = $title;
    $key_ref->[$n]   = $key;

    parse_hh_file($hh_file, \$hh_dgf, 
		  \$chi2_tp,     \$chi4_tp, 
		  \$area_diff2_tp,    \$area_diff4_tp, 
		  \$maxdiff2_tp, \$maxdiff2_tp_at, 
		  \$maxdiff4_tp, \$maxdiff4_tp_at, 
		  \$mindiff2_tp, \$mindiff2_tp_at, 
		  \$mindiff4_tp, \$mindiff4_tp_at,
		  \$avediff2_tp, \$stddiff2_tp,   
		  \$avediff4_tp, \$stddiff4_tp, 
		  $target_tp, \$diff2_tp_target, \$diff4_tp_target, 
		  \$chi2_d1,     \$chi4_d1, 
		  \$area_diff2_d1,    \$area_diff4_d1, 
		  \$maxdiff2_d1, \$maxdiff2_d1_at,
		  \$maxdiff4_d1, \$maxdiff4_d1_at,
		  \$mindiff2_d1, \$mindiff2_d1_at,
		  \$mindiff4_d1, \$mindiff4_d1_at,
		  \$avediff2_d1, \$stddiff2_d1,
		  \$avediff4_d1, \$stddiff4_d1,
		  $target_d1, \$diff2_d1_target, \$diff4_d1_target, 
		  \$chi2_d2,     \$chi4_d2, 
		  \$area_diff2_d2,    \$area_diff4_d2, 
		  \$maxdiff2_d2, \$maxdiff2_d2_at,
		  \$maxdiff4_d2, \$maxdiff4_d2_at,
		  \$mindiff2_d2, \$mindiff2_d2_at,
		  \$mindiff4_d2, \$mindiff4_d2_at,
		  \$avediff2_d2, \$stddiff2_d2,
		  \$avediff4_d2, \$stddiff4_d2,
		  $target_d2, \$diff2_d2_target, \$diff4_d2_target, 
		  $n, 
		  $N, $k,
		  $binused_ref,
		  $tpa_ref, $tpb_ref, $tpc_ref, 
		  $d1a_ref, $d1b_ref, $d1c_ref, 
		  $d2a_ref, $d2b_ref, $d2c_ref);

    parse_hp_file($hp_file, \$hp_dgf, \$chi2_like, \$chi4_like, \$chi2_abl, \$chi4_abl);

    open(OUT, ">$outfile");
    print OUT "$title\n";
    print OUT "$key\n";
    print OUT "AREA DIFF\n";
    print OUT "\nTP\n$hh_dgf\t$area_diff2_tp\t$area_diff4_tp\n";
    print OUT "\nD1\n$hh_dgf\t$area_diff2_d1\t$area_diff4_d1\n";
    print OUT "\nD2\n$hh_dgf\t$area_diff2_d2\t$area_diff4_d2\n";

    print OUT "\nCHI_SQUARE\n";
    print OUT "\nTP\n$hh_dgf\t$chi2_tp\t$chi4_tp\n";
    print OUT "\nD1\n$hh_dgf\t$chi2_d1\t$chi4_d1\n";
    print OUT "\nD2\n$hh_dgf\t$chi2_d2\t$chi4_d2\n";
    print OUT "\nLIKE\n$hp_dgf\t$chi2_like\t$chi4_like\n";
    print OUT "\nABL\n$hp_dgf\t$chi2_abl\t$chi4_abl\n";

    print OUT "\nAVE DIFF [dnaml-erate.2 - dnaml]\n";
    print OUT "\nTP\t$avediff2_tp +/- $stddiff2_tp\n";
    print OUT "\nD1\t$avediff2_d1 +/- $stddiff2_d1\n";
    print OUT "\nD2\t$avediff2_d2 +/- $stddiff2_d2\n";

    print OUT "max DIFF [dnaml-erate.2 - dnaml]\n";
    print OUT "\nTP\t$maxdiff2_tp ($maxdiff2_tp_at)\n";
    print OUT "\nD1\t$maxdiff2_d1 ($maxdiff2_d1_at)\n";
    print OUT "\nD2\t$maxdiff2_d2 ($maxdiff2_d2_at)\n";

    print OUT "min DIFF [dnaml-erate.2 - dnaml]\n";
    print OUT "\nTP\t$mindiff2_tp ($mindiff2_tp_at)\n";
    print OUT "\nD1\t$mindiff2_d1 ($mindiff2_d1_at)\n";
    print OUT "\nD2\t$mindiff2_d2 ($mindiff2_d2_at)\n";

    print OUT "target DIFF [dnaml-erate.2 - dnaml]\n";
    print OUT "\nTP\t$diff2_tp_target ($target_tp)\n";
    print OUT "\nD1\t$diff2_d1_target ($target_d1)\n";
    print OUT "\nD2\t$diff2_d2_target ($target_d2) \n";

    print OUT "\nAVE DIFF [dnaml-erate.4 - dnaml]\n";
    print OUT "\nTP\t$avediff4_tp +/- $stddiff4_tp\n";
    print OUT "\nD1\t$avediff4_d1 +/- $stddiff4_d1\n";
    print OUT "\nD2\t$avediff4_d2 +/- $stddiff4_d2\n";
    
    print OUT "max DIFF [dnaml-erate.4 - dnaml]\n";
    print OUT "\nTP\t$maxdiff4_tp ($maxdiff4_tp_at)\n";
    print OUT "\nD1\t$maxdiff4_d1 ($maxdiff4_d1_at)\n";
    print OUT "\nD2\t$maxdiff4_d2 ($maxdiff4_d2_at)\n";

    print OUT "min DIFF [dnaml-erate.4 - dnaml]\n";
    print OUT "\nTP\t$mindiff4_tp ($mindiff4_tp_at)\n";
    print OUT "\nD1\t$mindiff4_d1 ($mindiff4_d1_at)\n";
    print OUT "\nD2\t$mindiff4_d2 ($mindiff4_d2_at)\n";

    print OUT "target DIFF [dnaml-erate.4 - dnaml]\n";
    print OUT "\nTP\t$diff4_tp_target ($target_tp)\n";
    print OUT "\nD1\t$diff4_d1_target ($target_d1)\n";
    print OUT "\nD2\t$diff4_d2_target ($target_d2) \n";

    close(OUT);   
}

sub calculate_max_diff {
    my ($ndir, $m_ref, $merr_ref, $a_ref, $aerr_ref, $max_diff_ref, $max_diff_at_ref)  = @_;

    if ($ndir != 2) { return; }

    my $xmin = 50;
    my $xmax = 1000;
    my $xinc = 5;

    my $logx;
    my $f1;
    my $f2;
    my $diff;

    my $pmax_diff = -1;
    my $pmax_diff_at = 0;
    my $nmax_diff = -1;
    my $nmax_diff_at = 0;

    my $winner;
    my $max_diff;
    my $max_diff_at;

    for (my $x = $xmin; $x < $xmax; $x += $xinc) {

	my $logx = log($x);
	
	$f1 = $a_ref->[0] + $m_ref->[0]*$logx;
	$f2 = $a_ref->[1] + $m_ref->[1]*$logx;
	
	$diff = exp($f1) - exp($f2);
	
	print "x=$x f1 $f1 ", exp($f1), " f2 $f2 ", exp($f2), " $diff\n";
	if ($diff > 0) { if ($pmax_diff <     $diff)  { $pmax_diff =      $diff; $pmax_diff_at = $x; } }
	if ($diff < 0) { if ($nmax_diff < abs($diff)) { $nmax_diff = abs($diff); $nmax_diff_at = $x; } }	
    }

    if ($pmax_diff > $nmax_diff) { $max_diff = $pmax_diff; $max_diff_at = $pmax_diff_at; }
    if ($pmax_diff < $nmax_diff) { $max_diff = $nmax_diff; $max_diff_at = $nmax_diff_at; }

    if (1) {
	print "PMAX $pmax_diff($pmax_diff_at)\n";
	print "NMAX $nmax_diff($nmax_diff_at)\n";
	print "MAX $max_diff($max_diff_at)\n";
    }

    $$max_diff_ref    = $max_diff;
    $$max_diff_at_ref = $max_diff_at;
}

sub calculate_meantp {
    my ($hh_file, $xlabel) = @_;

    my $title;
    my $key;

    my $ave_tp_a = 0.0;
    my $std_tp_a = 0.0;
    my $ave_tp_b = 0.0;
    my $std_tp_b = 0.0;
    my $ave_tp_c = 0.0;
    my $std_tp_c = 0.0;

    my $outfile = "$hh_file.tp";

    parse_histo_header($hh_file, \$title, \$key);
    parse_hh_file_tp($hh_file, \$ave_tp_a, \$std_tp_a, \$ave_tp_b, \$std_tp_b, \$ave_tp_c, \$std_tp_c);

    open(OUT, ">$outfile");
    print OUT "$title\n";
    print OUT "$key\n";
    print OUT "\nTP\n";
    print OUT "DNAML\t$ave_tp_a +\/- $std_tp_a\n";
    print OUT "DNAML-ERATE.2\t$ave_tp_b +\/- $std_tp_b\n";
    print OUT "DNAML-ERATE.4\t$ave_tp_c +\/- $std_tp_c\n";
    close(OUT);

}

sub calculate_maxtime_meanabl {
    my ($hh_file, $hp_file, $xlabel) = @_;

    my $title;
    my $key;

    my $max_time_ave_a = 0.0;
    my $max_time_std_a = 0.0;
    my $max_time_ave_b = 0.0;
    my $max_time_std_b = 0.0;
    my $max_time_ave_c = 0.0;
    my $max_time_std_c = 0.0;

    my $ave_abl_a = 0.0;
    my $std_abl_a = 0.0;
    my $ave_abl_b = 0.0;
    my $std_abl_b = 0.0;
    my $ave_abl_c = 0.0;
    my $std_abl_c = 0.0;

    my $outfile = "$hh_file.time_abl";

    parse_histo_header($hh_file, \$title, \$key);
    parse_hh_file_time($hh_file, \$max_time_ave_a, \$max_time_std_a, \$max_time_ave_b, \$max_time_std_b, \$max_time_ave_c, \$max_time_std_c);
    parse_hp_file_abl($hp_file, \$ave_abl_a, \$std_abl_a, \$ave_abl_b, \$std_abl_b, \$ave_abl_c, \$std_abl_c);

    open(OUT, ">$outfile");
    print OUT "$title\n";
    print OUT "$key\n";
    print OUT "\nmax TIME\n";
    print OUT "DNAML\t$max_time_ave_a +\/- $max_time_std_a\n";
    print OUT "DNAML-ERATE.2\t$max_time_ave_b +\/- $max_time_std_b\n";
    print OUT "DNAML-ERATE.4\t$max_time_ave_c +\/- $max_time_std_c\n";
    print OUT "\nABL\n";
    print OUT "DNAML\t$ave_abl_a +\/- $std_abl_a\n";
    print OUT "DNAML-ERATE.2\t$ave_abl_b +\/- $std_abl_b\n";
    print OUT "DNAML-ERATE.4\t$ave_abl_c +\/- $std_abl_c\n";
    close(OUT);
    
}

sub compare_by_model {
    my ($outdir, $filehisto_ref,
	$ndir, 
	$title_ref, $key_ref,
	$N, $k,
	$binused_ref,
	$tpa_ref, $tpb_ref, $tpc_ref,  
	$d1a_ref, $d1b_ref, $d1c_ref, 
	$d2a_ref, $d2b_ref, $d2c_ref, ) = @_;

    if ($ndir != 2) { return; }

    my @histoname;
    for (my $n = 0; $n < $ndir; $n ++) { if ($filehisto_ref->[$n] =~ /\/([^\/]+)\/[^\/]+$/) { $histoname[$n] = $1; } }

    my $outfile = "$outdir/$histoname[0]";
    for (my $n = 1; $n < $ndir; $n ++) { $outfile .= "_$histoname[$n]"; }
    $outfile .= "_compare";

    my $case;

    system("rm $outfile\n");
    $case = "DNAML";
    compare_two_values($outfile, $case, "TP",  $N, $k, $binused_ref, $tpa_ref, $title_ref, $key_ref);
    compare_two_values($outfile, $case, "BSD", $N, $k, $binused_ref, $d1a_ref, $title_ref, $key_ref);
    compare_two_values($outfile, $case, "SDD", $N, $k, $binused_ref, $d2a_ref, $title_ref, $key_ref);

    $case = "DNAML-erate.2";
    compare_two_values($outfile, $case, "TP",  $N, $k, $binused_ref, $tpb_ref, $title_ref, $key_ref);
    compare_two_values($outfile, $case, "BSD", $N, $k, $binused_ref, $d1b_ref, $title_ref, $key_ref);
    compare_two_values($outfile, $case, "SDD", $N, $k, $binused_ref, $d2b_ref, $title_ref, $key_ref);

    $case = "DNAML-erate.4";
    compare_two_values($outfile, $case, "TP",  $N, $k, $binused_ref, $tpc_ref, $title_ref, $key_ref);
    compare_two_values($outfile, $case, "BSD", $N, $k, $binused_ref, $d1c_ref, $title_ref, $key_ref);
    compare_two_values($outfile, $case, "SDD", $N, $k, $binused_ref, $d2c_ref, $title_ref, $key_ref);

}

sub compare_two_values {
    my ($outfile, $case, $val, $N, $k, $binused_ref, $val_ref, $title_ref, $key_ref) = @_;

    my $maxdiff = -100;
    my $mindiff =  100;

    my $maxdiff_at   = 0;
    my $mindiff_at   = 0;

    my $avediff = 0;
    my $stddiff = 0;

    my $val1;
    my $val2;
    my $diff;
    my $val_at;

    my $target_at = 200;
    my $diff_at_target;

    my $c = 0;

    my $winner;

    my $dim = $N*$k;

    for (my $i = 0; $i < $dim; $i ++) {
	$val1 = $val_ref->[$i];
	$val2 = $val_ref->[$dim+$i];

	#print ">>$val idx1 $x  val1 $val1 val_at $val_at_ref->[$x] idx2 ", $dim1+$x," val2 $val2 val_at $val_at_ref->[$dim1+$x] \n";

	$val_at = $i/$k;

	$diff = $val2 - $val1; 

	if ($binused_ref->[$i] > 0 || $binused_ref->[$dim+$i] > 0) {
	    $c ++;
	    accumulate_averages($diff, \$avediff, \$stddiff);
	}
	
	if ($val_at > 50  && $val_at < 1000) {
	    if ($maxdiff < $diff) { $maxdiff = $diff; $maxdiff_at = $val_at; } 
	    if ($mindiff > $diff) { $mindiff = $diff; $mindiff_at = $val_at; } 
	}

	if ($val_at == $target_at) { $diff_at_target = $diff; }
    }
    
    calculate_averages(\$avediff, \$stddiff, $c);

    open(OUT, ">>$outfile");
    print OUT "\n$case\n";
    print OUT "$val\n";
    print OUT "C1:\n$title_ref->[0]\n$key_ref->[0]\n";
    print OUT "C2:\n$title_ref->[1]\n$key_ref->[1]\n";

    print OUT "AVE DIFF [C2-C1]\n";
    print OUT "$avediff +/- $stddiff\n";


    print OUT "MAX DIFF [C2-C1]:\n";
    print OUT "$maxdiff\t$maxdiff_at\n";

    print OUT "MIN DIFF [C2-C1]:\n";
    print OUT "$mindiff\t$mindiff_at\n";

    print OUT "AT $target_at DIFF [C2-C1]:\n";
    print OUT "$diff_at_target\n";

    close(OUT);
}

sub extract_exp_fit_info {

    my ($fitlog, $plotfile, $m_ref, $a_ref) = @_;

    my $read = 0;

    open (FIT,"$fitlog") || die;
    while (<FIT>) {
	
	if (/^FIT:.+$plotfile/) { $read = 1; }
	elsif (/^FIT:/)         { $read = 0; }

	if    (/^m\s+=\s+(\S+)\s+\+\/\-\s+\S+\s+(\(\S+\%\))/)  { if ($read == 1) { $$m_ref  = $1.$2; } }
	elsif (/^a\s+=\s+(\S+)\s+\+\/\-\s+\S+\s+(\(\S+\%\))/)  { if ($read == 1) { $$a_ref  = $1.$2; } }
    }
    close (FIT);
 
}

sub fit_all_curves {
    my ($ndir, $hh_ref, $xlabel, 
	$m_tp_ref, $merr_tp_ref, $a_tp_ref, $aerr_tp_ref,
	$m_d1_ref, $merr_d1_ref, $a_d1_ref, $aerr_d1_ref,
	$m_d2_ref, $merr_d2_ref, $a_d2_ref, $aerr_d2_ref) = @_;

    for (my $n = 0; $n < $ndir; $n ++) {
	fit_three_curves($hh_ref->[$n], $xlabel, 
			 \$m_tp_ref->[$n], \$merr_tp_ref->[$n], \$a_tp_ref->[$n], \$aerr_tp_ref->[$n],
			 \$m_d1_ref->[$n], \$merr_d1_ref->[$n], \$a_d1_ref->[$n], \$aerr_d1_ref->[$n],
			 \$m_d2_ref->[$n], \$merr_d2_ref->[$n], \$a_d2_ref->[$n], \$aerr_d2_ref->[$n]);
	
    }
}

sub fit_three_curves {
    my ($hfile, $xlabel, 
	$m_tp_ref, $merr_tp_ref, $a_tp_ref, $aerr_tp_ref,
	$m_d1_ref, $merr_d1_ref, $a_d1_ref, $aerr_d1_ref,
	$m_d2_ref, $merr_d2_ref, $a_d2_ref, $aerr_d2_ref) = @_;
    
    fit_one_curve($hfile, "TP", $m_tp_ref, $merr_tp_ref, $a_tp_ref, $aerr_tp_ref);
    fit_one_curve($hfile, "D1", $m_d1_ref, $merr_d1_ref, $a_d1_ref, $aerr_d1_ref);
    fit_one_curve($hfile, "D2", $m_d2_ref, $merr_d2_ref, $a_d2_ref, $aerr_d2_ref);

}

sub extract_tpfit_logvals {
    my ($hfile, $tmpfile) = @_;
    
    my $x;
    my $tp;
    my $tp1;
    my $tp2;
    my $tp3;

    open (TMP,">$tmpfile") || die;
    open (FILE,"$hfile") || die;
    while(<FILE>) {

	if (/^(\S+)\s+\S+\s+\S+\s+(\S+)\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+(\S+)\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+(\S+)\s+\S+\s+\S+\s+\S+\s+/) {
	    $x   = $1;
	    $tp1 = $2;
	    $tp2 = $3;
	    $tp3 = $4;

	    $tp = $tp3;
	    if ($opt_x) { $tp = $tp1; }
	    if ($opt_y) { $tp = $tp2; }
	    
	    #print "$x $tp\n";

	    if ($x > 0 && $tp > 0) {
		printf(TMP "%.4f %.4f\n", log($x), log($tp));
	    }
	}  
    }
    close (FILE);
    close(TMP);
}

sub extract_d1fit_logvals {
    my ($hfile, $tmpfile) = @_;
    
    my $x;
    my $d1;
    my $d11;
    my $d12;
    my $d13;

    open (TMP,">$tmpfile") || die;
    open (FILE,"$hfile") || die;
    while(<FILE>) {

	if (/^(\S+)\s+\S+\s+\S+\s+\S+\s+(\S+)\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+(\S+)\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+(\S+)\s+\S+\s+\S+\s+/) {
	    $x   = $1;
	    $d11 = $2;
	    $d12 = $3;
	    $d13 = $4;

	    $d1 = $d13;
	    if ($opt_x) { $d1 = $d11; }
	    if ($opt_y) { $d1 = $d12; }
	    
	    if ($x > 0 && $d1 > 0) {
		printf(TMP "%.4f %.4f\n", log($x), log($d1));
	    }
	}  
    }
    close (FILE);
    close(TMP);
}

sub extract_d2fit_logvals {
    my ($hfile, $tmpfile) = @_;
    
    my $x;
    my $d2;
    my $d21;
    my $d22;
    my $d23;

    open (TMP,">$tmpfile") || die;
    open (FILE,"$hfile") || die;
    while(<FILE>) {

	if (/^(\S+)\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+(\S+)\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+(\S+)\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+(\S+)\s+/) {
	    $x   = $1;
	    $d21 = $2;
	    $d22 = $3;
	    $d23 = $4;

	    $d2 = $d23;
	    if ($opt_x) { $d2 = $d21; }
	    if ($opt_y) { $d2 = $d22; }
	    
	    if ($x > 0 && $d2 > 0) {
		printf(TMP "%.4f %.4f\n", log($x), log($d2));
	    }
	}  
    }
    close (FILE);
    close(TMP);
}

sub fit_one_curve {
    my ($hfile, $which, $m_ref, $merr_ref, $a_ref, $aerr_ref) = @_;

    my $m    = 0;
    my $merr = 0;
    my $a    = 0;
    my $aerr = 0;

    my $tmpfile = "$hfile.tmp";

    if   ($which =~ /^TP$/) { extract_tpfit_logvals($hfile, $tmpfile); }
    elsif($which =~ /^D1$/) { extract_d1fit_logvals($hfile, $tmpfile); } 
    elsif($which =~ /^D2$/) { extract_d2fit_logvals($hfile, $tmpfile); }
    else                    { print "bad case which=$which\n"; die;   }

    open(GP,'|'.GNUPLOT) || die "Gnuplot: $!";
    print GP "fe(x) = m*x + a\n"; 
    print GP "fit fe(x) '$tmpfile' using 1:2  via m,a \n";
    close (GP);
    
    extract_exp_fit_info ("fit.log", "$hfile", \$m, \$a);
    
    $m =~ /(\S+)(\(\S+\%\))/;  $m = $1; $merr = $2;
    $a =~ /(\S+)(\(\S+\%\))/;  $a = $1; $aerr = $2;
    
    $$m_ref    = $m;
    $$merr_ref = $merr;
    $$a_ref    = $a;
    $$aerr_ref = $aerr;
    
    system ("rm 'fit.log'\n"); 
    system ("rm $tmpfile\n"); 
}

sub get_file {
    my ($dir, $index) = @_;

    my @file;

    sorted_files($dir, \@file, $index);
    if ($file[1]) { print "there is more than one file\n $file[0]\n$file[1]\n"; die; }

    return $file[0];
}

sub sorted_files {
    my ($dir, $files_ref, $index) = @_;

    local *DIRH;

    opendir DIRH, $dir or die "eh? $dir: $!";

    @$files_ref = grep { /^\S+$index$/ }
    map { "$dir/$_" } readdir DIRH;

    sort @$files_ref;
}

sub gnuplot_jointly_hhhisto {

    my ($ndir, $filehisto_ref, $title_ref, $xlabel, $key_ref, $seeplots, 
	$m_tp_ref, $merr_tp_ref, $a_tp_ref, $aerr_tp_ref,
	$m_d1_ref, $merr_d1_ref, $a_d1_ref, $aerr_d1_ref, 
	$m_d2_ref, $merr_d2_ref, $a_d2_ref, $aerr_d2_ref) = @_;

    # Itemized plots
    plot_jointly_tp($ndir, $filehisto_ref, $title_ref, $xlabel, $key_ref, $seeplots, 
		    $m_tp_ref, $merr_tp_ref, $a_tp_ref, $aerr_tp_ref);
    plot_jointly_d1($ndir, $filehisto_ref, $title_ref, $xlabel, $key_ref, $seeplots,
		    $m_d1_ref, $merr_d1_ref, $a_d1_ref, $aerr_d1_ref);
    plot_jointly_d2($ndir, $filehisto_ref, $title_ref, $xlabel, $key_ref, $seeplots,
		    $m_d2_ref, $merr_d2_ref, $a_d2_ref, $aerr_d2_ref);
    if (!$opt_w && !$opt_z) { plot_jointly_time($ndir, $filehisto_ref, $title_ref, $xlabel, $key_ref, $seeplots); }
}

sub gnuplot_hhhisto {

    my ($filehisto, $title, $xlabel, $key, $seeplots) = @_;

    my $outplot = "$filehisto.ps";
   
    # Itemized plots
    plot_tp($filehisto, $title, $xlabel, $key, $seeplots);
    plot_d1($filehisto, $title, $xlabel, $key, $seeplots);
    plot_d2($filehisto, $title, $xlabel, $key, $seeplots);
    plot_time($filehisto, $title, $xlabel, $key, $seeplots);
    
    # A file with all the plots
    $seeplots = 0;
    my $ylabel;

    open(GP,'|'.GNUPLOT) || die "Gnuplot: $!";

    print GP "set terminal postscript color solid 14\n";
    print GP "set style line 1 lt 1 lw 4\n";
    print GP "set style line 2 lt 2 lw 4\n";
    print GP "set style line 3  lt 3 lw 4\n";
    print GP "set style line 4 lt 4 lw 4\n";
    print GP "set style line 5 lt 5 lw 4\n";
    print GP "set style line 6 lt 6 lw 4\n";
    print GP "set style line 7 lt 7 lw 4\n";
    print GP "set style line 8 lt 8 lw 4\n";
    print GP "set style line 9 lt 9 lw 4\n";
    print GP "set style line 10 lt 10 lw 4\n";
    print GP "set style line 11 lt 11 lw 4\n";
    print GP "set style line 12 lt 12 lw 4\n";
    print GP "set style line 13 lt 13 lw 4\n";
    print GP "set style line 14 lt 14 lw 4\n";
    print GP "set style line 15 lt 15 lw 5\n";

    print GP "set output '$outplot'\n";
    #print GP "set nokey\n";
    print GP "set xlabel '$xlabel'\n";

    print GP "set title \"$title\\n\\n$key\"\n";
    my $newkeyx = "$dnamlx";
    my $newkeye = "$dnamle";
    my $newkeyo = "$dnamlo";
    my $newkeyxe = "$dnamlxe";
    my $newkeyxo = "$dnamlxo";

    my $lsx = "3";
    my $lse = "2";
    my $lso = "1";
    my $lsxe = "2";
    my $lsxo = "1";

    if ($opt_b || $opt_o) {
	$ylabel = "fraction TRUE POSITIVE TREES  (TOPOLOGY)";
	print GP "set ylabel '$ylabel'\n";
	print GP "plot '$filehisto' using 1:4 title '$newkeyx' ls $lsx, '$filehisto' using 1:13 title '$newkeye' ls $lse \n";

	$ylabel = "TREE  DISTANCE (Branch Score Distance)";
	print GP "set ylabel '$ylabel'\n";
	print GP "plot '$filehisto' using 1:5:6 with yerrorbars title '$newkeyx' ls $lsx, '$filehisto' using 1:14:15  with yerrorbars title '$newkeye' ls $lse\n";

	$ylabel = "TREE  DISTANCE (Symmetric Difference Distance)";
	print GP "set ylabel '$ylabel'\n";
	print GP "plot '$filehisto' using 1:7:8 with yerrorbars title '$newkeyx' ls $lsx, '$filehisto' using 1:16:17  with yerrorbars title '$newkeye' ls $lse\n";

	$ylabel = "RUN TIME (sec)";
	print GP "set ylabel '$ylabel'\n";
	print GP "plot '$filehisto' using 1:9:10 with yerrorbars title '$newkeyx' ls $lsx, '$filehisto' using 1:18:19  with yerrorbars title '$newkeye' ls $lse\n";
    
	if ($opt_o) {
	    $ylabel = "fraction TRUE POSITIVE TREES  (TOPOLOGY)";
	    print GP "set ylabel '$ylabel'\n";
	    print GP "plot '$filehisto' using 1:4 title '$newkeyx' ls $lsx, '$filehisto' using 1:22 title '$newkeyo' ls $lso\n";
	    
	    $ylabel = "TREE  DISTANCE (Branch Score Distance)";
	    print GP "set ylabel '$ylabel'\n";
	    print GP "plot '$filehisto' using 1:5:6 with yerrorbars title '$newkeyx' ls $lsx, '$filehisto' using 1:23:24  with yerrorbars title '$newkeyo' ls $lso\n";
	    
	    $ylabel = "TREE  DISTANCE (Symmetric Difference Distance)";
	    print GP "set ylabel '$ylabel'\n";
	    print GP "plot '$filehisto' using 1:7:8 with yerrorbars title '$newkeyx' ls $lsx, '$filehisto' using 1:25:26  with yerrorbars title '$newkeyo' ls $lso\n";
	    
	    $ylabel = "RUN TIME (sec)";
	    print GP "set ylabel '$ylabel'\n";
	    print GP "plot '$filehisto' using 1:9:10 with yerrorbars title '$newkeyx' ls $lsx, '$filehisto' using 1:27:28  with yerrorbars title '$newkeyo' ls $lso\n";
	    
	    $ylabel = "fraction TRUE POSITIVE TREES  (TOPOLOGY)";
	    print GP "set ylabel '$ylabel'\n";
	    print GP "plot '$filehisto' using 1:4 title '$newkeyx' ls $lsx, '$filehisto' using 1:13 title '$newkeye' ls $lse, '$filehisto' using 1:22 title '$newkeyo' ls $lso\n";
	    
	    $ylabel = "TREE  DISTANCE  (Branch Score Distance)";
	    print GP "set ylabel '$ylabel'\n";
	    print GP "plot '$filehisto' using 1:5:6 with yerrorbars title '$newkeyx' ls $lsx, '$filehisto' using 1:14:15  with yerrorbars title '$newkeye' ls $lse, '$filehisto' using 1:23:24  with yerrorbars title '$newkeyo' ls $lso\n";
	    
	    $ylabel = "TREE  DISTANCE  (Symmetric Difference Distance)";
	    print GP "set ylabel '$ylabel'\n";
	    print GP "plot '$filehisto' using 1:7:8 with yerrorbars title '$newkeyx' ls $lsx, '$filehisto' using 1:16:17  with yerrorbars title '$newkeye' ls $lse, '$filehisto' using 1:25:26  with yerrorbars title '$newkeyo' ls $lso\n";
	    
	    $ylabel = "RUN TIME (sec)";
	    print GP "set ylabel '$ylabel'\n";
	    print GP "plot '$filehisto' using 1:9:10 with yerrorbars title '$newkeyx' ls $lsx, '$filehisto' using 1:18:19  with yerrorbars title '$newkeye' ls $lse, '$filehisto' using 1:27:28  with yerrorbars title '$newkeyo' ls $lso\n";

	    $ylabel = "fraction TRUE POSITIVE TREES  (TOPOLOGY)";
	    print GP "set ylabel '$ylabel'\n";
	    print GP "plot '$filehisto' using 1:31 title '$newkeyxe' ls $lsxe\n";
	    
	    $ylabel = "TREE  DISTANCE (Branch Score Distance)";
	    print GP "set ylabel '$ylabel'\n";
	    print GP "plot '$filehisto' using 1:32:33 with yerrorbars title '$newkeyxe' ls $lsxe\n";
	    
	    $ylabel = "TREE  DISTANCE (Symmetric Difference Distance)";
	    print GP "set ylabel '$ylabel'\n";
	    print GP "plot '$filehisto' using 1:34:35 with yerrorbars title '$newkeyxe' ls $lsxe\n";

	    $ylabel = "fraction TRUE POSITIVE TREES  (TOPOLOGY)";
	    print GP "set ylabel '$ylabel'\n";
	    print GP "plot '$filehisto' using 1:38 title '$newkeyxe' ls $lsxe\n";
	    
	    $ylabel = "TREE  DISTANCE (Branch Score Distance)";
	    print GP "set ylabel '$ylabel'\n";
	    print GP "plot '$filehisto' using 1:39:40 with yerrorbars title '$newkeyxe' ls $lsxe\n";
	    
	    $ylabel = "TREE  DISTANCE (Symmetric Difference Distance)";
	    print GP "set ylabel '$ylabel'\n";
	    print GP "plot '$filehisto' using 1:41:42 with yerrorbars title '$newkeyxe' ls $lsxe\n";
	    

	}
    }
    else {
	my $singlekey = "$dnamlx";

	my $ls = $lsx;
	if ($opt_e) { $ls = $lse; }

	$ylabel = "fraction TRUE POSITIVE TREES  (TOPOLOGY)";
	print GP "set ylabel '$ylabel'\n";
	print GP "plot '$filehisto' using 1:4 title '$singlekey' ls $ls\n";

	$ylabel = "TREE  DISTANCE  (Branch Score Distance)";
	print GP "set ylabel '$ylabel'\n";
	print GP "plot '$filehisto' using 1:5:6  with yerrorbars title '$singlekey' ls $ls\n";

	$ylabel = "TREE  DISTANCE  (Symmetric Difference  Distance)";
	print GP "set ylabel '$ylabel'\n";
	print GP "plot '$filehisto' using 1:7:8  with yerrorbars title '$singlekey' ls $ls\n";
	
	$ylabel = "RUN TIME (sec)";
	print GP "set ylabel '$ylabel'\n";
	print GP "plot '$filehisto' using 1:9:10  with yerrorbars title '$singlekey' ls $ls\n";
    }
    
    close (GP);

    if ($seeplots) { system ("$evince $outplot&\n"); }
}

sub gnuplot_jointly_hphisto {

    my ($ndir, $filehisto_ref, $title_ref, $xlabel, $key_ref, $seeplots) = @_;

    # Itemized plots
    plot_jointly_like($ndir, $filehisto_ref, $title_ref, $xlabel, $key_ref, $seeplots);
    plot_jointly_abl($ndir, $filehisto_ref, $title_ref, $xlabel, $key_ref, $seeplots);
    plot_jointly_param($ndir, $filehisto_ref, $title_ref, $xlabel, $key_ref, $seeplots);
    plot_jointly_ttr($ndir, $filehisto_ref, $title_ref, $xlabel, $key_ref, $seeplots);
  }

sub gnuplot_hphisto {

    my ($filehisto, $title, $xlabel, $key, $seeplots) = @_;

    my $outplot = "$filehisto.ps";
    my $ylabel;

    my $lsx = "3";
    my $lse = "2";
    my $lso = "1";

    # Itemized plots
    plot_like($filehisto, $title, $xlabel, $key, $seeplots);
    plot_abl($filehisto, $title, $xlabel, $key, $seeplots);
    plot_param($filehisto, $title, $xlabel, $key, $seeplots);
    plot_ttr($filehisto, $title, $xlabel, $key, $seeplots);
    
    # A file with all the plots
    $seeplots = 0;
    open(GP,'|'.GNUPLOT) || die "Gnuplot: $!";

    print GP "set terminal postscript color solid 14\n";
    print GP "set style line 1 lt 1 lw 4\n";
    print GP "set style line 2 lt 2 lw 4\n";
    print GP "set style line 3  lt 3 lw 4\n";
    print GP "set style line 4 lt 4 lw 4\n";
    print GP "set style line 5 lt 5 lw 4\n";
    print GP "set style line 6 lt 6 lw 4\n";
    print GP "set style line 7 lt 7 lw 4\n";
    print GP "set style line 8 lt 8 lw 4\n";
    print GP "set style line 9 lt 9 lw 4\n";
    print GP "set style line 10 lt 10 lw 4\n";
    print GP "set style line 11 lt 11 lw 4\n";
    print GP "set style line 12 lt 12 lw 4\n";
    print GP "set style line 13 lt 13 lw 4\n";
    print GP "set style line 14 lt 14 lw 4\n";
    print GP "set style line 15 lt 15 lw 5\n";

    print GP "set output '$outplot'\n";
    print GP "set xlabel '$xlabel'\n";

    print GP "set title \"$title\\n\\n$key\"\n";
    my $newkeyx = "$dnamlx";
    my $newkeye = "$dnamle";
    my $newkeyo = "$dnamlo";

    if ($opt_b || $opt_o) {
	$ylabel = "LN LIKELIHOOD";
	print GP "set ylabel '$ylabel'\n";
	print GP "plot '$filehisto' using 1:3:4  with yerrorbars title '$newkeyx' ls $lsx, '$filehisto' using 1:20:21  with yerrorbars title '$newkeye' ls $lse\n";

	$ylabel = "AVERAGE BRANCH LENGTH";
	print GP "set ylabel '$ylabel'\n";
	print GP "plot '$filehisto' using 1:5:6  with yerrorbars title '$newkeyx' ls $lsx, '$filehisto' using 1:22:23  with yerrorbars title '$newkeye' ls $lse\n";

	$ylabel = "RATE PARAMETERS ** $newkeye";
	print GP "set ylabel '$ylabel'\n";
	print GP "plot '$filehisto' using 1:24:25  with yerrorbars title 'ALPHA' ls 1, '$filehisto' using 1:26:27  with yerrorbars title 'BETA' ls 2, '$filehisto' using 1:28:29  with yerrorbars title 'INS' ls 3, '$filehisto' using 1:30:31  with yerrorbars title 'DEL' ls 4\n";

	#print GP "set yrange [-2:10]\n";
	print GP "plot '$filehisto' using 1:32:33  with yerrorbars title 'TTR' ls 1, '$filehisto' using 1:34:35  with yerrorbars title 'ALPHA + BETA' ls 2\n";
	print GP "unset yrange\n";

    }
    if ($opt_o) {
	
	$ylabel = "LN LIKELIHOOD";
	print GP "set ylabel '$ylabel'\n";
	print GP "plot '$filehisto' using 1:3:4  with yerrorbars title '$newkeyx' ls $lsx, '$filehisto' using 1:20:21  with yerrorbars title '$newkeye' ls $lse, '$filehisto' using 1:37:38  with yerrorbars title '$newkeyo' ls $lso\n";

	$ylabel = "AVERAGE BRANCH LENGTH";
	print GP "set ylabel '$ylabel'\n";
	print GP "plot '$filehisto' using 1:5:6  with yerrorbars title '$newkeyx' ls $lsx, '$filehisto' using 1:22:23  with yerrorbars title '$newkeye' ls $lse, '$filehisto' using 1:39:40  with yerrorbars title '$newkeyo' ls $lso\n";

	$ylabel = "RATE PARAMETERS ** $newkeyo";
	print GP "set ylabel '$ylabel'\n";
	print GP "plot '$filehisto' using 1:41:42  with yerrorbars title 'ALPHA' ls 1, '$filehisto' using 1:43:44  with yerrorbars title 'BETA' ls 2, '$filehisto' using 1:45:46  with yerrorbars title 'INS' ls 3, '$filehisto' using 1:47:48  with yerrorbars title 'DEL' ls 4\n";

	#print GP "set yrange [-2:10]\n";
	print GP "plot '$filehisto' using 1:49:50  with yerrorbars title 'TTR' ls 1, '$filehisto' using 1:51:52  with yerrorbars title 'ALPHA + BETA' ls 2\n";
	print GP "unset yrange\n";
    }
    
    if ($opt_e) { 
	my $singlekey .= "$dnamlx"; 

	my $ls = $lsx;
	if ($opt_e) { $ls = $lse; }
	
	$ylabel = "LN LIKELIHOOD";
	print GP "set ylabel '$ylabel'\n";
	print GP "plot '$filehisto' using 1:3:4  with yerrorbars title '$singlekey' ls $ls\n";

	$ylabel = "AVERAGE BRANCH LENGTH";
	print GP "set ylabel '$ylabel'\n";
	print GP "plot '$filehisto' using 1:5:6  with yerrorbars title '$singlekey' ls $ls\n";

	$ylabel = "RATE PARAMETERS ** $singlekey";
	print GP "set ylabel '$ylabel'\n";
	print GP "plot '$filehisto' using 1:7:8  with yerrorbars title 'ALPHA' ls 1, '$filehisto' using 1:9:10  with yerrorbars title 'BETA' ls 2, '$filehisto' using 1:11:12  with yerrorbars title 'INS' ls 3, '$filehisto' using 1:13:14  with yerrorbars title 'DEL' ls 4\n";

	#print GP "set yrange [-2:10]\n";
	print GP "plot '$filehisto' using 1:15:16  with yerrorbars title 'TTR' ls 1, '$filehisto' using 1:17:18  with yerrorbars title 'ALPHA + BETA' ls 2\n";
	print GP "unset yrange\n";
    }
    
    close (GP);
    
    if ($seeplots) { system ("$evince $outplot&\n"); }
}

# initialize a histogram array
sub init_histo_array {
    my ($N, $k, $his_ref) = @_;
    my $dim = $N * $k;
    for (my $i=0; $i<=$dim; $i++) { $his_ref->[$i] = 0; }    
}

sub parse_hh_file {
    my ($hh_file, $dgf_ref, 
	$chi2_tp_ref,     $chi4_tp_ref, 
	$area_diff2_tp_ref,    $area_diff4_tp_ref, 
	$maxdiff2_tp_ref, $maxdiff2_tp_at_ref, 
	$maxdiff4_tp_ref, $maxdiff4_tp_at_ref, 
	$mindiff2_tp_ref, $mindiff2_tp_at_ref, 
	$mindiff4_tp_ref, $mindiff4_tp_at_ref,		  
	$avediff2_tp_ref, $stddiff2_tp_ref,  
	$avediff4_tp_ref, $stddiff4_tp_ref,
	$target_tp, $diff2_tp_target_ref, $diff4_tp_target_ref, 
	$chi2_d1_ref,     $chi4_d1_ref, 
	$area_diff2_d1_ref,    $area_diff4_d1_ref, 
	$maxdiff2_d1_ref, $maxdiff2_d1_at_ref, 
	$maxdiff4_d1_ref, $maxdiff4_d1_at_ref,
	$mindiff2_d1_ref, $mindiff2_d1_at_ref,
	$mindiff4_d1_ref, $mindiff4_d1_at_ref,
	$avediff2_d1_ref, $stddiff2_d1_ref,
	$avediff4_d1_ref, $stddiff4_d1_ref,
	$target_d1, $diff2_d1_target_ref, $diff4_d1_target_ref, 
	$chi2_d2_ref,     $chi4_d2_ref, 
	$area_diff2_d2_ref,    $area_diff4_d2_ref, 
	$maxdiff2_d2_ref, $maxdiff2_d2_at_ref,
	$maxdiff4_d2_ref, $maxdiff4_d2_at_ref,
	$mindiff2_d2_ref, $mindiff2_d2_at_ref,
	$mindiff4_d2_ref, $mindiff4_d2_at_ref,
	$avediff2_d2_ref, $stddiff2_d2_ref,
	$avediff4_d2_ref, $stddiff4_d2_ref,
	$target_d2, $diff2_d2_target_ref, $diff4_d2_target_ref, 
	$n, 
	$N, $k,
	$binused_ref,
	$tpa_ref, $tpb_ref, $tpc_ref, 
	$d1a_ref, $d1b_ref, $d1c_ref,  
	$d2a_ref, $d2b_ref, $d2c_ref) = @_;
    
    my $dgf = 0;

    my $chi2_tp = 0;
    my $chi4_tp = 0;
    my $chi2_d1 = 0;
    my $chi4_d1 = 0;
    my $chi2_d2 = 0;
    my $chi4_d2 = 0;
    
    my $area_diff2_tp = 0;
    my $area_diff4_tp = 0;
    my $area_diff2_d1 = 0;
    my $area_diff4_d1 = 0;
    my $area_diff2_d2 = 0;
    my $area_diff4_d2 = 0;
    
    my $max_area2_tp = 0;
    my $max_area4_tp = 0;
    my $max_area2_d1 = 0;
    my $max_area4_d1 = 0;
    my $max_area2_d2 = 0;
    my $max_area4_d2 = 0;
    
    my $min_area2_tp = 0;
    my $min_area4_tp = 0;
    my $min_area2_d1 = 0;
    my $min_area4_d1 = 0;
    my $min_area2_d2 = 0;
    my $min_area4_d2 = 0;
    
    my $maxdiff2_tp = -100;
    my $maxdiff4_tp = -100;
    my $mindiff2_tp =  100;
    my $mindiff4_tp =  100;

    my $maxdiff2_d1 = -100;
    my $maxdiff4_d1 = -100;
    my $mindiff2_d1 =  100;
    my $mindiff4_d1 =  100;

    my $maxdiff2_d2 = -100;
    my $maxdiff4_d2 = -100;
    my $mindiff2_d2 =  100;
    my $mindiff4_d2 =  100;

    my $maxdiff2_tp_at = 0;
    my $maxdiff2_d1_at = 0;
    my $maxdiff2_d2_at = 0;
    my $mindiff2_tp_at = 0;
    my $mindiff2_d1_at = 0;
    my $mindiff2_d2_at = 0;
    
    my $maxdiff4_tp_at = 0;
    my $maxdiff4_d1_at = 0;
    my $maxdiff4_d2_at = 0;
    my $mindiff4_tp_at = 0;
    my $mindiff4_d1_at = 0;
    my $mindiff4_d2_at = 0;

    my $avediff2_tp = 0;
    my $avediff2_d1 = 0;
    my $avediff2_d2 = 0;

    my $stddiff2_tp = 0;
    my $stddiff2_d1 = 0;
    my $stddiff2_d2 = 0;

    my $avediff4_tp = 0;
    my $avediff4_d1 = 0;
    my $avediff4_d2 = 0;

    my $stddiff4_tp = 0;
    my $stddiff4_d1 = 0;
    my $stddiff4_d2 = 0;

    my $diff2_tp_target = 0;
    my $diff4_tp_target = 0;

    my $diff2_d1_target = 0;
    my $diff4_d1_target = 0;

    my $diff2_d2_target = 0;
    my $diff4_d2_target = 0;

    my $tpa;
    my $tpb;
    my $tpc;
   
    my $d1a;
    my $d1b;
    my $d1c;

    my $d2a;
    my $d2b;
    my $d2c;

    my $area_tp_a = 0;
    my $area_tp_b = 0;
    my $area_tp_c = 0;

    my $area_d1_a = 0;
    my $area_d1_b = 0;
    my $area_d1_c = 0;

    my $area_d2_a = 0;
    my $area_d2_b = 0;
    my $area_d2_c = 0;

    my $i;
    my $len;
    my $dim = $N*$k;

    my $len_old;
    my $c = 0;

    open(HIS, "$hh_file");
    while(<HIS>) {
	if (/^\#/) { 
	    next;
	}
	elsif (/^(\S+)\s+\S+\s+\S+\s+(\S+)\s+(\S+)\s+\S+\s+(\S+)\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+(\S+)\s+(\S+)\s+\S+\s+(\S+)\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+(\S+)\s+(\S+)\s+\S+\s+(\S+)\s+\S+/) {
	    $len = $1;
	    $i = $len*$k;

	    if ($len < 50 || $len > 1000) { next; }
	    
	    $c ++; 

	    if ($c == 1) { $len_old = $len; } 
	    my $len_diff = $len - $len_old;

	    $tpa = $2;
	    $d1a = $3;
	    $d2a = $4;

	    $tpb = $5;
	    $d1b = $6;
	    $d2b = $7;

	    $tpc = $8;
	    $d1c = $9;
	    $d2c = $10;

	    #print "TP: $tpa $tpb $tpc\n";
	    #print "D1: $d1a $d1b $d1c\n";
	    #print "D2: $d2a $d2b $d2c\n";

	    $binused_ref->[$dim*$n+$i] = 1;

	    $tpa_ref->[$dim*$n+$i] = $tpa;
	    $tpb_ref->[$dim*$n+$i] = $tpb;
	    $tpc_ref->[$dim*$n+$i] = $tpc;

	    #print "TP: $tpa $tpa_ref->[$dim*$n+$i] dgf $i dim $dim n $n idx ", $dim*$n+$i,"\n";

	    $d1a_ref->[$dim*$n+$i] = $d1a;
	    $d1b_ref->[$dim*$n+$i] = $d1b;
	    $d1c_ref->[$dim*$n+$i] = $d1c;

	    $d2a_ref->[$dim*$n+$i] = $d2a;
	    $d2b_ref->[$dim*$n+$i] = $d2b;
	    $d2c_ref->[$dim*$n+$i] = $d2c;

	    my $tp2;
	    my $tp4;
	    
	    my $diff2_tp = $tpb - $tpa;
	    my $diff4_tp = $tpc - $tpa;

	    $area_tp_a += $tpa * $len_diff;
	    $area_tp_b += $tpb * $len_diff;
	    $area_tp_c += $tpc * $len_diff;

	    accumulate_averages($diff2_tp, \$avediff2_tp, \$stddiff2_tp);
	    accumulate_averages($diff4_tp, \$avediff4_tp, \$stddiff4_tp);

	    if ($diff2_tp > $maxdiff2_tp) { $maxdiff2_tp = $diff2_tp; $maxdiff2_tp_at = $len; }
	    if ($diff4_tp > $maxdiff4_tp) { $maxdiff4_tp = $diff4_tp; $maxdiff4_tp_at = $len; }
	    
	    if ($diff2_tp < $mindiff2_tp) { $mindiff2_tp = $diff2_tp; $mindiff2_tp_at = $len; }
	    if ($diff4_tp < $mindiff4_tp) { $mindiff4_tp = $diff4_tp; $mindiff4_tp_at = $len; }
	    
	    $tp2 = 0;
	    if (abs($tpa) > 0) { $tp2 += 1.0/($tpa*$tpa); }
	    if (abs($tpb) > 0) { $tp2 += 1.0/($tpb*$tpb); }
	    $tp2 *= 0.5 * $diff2_tp * $diff2_tp;
	    $chi2_tp  += $tp2;
	    $area_diff2_tp += $diff2_tp * $len_diff;
	    
	    $tp4 = 0;
	    if (abs($tpa) > 0) { $tp4 += 1.0/($tpa*$tpa); }
	    if (abs($tpc) > 0) { $tp4 += 1.0/($tpc*$tpc); }
	    $tp4 *= 0.5 * $diff4_tp * $diff4_tp;
	    $chi4_tp  += $tp4;
	    $area_diff4_tp += $diff4_tp * $len_diff;
	    
	    if ($len == $target_tp) {
		$diff2_tp_target = $diff2_tp;
		$diff4_tp_target = $diff4_tp;
	    }

	    my $d12;
	    my $d14;
	    
	    my $diff2_d1 = $d1b - $d1a;
	    my $diff4_d1 = $d1c - $d1a;

	    $area_d1_a += $d1a * $len_diff;
	    $area_d1_b += $d1b * $len_diff;
	    $area_d1_c += $d1c * $len_diff;

	    accumulate_averages($diff2_d1, \$avediff2_d1, \$stddiff2_d1);
	    accumulate_averages($diff4_d1, \$avediff4_d1, \$stddiff4_d1);
	    
	    if ($diff2_d1 < $maxdiff2_d1) { $maxdiff2_d1 = $diff2_d1; $maxdiff2_d1_at = $len; }
	    if ($diff4_d1 < $maxdiff4_d1) { $maxdiff4_d1 = $diff4_d1; $maxdiff4_d1_at = $len; }
	    
	    if ($diff2_d1 > $mindiff2_d1) { $mindiff2_d1 = $diff2_d1; $mindiff2_d1_at = $len; }
	    if ($diff4_d1 > $mindiff4_d1) { $mindiff4_d1 = $diff4_d1; $mindiff4_d1_at = $len; }
	    
	    $d12 = 0;
	    if (abs($d1a) > 0) { $d12 += 1.0/($d1a*$d1a); }
	    if (abs($d1b) > 0) { $d12 += 1.0/($d1b*$d1b); }
	    $d12 *= 0.5 * ($d1a - $d1b) * ($d1a - $d1b);
	    $chi2_d1  += $d12;
	    $area_diff2_d1 += $diff2_d1 * $len_diff;
	    
	    $d14 = 0;
	    if (abs($d1a) > 0) { $d14 += 1.0/($d1a*$d1a); }
	    if (abs($d1c) > 0) { $d14 += 1.0/($d1c*$d1c); }
	    $d14 *= 0.5 * ($d1a - $d1c) * ($d1a - $d1c);
	    $chi4_d1  += $d14;
	    $area_diff4_d1 += $diff4_d1 * $len_diff;
	    
	    if ($len == $target_d1) {
		$diff2_d1_target = $diff2_d1;
		$diff4_d1_target = $diff4_d1;
	    }

	    my $d22;
	    my $d24;
	    
	    my $diff2_d2 = $d2b - $d2a;
	    my $diff4_d2 = $d2c - $d2a;

	    $area_d2_a += $d2a * $len_diff;
	    $area_d2_b += $d2b * $len_diff;
	    $area_d2_c += $d2c * $len_diff;

	    accumulate_averages($diff2_d2, \$avediff2_d2, \$stddiff2_d2);
	    accumulate_averages($diff4_d2, \$avediff4_d2, \$stddiff4_d2);

	    if ($diff2_d2 < $maxdiff2_d2) { $maxdiff2_d2 = $diff2_d2; $maxdiff2_d2_at = $len; }
	    if ($diff4_d2 < $maxdiff4_d2) { $maxdiff4_d2 = $diff4_d2; $maxdiff4_d2_at = $len; }
	    
	    if ($diff2_d2 > $mindiff2_d2) { $mindiff2_d2 = $diff2_d2; $mindiff2_d2_at = $len; }
	    if ($diff4_d2 > $mindiff4_d2) { $mindiff4_d2 = $diff4_d2; $mindiff4_d2_at = $len; }
	    
	    
	    $d22 = 0;
	    if (abs($d2a) > 0) { $d22 += 1.0/($d2a*$d2a); }
	    if (abs($d2b) > 0) { $d22 += 1.0/($d2b*$d2b); }
	    $d22 *= 0.5 * ($d2a - $d2b) * ($d2a - $d2b);
	    $chi2_d2  += $d22;
	    $area_diff2_d2 += $diff2_d2 * $len_diff;
	    
	    $d24 = 0;
	    if (abs($d2a) > 0) { $d24 += 1.0/($d2a*$d2a); }
	    if (abs($d2c) > 0) { $d24 += 1.0/($d2c*$d2c); }
	    $d24 *= 0.5 * ($d2a - $d2c) * ($d2a - $d2c);
	    $chi4_d2  += $d24;
	    $area_diff4_d2 += $diff4_d2 * $len_diff;
	    
	    if ($len == $target_d2) {
		$diff2_d2_target = $diff2_d2;
		$diff4_d2_target = $diff4_d2;
	    }

	    $dgf ++;
	    $len_old = $len;
	}
    }
    close(HIS);

    calculate_averages(\$avediff2_tp, \$stddiff2_tp, $dgf);
    calculate_averages(\$avediff4_tp, \$stddiff4_tp, $dgf);

    calculate_averages(\$avediff2_d1, \$stddiff2_d1, $dgf);
    calculate_averages(\$avediff4_d1, \$stddiff4_d1, $dgf);

    calculate_averages(\$avediff2_d2, \$stddiff2_d2, $dgf);
    calculate_averages(\$avediff4_d2, \$stddiff4_d2, $dgf);

    if ($area_tp_a > $area_tp_b) { $max_area2_tp = $area_tp_a; $min_area2_tp = $area_tp_b; }
    else                         { $max_area2_tp = $area_tp_b; $min_area2_tp = $area_tp_a; }

    if ($area_tp_a > $area_tp_c) { $max_area4_tp = $area_tp_a; $min_area4_tp = $area_tp_c; }
    else                         { $max_area4_tp = $area_tp_c; $min_area4_tp = $area_tp_a; }

    if ($area_d1_a > $area_d1_b) { $max_area2_d1 = $area_d1_a; $min_area2_d1 = $area_d1_b; }
    else                         { $max_area2_d1 = $area_d1_b; $min_area2_d1 = $area_d1_a; }

    if ($area_d1_a > $area_d1_c) { $max_area4_d1 = $area_d1_a; $min_area4_d1 = $area_d1_c; }
    else                         { $max_area4_d1 = $area_d1_c; $min_area4_d1 = $area_d1_a; }

    if ($area_d2_a > $area_d2_b) { $max_area2_d2 = $area_d2_a; $min_area2_d2 = $area_d2_b; }
    else                         { $max_area2_d2 = $area_d2_b; $min_area2_d2 = $area_d2_a; }

    if ($area_d2_a > $area_d2_c) { $max_area4_d2 = $area_d2_a; $min_area4_d2 = $area_d2_c; }
    else                         { $max_area4_d2 = $area_d2_c; $min_area4_d2 = $area_d2_a; }

    $$chi2_tp_ref = $chi2_tp;
    $$chi4_tp_ref = $chi4_tp;
    $$chi2_d1_ref = $chi2_d1;
    $$chi4_d1_ref = $chi4_d1;
    $$chi2_d2_ref = $chi2_d2;
    $$chi4_d2_ref = $chi4_d2;

    if ($max_area2_tp > 0) { $$area_diff2_tp_ref = $area_diff2_tp / $max_area2_tp; }
    if ($max_area4_tp > 0) { $$area_diff4_tp_ref = $area_diff4_tp / $max_area4_tp; }
    if ($max_area2_d1 > 0) { $$area_diff2_d1_ref = $area_diff2_d1 / $max_area2_d1; }
    if ($max_area4_d1 > 0) { $$area_diff4_d1_ref = $area_diff4_d1 / $max_area4_d1; }
    if ($max_area2_d2 > 0) { $$area_diff2_d2_ref = $area_diff2_d2 / $max_area2_d2; }
    if ($max_area4_d2 > 0) {  $$area_diff4_d2_ref = $area_diff4_d2 / $max_area4_d2; }

    $$maxdiff2_tp_ref = $maxdiff2_tp;
    $$maxdiff4_tp_ref = $maxdiff4_tp;
    $$mindiff2_tp_ref = $mindiff2_tp;
    $$mindiff4_tp_ref = $mindiff4_tp;

    $$maxdiff2_d1_ref = $maxdiff2_d1;
    $$maxdiff4_d1_ref = $maxdiff4_d1;
    $$mindiff2_d1_ref = $mindiff2_d1;
    $$mindiff4_d1_ref = $mindiff4_d1;

    $$maxdiff2_d2_ref = $maxdiff2_d2;
    $$maxdiff4_d2_ref = $maxdiff4_d2;
    $$mindiff2_d2_ref = $mindiff2_d2;
    $$mindiff4_d2_ref = $mindiff4_d2;

    $$maxdiff2_tp_at_ref = $maxdiff2_tp_at;
    $$maxdiff2_d1_at_ref = $maxdiff2_d1_at;
    $$maxdiff2_d2_at_ref = $maxdiff2_d2_at;
    $$mindiff2_tp_at_ref = $mindiff2_tp_at;
    $$mindiff2_d1_at_ref = $mindiff2_d1_at;
    $$mindiff2_d2_at_ref = $mindiff2_d2_at;
    
    $$maxdiff4_tp_at_ref = $maxdiff4_tp_at;
    $$maxdiff4_d1_at_ref = $maxdiff4_d1_at;
    $$maxdiff4_d2_at_ref = $maxdiff4_d2_at;
    $$mindiff4_tp_at_ref = $mindiff4_tp_at;
    $$mindiff4_d1_at_ref = $mindiff4_d1_at;
    $$mindiff4_d2_at_ref = $mindiff4_d2_at;
    
    $$avediff2_tp_ref = $avediff2_tp;
    $$avediff2_d1_ref = $avediff2_d1;
    $$avediff2_d2_ref = $avediff2_d2;
    
    $$stddiff2_tp_ref = $stddiff2_tp;
    $$stddiff2_d1_ref = $stddiff2_d1;
    $$stddiff2_d2_ref = $stddiff2_d2;
    
    $$avediff4_tp_ref = $avediff4_tp;
    $$avediff4_d1_ref = $avediff4_d1;
    $$avediff4_d2_ref = $avediff4_d2;
    
    $$stddiff4_tp_ref = $stddiff4_tp;
    $$stddiff4_d1_ref = $stddiff4_d1;
    $$stddiff4_d2_ref = $stddiff4_d2;
    
    $$diff2_tp_target_ref = $diff2_tp_target;
    $$diff4_tp_target_ref = $diff4_tp_target;
    $$diff2_d1_target_ref = $diff2_d1_target;
    $$diff4_d1_target_ref = $diff4_d1_target;
    $$diff2_d2_target_ref = $diff2_d2_target;
    $$diff4_d2_target_ref = $diff4_d2_target;

    $$dgf_ref = $dgf;
}

sub parse_hh_file_time {
    my ($hh_file, 
	$max_time_ave_a_ref, $max_time_std_a_ref, 
	$max_time_ave_b_ref, $max_time_std_b_ref, 
	$max_time_ave_c_ref, $max_time_std_c_ref) = @_;

    my $max_time_ave_a = 0.0;
    my $max_time_std_a = 0.0;
    my $max_time_ave_b = 0.0;
    my $max_time_std_b = 0.0;
    my $max_time_ave_c = 0.0;
    my $max_time_std_c = 0.0;

    my $len;

    my $time_ave_a;
    my $time_ave_b;
    my $time_ave_c;

    my $time_std_a;
    my $time_std_b;
    my $time_std_c;

    open(HIS, "$hh_file");
    while(<HIS>) {
	if (/^\#/) { 
	    next;
	}
	elsif (/^(\S+)\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+(\S+)\s+(\S+)\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+(\S+)\s+(\S+)\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+(\S+)\s+(\S+)/) {
	    $len = $1;

	    $time_ave_a = $2;
	    $time_std_a = $3;

	    $time_ave_b = $4;
	    $time_std_b = $5;

	    $time_ave_c = $6;
	    $time_std_c = $7;

	    if ($len == 1000) {
		if ($time_ave_a > $max_time_ave_a) { $max_time_ave_a = $time_ave_a; $max_time_std_a = $time_std_a; } 
		if ($time_ave_b > $max_time_ave_b) { $max_time_ave_b = $time_ave_b; $max_time_std_b = $time_std_b; } 
		if ($time_ave_c > $max_time_ave_c) { $max_time_ave_c = $time_ave_c; $max_time_std_c = $time_std_c; } 
	    }
	    
	}
    }
    close(HIS);

    $$max_time_ave_a_ref = $max_time_ave_a;
    $$max_time_std_a_ref = $max_time_std_a;
    $$max_time_ave_b_ref = $max_time_ave_b;
    $$max_time_std_b_ref = $max_time_std_b;
    $$max_time_ave_c_ref = $max_time_ave_c;
    $$max_time_std_c_ref = $max_time_std_c;
}

sub parse_hp_file{
    my($hp_file, $dgf_ref, $chi2_like_ref, $chi4_like_ref, $chi2_abl_ref, $chi4_abl_ref) = @_;

    my $dgf = 0;
    my $chi2_like = 0;
    my $chi4_like = 0;
    my $chi2_abl = 0;
    my $chi4_abl = 0;
    
    my $likea;
    my $likeb;
    my $likec;    

    my $abla;
    my $ablb;
    my $ablc;

    open(HIS, "$hp_file");
    while(<HIS>) {
	if (/^\#/) { 
	    next;
	}
	elsif (/^\S+\s+\S+\s+(\S+)\s+\S+\s+(\S+)\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+(\S+)\s+\S+\s+(\S+)\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+(\S+)\s+\S+\s+(\S+)\s+/) {

	    $likea = $1;
	    $abla  = $2;

	    $likeb = $3;
	    $ablb  = $4;

	    $likec = $5;
	    $ablc  = $6;

	    $dgf ++;

	    #print "LIKE: $likea $likeb $likec\n";
	    #print "ABL:  $abla $ablb $ablc\n";
	    
	    my $like2;
	    my $like4;
	    
	    $like2 = 0;
	    if (abs($likea) > 0) { $like2 += 1.0/($likea*$likea); }
	    if (abs($likeb) > 0) { $like2 += 1.0/($likeb*$likeb); }
	    $like2 *= 0.5 * ($likea-$likeb) * ($likea-$likeb);
	    $chi2_like += $like2;
	    
	    $like4 = 0;
	    if (abs($likea) > 0) { $like4 += 1.0/($likea*$likea); }
	    if (abs($likec) > 0) { $like4 += 1.0/($likec*$likec); }
	    $like4 *= 0.5 * ($likea-$likec) * ($likea-$likec);
	    $chi4_like += $like4;
	    
	    my $abl2;
	    my $abl4;
	    
	    $abl2 =  0;
	    if (abs($abla) > 0) { $abl2 += 1.0/($abla*$abla); }
	    if (abs($ablb) > 0) { $abl2 += 1.0/($ablb*$ablb); }
	    $abl2 *=  0.5 * ($abla-$ablb) * ($abla-$ablb);
	    $chi2_abl += $abl2;
	    
	    $abl4 = 0;
	    if (abs($abla) > 0) { $abl4 += 1.0/($abla*$abla); }
	    if (abs($ablc) > 0) { $abl4 += 1.0/($ablc*$ablc); }
	    $abl4 *= 0.5 * ($abla-$ablc) * ($abla-$ablc);
	    $chi4_abl += $abl4;    
	}
    }
    close(HIS);

    $$chi2_like_ref = $chi2_like;
    $$chi4_like_ref = $chi4_like;
    $$chi2_abl_ref  = $chi2_abl;
    $$chi4_abl_ref  = $chi4_abl;

    $$dgf_ref = $dgf;
}

sub parse_hh_file_tp{
    my ($hh_file,  
	$ave_tp_a_ref, $std_tp_a_ref, 
	$ave_tp_b_ref, $std_tp_b_ref, 
	$ave_tp_c_ref, $std_tp_c_ref) = @_;

    my $ave_tp_a = 0.0;
    my $std_tp_a = 0.0;
    my $ave_tp_b = 0.0;
    my $std_tp_b = 0.0;
    my $ave_tp_c = 0.0;
    my $std_tp_c = 0.0;

    my $tpa;
    my $tpb;
    my $tpc;

    my $nt = 0;
    open(HIS, "$hh_file");
    while(<HIS>) {
	if (/^\#/) { 
	    next;
	}
	elsif (/^\S+\s+\S+\s+\S+\s+(\S+)\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+(\S+)\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+(\S+)\s+\S+\s+\S+\s+\S+\s+\S+/) {

	    $nt ++;

	    $tpa  = $1;
	    $tpb  = $2;
	    $tpc  = $3;

	    accumulate_averages($tpa, \$ave_tp_a, \$std_tp_a);
	    accumulate_averages($tpb, \$ave_tp_b, \$std_tp_b);
	    accumulate_averages($tpc, \$ave_tp_c, \$std_tp_c);	    
	}
    }
    close(HIS);

    calculate_averages(\$ave_tp_a, \$std_tp_a, $nt);
    calculate_averages(\$ave_tp_b, \$std_tp_b, $nt);
    calculate_averages(\$ave_tp_c, \$std_tp_c, $nt);

    $$ave_tp_a_ref = $ave_tp_a;
    $$std_tp_a_ref = $std_tp_a;
    $$ave_tp_b_ref = $ave_tp_b;
    $$std_tp_b_ref = $std_tp_b;
    $$ave_tp_c_ref = $ave_tp_c;
    $$std_tp_c_ref = $std_tp_c;


}
sub parse_hp_file_abl {
    my ($hp_file, 
	$ave_abl_a_ref, $std_abl_a_ref, 
	$ave_abl_b_ref, $std_abl_b_ref, 
	$ave_abl_c_ref, $std_abl_c_ref) = @_;

    my $ave_abl_a = 0.0;
    my $std_abl_a = 0.0;
    my $ave_abl_b = 0.0;
    my $std_abl_b = 0.0;
    my $ave_abl_c = 0.0;
    my $std_abl_c = 0.0;

    my $abla;
    my $ablb;
    my $ablc;

    my $nt = 0;
    open(HIS, "$hp_file");
    while(<HIS>) {
	if (/^\#/) { 
	    next;
	}
	elsif (/^\S+\s+\S+\s+\S+\s+\S+\s+(\S+)\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+(\S+)\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+\S+\s+(\S+)\s+/) {

	    $nt ++;

	    $abla  = $1;
	    $ablb  = $2;
	    $ablc  = $3;

	    accumulate_averages($abla, \$ave_abl_a, \$std_abl_a);
	    accumulate_averages($ablb, \$ave_abl_b, \$std_abl_b);
	    accumulate_averages($ablc, \$ave_abl_c, \$std_abl_c);	    
	}
    }
    close(HIS);

    calculate_averages(\$ave_abl_a, \$std_abl_a, $nt);
    calculate_averages(\$ave_abl_b, \$std_abl_b, $nt);
    calculate_averages(\$ave_abl_c, \$std_abl_c, $nt);

    $$ave_abl_a_ref = $ave_abl_a;
    $$std_abl_a_ref = $std_abl_a;
    $$ave_abl_b_ref = $ave_abl_b;
    $$std_abl_b_ref = $std_abl_b;
    $$ave_abl_c_ref = $ave_abl_c;
    $$std_abl_c_ref = $std_abl_c;

}

sub parse_histo_header{
    my ($filehisto, $title_ref, $key_ref) = @_;

    my $omag = 1000;

    my $title;
    my $key;

    my $ntaxa                 = -1;
    my $target_abl            = -1;
    my $ntree                 = -1;
    my $ntreesofar            = -1;
    my $nsamples_per_len_tree = -1;
    my $nsamples_per_tree     = -1;
    my $meanindelfrq          = -1;
    my $stdindelfrq           = -1;
    my $meanpairid            = -1;
    my $stdpairid             = -1;
    my $meanpairmut           = -1;
    my $stdpairmut            = -1;
    my $meanpairindl          = -1;
    my $stdpairindl           = -1;

    open(HIS, "$filehisto");
    while(<HIS>) {
	if    (/\#\s+Taxa:\s+(\S+)/)                         { $ntaxa = $1; }
	elsif (/\#\s+Trees:\s+(\S+)/)                        { $ntree = $1; }
	elsif (/Average branch length:\s+(\S+)/)             { $target_abl = $1; }
	elsif (/\# Alignments per length and tree:\s+(\S+)/) { $nsamples_per_len_tree = $1; }
	elsif (/Total \# tests per tree:\s+(\S+)/)           { $nsamples_per_tree = $1; }
	elsif (/\#\s+trees\s+so\s+far:\s+(\S+)/)             { $ntreesofar = $1 - 1; }
	elsif (/pairwise id:\s+(\S+)\s+\+\/\-\s+(\S+)/)      { $meanpairid = $1; $stdpairid = $2;}
	elsif (/pairwise mut:\s+(\S+)\s+\+\/\-\s+(\S+)/)     { $meanpairmut = $1; $stdpairmut = $2;}
	elsif (/pairwise indl:\s+(\S+)\s+\+\/\-\s+(\S+)/)    { $meanpairindl = $1; $stdpairindl = $2;}
	elsif (/indel freq:\s+(\S+)\s+\+\/\-\s+(\S+)/)       { $meanindelfrq = $1; $stdindelfrq = $2;}
    }
    close(HIS);

    $meanpairid   = int($meanpairid*$omag)/$omag;
    $stdpairid    = int($stdpairid*$omag)/$omag;
    $meanpairmut  = int($meanpairmut*$omag)/$omag;
    $stdpairmut   = int($stdpairmut*$omag)/$omag;
    $meanpairindl = int($meanpairindl*$omag)/$omag;
    $stdpairindl  = int($stdpairindl*$omag)/$omag;
    $meanindelfrq = int($meanindelfrq*$omag)*100/$omag;
    $stdindelfrq  = int($stdindelfrq*$omag)*100/$omag;
    $target_abl   = int($target_abl*$omag)/$omag;

    $title  = "Ntaxa: $ntaxa ** <branch len> $target_abl ** Ntrees $ntreesofar ** Nsamples/len_and_tree $nsamples_per_len_tree ** ";
    $title .= "\% indels: $meanindelfrq +/- $stdindelfrq";
    $key  = "pairwise \% ID: $meanpairid +/- $stdpairid ** ";
    $key .= "pairwise \% MUT: $meanpairmut +/- $stdpairmut ** ";
    $key .= "pairwise \% INDL: $meanpairindl +/- $stdpairindl";

    $$title_ref = $title;
    $$key_ref = $key;
}

sub plot_individualy {
    my ($ndir, 
	$hh_fo_ref, $hh_msalen_ref, $hh_sqleng_ref, $hh_sqlena_ref,
	$hp_fo_ref, $hp_msalen_ref, $hp_sqleng_ref, $hp_sqlena_ref) = @_;

    $seeplots = 0;
    my $xlabel_fo = "FRACTION TOTAL INDELS";
    for (my $n = 0; $n < $ndir; $n ++) {
	plot_hh($hh_fo_ref->[$n], $xlabel_fo, $seeplots);
	plot_hp($hp_fo_ref->[$n], $xlabel_fo, $seeplots);
    }
    
    $seeplots = 0;
    my $xlabel_sqleng = "GEOMETRIC MEAN SEQUENCES LENGTH";
    for (my $n = 0; $n < $ndir; $n ++) {
	plot_hh($hh_sqleng_ref->[$n], $xlabel_sqleng, $seeplots);
	plot_hp($hp_sqleng_ref->[$n], $xlabel_sqleng, $seeplots);
    }
    
    $seeplots = 0;
    my $xlabel_msalen = "ALIGNMENT LENGTH";
    for (my $n = 0; $n < $ndir; $n ++) {
	plot_hh($hh_msalen_ref->[$n], $xlabel_msalen, $seeplots);
	plot_hp($hp_msalen_ref->[$n], $xlabel_msalen, $seeplots);
    }
    
    my $xlabel_sqlena = "ARITHMETIC MEAN SEQUENCES LENGTH";
    for (my $n = 0; $n < $ndir; $n ++) {
	plot_hh($hh_sqlena_ref->[$n], $xlabel_sqlena, $seeplots);
	plot_hp($hp_sqlena_ref->[$n], $xlabel_sqlena, $seeplots);
    }

}

sub plot_jointly {
    my ($ndir, 
	$hh_fo_ref, $hh_msalen_ref, $hh_sqleng_ref, $hh_sqlena_ref,
	$hp_fo_ref, $hp_msalen_ref, $hp_sqleng_ref, $hp_sqlena_ref) = @_;

    my @m_fo_tp;
    my @a_fo_tp;
    my @merr_fo_tp;
    my @aerr_fo_tp;

    my @m_sqleng_tp;
    my @a_sqleng_tp;
    my @merr_sqleng_tp;
    my @aerr_sqleng_tp;
    
    my @m_msalen_tp;
    my @a_msalen_tp;
    my @merr_msalen_tp;
    my @aerr_msalen_tp;
    
    my @m_sqlena_tp;
    my @a_sqlena_tp;
    my @merr_sqlena_tp;
    my @aerr_sqlena_tp;
    
    my @m_fo_d1;
    my @a_fo_d1;
    my @merr_fo_d1;
    my @aerr_fo_d1;
    
    my @m_sqleng_d1;
    my @a_sqleng_d1;
    my @merr_sqleng_d1;
    my @aerr_sqleng_d1;
    
    my @m_msalen_d1;
    my @a_msalen_d1;
    my @merr_msalen_d1;
    my @aerr_msalen_d1;
    
    my @m_sqlena_d1;
    my @a_sqlena_d1;
    my @merr_sqlena_d1;
    my @aerr_sqlena_d1;
    
    my @m_fo_d2;
    my @a_fo_d2;
    my @merr_fo_d2;
    my @aerr_fo_d2;
    
    my @m_sqleng_d2;
    my @a_sqleng_d2;
    my @merr_sqleng_d2;
    my @aerr_sqleng_d2;
    
    my @m_msalen_d2;
    my @a_msalen_d2;
    my @merr_msalen_d2;
    my @aerr_msalen_d2;
    
    my @m_sqlena_d2;
    my @a_sqlena_d2;
    my @merr_sqlena_d2;
    my @aerr_sqlena_d2;
    
    if (0) {
	$seeplots = 0;
	my $xlabel_fo = "FRACTION TOTAL INDELS";
	if ($opt_f) {
	    fit_all_curves($ndir, $hh_fo_ref, $xlabel_fo, 
			   \@m_fo_tp, \@merr_fo_tp, \@a_fo_tp, \@aerr_fo_tp,
			   \@m_fo_d1, \@merr_fo_d1, \@a_fo_d1, \@aerr_fo_d1, 
			   \@m_fo_d2, \@merr_fo_d2, \@a_fo_d2, \@aerr_fo_d2);
	}
	
	plot_jointly_hh($ndir, $hh_fo_ref, $xlabel_fo, $seeplots, 
			\@m_fo_tp, \@merr_fo_tp, \@a_fo_tp, \@aerr_fo_tp,
			\@m_fo_d1, \@merr_fo_d1, \@a_fo_d1, \@aerr_fo_d1, 
			\@m_fo_d2, \@merr_fo_d2, \@a_fo_d2, \@aerr_fo_d2);
	plot_jointly_hp($ndir, $hp_fo_ref, $xlabel_fo, $seeplots);
    }
    
    my $xlabel_sqleng = "GEOMETRIC MEAN SEQUENCES LENGTH";
    if ($opt_f) {
	fit_all_curves($ndir, $hh_sqleng_ref, $xlabel_sqleng, 
		       \@m_sqleng_tp, \@merr_sqleng_tp, \@a_sqleng_tp, \@aerr_sqleng_tp,
		       \@m_sqleng_d1, \@merr_sqleng_d1, \@a_sqleng_d1, \@aerr_sqleng_d1, 
		       \@m_sqleng_d2, \@merr_sqleng_d2, \@a_sqleng_d2, \@aerr_sqleng_d2);
    }

    plot_jointly_hh($ndir, $hh_sqleng_ref, $xlabel_sqleng, $seeplots, 
		    \@m_sqleng_tp, \@merr_sqleng_tp, \@a_sqleng_tp, \@aerr_sqleng_tp,
		    \@m_sqleng_d1, \@merr_sqleng_d1, \@a_sqleng_d1, \@aerr_sqleng_d1, 
		    \@m_sqleng_d2, \@merr_sqleng_d2, \@a_sqleng_d2, \@aerr_sqleng_d2);
    if (!$opt_w && !$opt_z) { plot_jointly_hp($ndir, $hp_sqleng_ref, $xlabel_sqleng, $seeplots); }
    
    if (0) {
	$seeplots = 0;
	my $xlabel_msalen = "ALIGNMENT LENGTH";
	if ($opt_f) {
	    fit_all_curves($ndir, $hh_msalen_ref, $xlabel_msalen, 
			   \@m_msalen_tp, \@merr_msalen_tp, \@a_msalen_tp, \@aerr_msalen_tp,
			   \@m_msalen_d1, \@merr_msalen_d1, \@a_msalen_d1, \@aerr_msalen_d1, 
			   \@m_msalen_d2, \@merr_msalen_d2, \@a_msalen_d2, \@aerr_msalen_d2);
	    
	}

	plot_jointly_hh($ndir, $hh_msalen_ref, $xlabel_msalen, $seeplots, 
			\@m_msalen_tp, \@merr_msalen_tp, \@a_msalen_tp, \@aerr_msalen_tp,
			\@m_msalen_d1, \@merr_msalen_d1, \@a_msalen_d1, \@aerr_msalen_d1, 
			\@m_msalen_d2, \@merr_msalen_d2, \@a_msalen_d2, \@aerr_msalen_d2);
	plot_jointly_hp($ndir, $hp_msalen_ref, $xlabel_msalen, $seeplots);
	
	my $xlabel_sqlena = "ARITHMETIC MEAN SEQUENCES LENGTH";
	if ($opt_f) {
	    fit_all_curves($ndir, $hh_sqlena_ref, $xlabel_sqlena, 
			   \@m_sqlena_tp, \@merr_sqlena_tp, \@a_sqlena_tp, \@aerr_sqlena_tp,
			   \@m_sqlena_d1, \@merr_sqlena_d1, \@a_sqlena_d1, \@aerr_sqlena_d1, 
			   \@m_sqlena_d2, \@merr_sqlena_d2, \@a_sqlena_d2, \@aerr_sqlena_d2);
	}
	plot_jointly_hh($ndir, $hh_sqlena_ref, $xlabel_sqlena, $seeplots, 
			\@m_sqlena_tp, \@merr_sqlena_tp, \@a_sqlena_tp, \@aerr_sqlena_tp,
			\@m_sqlena_d1, \@merr_sqlena_d1, \@a_sqlena_d1, \@aerr_sqlena_d1, 
			\@m_sqlena_d2, \@merr_sqlena_d2, \@a_sqlena_d2, \@aerr_sqlena_d2);
	plot_jointly_hp($ndir, $hp_sqlena_ref, $xlabel_sqlena, $seeplots);
    }
}

sub plot_hh {
    my ($filehisto, $xlabel, $seeplots) = @_;

    my $title;
    my $key;

    parse_histo_header($filehisto, \$title, \$key);    
    gnuplot_hhhisto($filehisto, $title, $xlabel, $key, $seeplots);
}

sub plot_jointly_hh {
    my ($ndir, $filehisto_ref, $xlabel, $seeplots, 
	$m_tp_ref, $merr_tp_ref, $a_tp_ref, $aerr_tp_ref,
	$m_d1_ref, $merr_d1_ref, $a_d1_ref, $aerr_d1_ref, 
	$m_d2_ref, $merr_d2_ref, $a_d2_ref, $aerr_d2_ref) = @_;

    my @title;
    my @key;

    for (my $n = 0; $n < $ndir; $n ++) {
	parse_histo_header($filehisto_ref->[$n], \$title[$n], \$key[$n]);
    }
    gnuplot_jointly_hhhisto($ndir, $filehisto_ref, \@title, $xlabel, \@key, $seeplots, 
			    $m_tp_ref, $merr_tp_ref, $a_tp_ref, $aerr_tp_ref,
			    $m_d1_ref, $merr_d1_ref, $a_d1_ref, $aerr_d1_ref, 
			    $m_d2_ref, $merr_d2_ref, $a_d2_ref, $aerr_d2_ref);
}

sub plot_hp {
    my ($filehisto, $xlabel, $seeplots) = @_;

    my $title;
    my $key;
    
    parse_histo_header($filehisto, \$title, \$key);
    gnuplot_hphisto($filehisto, $title, $xlabel, $key, $seeplots);
}

sub plot_jointly_hp {
    my ($ndir, $filehisto_ref, $xlabel, $seeplots) = @_;

    my @title;
    my @key;
    
    for (my $n = 0; $n < $ndir; $n ++) {
	parse_histo_header($filehisto_ref->[$n], \$title[$n], \$key[$n]);
    }
    gnuplot_jointly_hphisto($ndir, $filehisto_ref, \@title, $xlabel, \@key, $seeplots);
}

sub plot_jointly_tp {
    my ($ndir, $filehisto_ref, $title_ref, $xlabel, $key_ref, $seeplots, 
	$m_ref, $merr_ref, $a_ref, $aerr_ref) = @_;
    
    my $ylabel  = "fraction TRUE POSITIVE TREES  (TOPOLOGY)";

    my $histotype;
    if ($filehisto_ref->[0] =~ /\/([^\/]+)$/) { $histotype = $1; }

    my @histoname;
    for (my $n = 0; $n < $ndir; $n ++) { if ($filehisto_ref->[$n] =~ /\/([^\/]+)\/[^\/]+$/) { $histoname[$n] = $1; } }

    my $max_diff = 0;
    my $max_diff_at = 0;
    calculate_max_diff($ndir, $m_ref, $merr_ref, $a_ref, $aerr_ref, \$max_diff, \$max_diff_at);
    my $maxkey = "max_diff $max_diff at $max_diff_at";

    my $outplot = "$outdir/$histoname[0]";
    for (my $n = 1; $n < $ndir; $n ++) { $outplot .= "_$histoname[$n]"; }
    $outplot .= ".$histotype.tp";
    if    ($opt_x) { $outplot .= ".dnaml";        }
    elsif ($opt_y) { $outplot .= ".dnaml-erate";  }
    elsif ($opt_w) { $outplot .= ".dnaml_dnaml-erate";  }
    elsif ($opt_z) { $outplot .= ".dnaml_dnaml-erate4";  }
    else           { $outplot .= ".dnaml-erate4"; }
    $outplot .= ".ps";

    open(GP,'|'.GNUPLOT) || die "Gnuplot: $!";
    print GP "set terminal postscript color solid 14\n";
    print GP "set style line 1 lt 1 lw 4\n";
    print GP "set style line 2 lt 2 lw 4\n";
    print GP "set style line 3  lt 3 lw 4\n";
    print GP "set style line 4 lt 4 lw 4\n";
    print GP "set style line 5 lt 5 lw 4\n";
    print GP "set style line 6 lt 6 lw 4\n";
    print GP "set style line 7 lt 7 lw 4\n";
    print GP "set style line 8 lt 8 lw 4\n";
    print GP "set style line 9 lt 9 lw 4\n";
    print GP "set style line 10 lt 10 lw 4\n";
    print GP "set style line 11 lt 11 lw 4\n";
    print GP "set style line 12 lt 12 lw 4\n";
    print GP "set style line 13 lt 13 lw 4\n";
    print GP "set style line 14 lt 14 lw 4\n";
    print GP "set style line 15 lt 15 lw 5\n";

    print GP "set output '$outplot'\n";
    print GP "set xlabel '$xlabel'\n";
    print GP "set ylabel '$ylabel'\n";
    print GP "set yrange [0:1]\n";

    print GP "set nokey\n";
    my $title = "100 RANDOM 8-TAXON TREES";
    print GP "set title '$title'\n";
    #print GP "set title \"$title\\n\\n$maxkey\"\n";
    
    my $which = "22";
    if ($opt_x) { $which =  "4"; }
    if ($opt_y) { $which = "13"; }
    if ($opt_w) { $which = "31"; }
    if ($opt_z) { $which = "38"; }
    
    my @aexp;
    if ($opt_f) {
	for (my $n = 0; $n < $ndir; $n ++) { $aexp[$n] = exp($a_ref->[$n]); }
    }
    
    my $string = "";
    my $color = 1;
    for (my $n = 0; $n < $ndir; $n ++) {
	if ($opt_f) {
	    my $key = "fit\[$aexp[$n]($aerr_ref->[$n]) * x ^ $m_ref->[$n]($merr_ref->[$n])\]";
	    print GP "fe$n(x) = $aexp[$n] * x ** $m_ref->[$n]\n"; 
	    $string .= "fe$n(x) title '$key' ls $color\, ";
	}

	if ($n < $ndir-1) { 
	    $string .= "'$filehisto_ref->[$n]' using 1:$which title '$key_ref->[$n]' ls $color\, "; 
	}
	else { 
	    $string .= "'$filehisto_ref->[$n]' using 1:$which title '$key_ref->[$n]' ls $color";    
	 }   
	$color += 2; if ($color > 8) { $color = 1; }
    }
    print GP "plot $string\n";
    
    close (GP);
    
    if ($seeplots) { system ("$evince $outplot&\n"); }
}

sub plot_tp {
    my ($filehisto, $title, $xlabel, $key, $seeplots) = @_;

    my $outplot = "$filehisto.tp.ps";
    my $ylabel  = "fraction TRUE POSITIVE TREES  (TOPOLOGY)";

    my $newkeyx = "$dnamlx";
    my $newkeye = "$dnamle";
    my $newkeyo = "$dnamlo";

    my $lsx = "3";
    my $lse = "2";
    my $lso = "1";

    open(GP,'|'.GNUPLOT) || die "Gnuplot: $!";

    print GP "set terminal postscript color solid 14\n";
    print GP "set style line 1 lt 1 lw 4\n";
    print GP "set style line 2 lt 2 lw 4\n";
    print GP "set style line 3  lt 3 lw 4\n";
    print GP "set style line 4 lt 4 lw 4\n";
    print GP "set style line 5 lt 5 lw 4\n";
    print GP "set style line 6 lt 6 lw 4\n";
    print GP "set style line 7 lt 7 lw 4\n";
    print GP "set style line 8 lt 8 lw 4\n";
    print GP "set style line 9 lt 9 lw 4\n";
    print GP "set style line 10 lt 10 lw 4\n";
    print GP "set style line 11 lt 11 lw 4\n";
    print GP "set style line 12 lt 12 lw 4\n";
    print GP "set style line 13 lt 13 lw 4\n";
    print GP "set style line 14 lt 14 lw 4\n";
    print GP "set style line 15 lt 15 lw 5\n";

    print GP "set output '$outplot'\n";
    print GP "set xlabel '$xlabel'\n";
    print GP "set ylabel '$ylabel'\n";
    print GP "set title \"$title\\n\\n$key\"\n";
    print GP "set yrange [0:1]\n";

    if ($opt_b || $opt_o) {
	print GP "plot '$filehisto' using 1:4 title '$newkeyx' ls $lsx, '$filehisto' using 1:13 title '$newkeye' ls $lse \n";
	if ($opt_o) {
	    print GP "plot '$filehisto' using 1:4 title '$newkeyx' ls $lsx, '$filehisto' using 1:22 title '$newkeyo' ls $lso\n";
	    print GP "plot '$filehisto' using 1:4 title '$newkeyx' ls $lsx, '$filehisto' using 1:13 title '$newkeye' ls $lse, '$filehisto' using 1:22 title '$newkeyo' ls $lso\n";
	}
    }
    else {
	my $singlekey = "$dnamlx";
	my $ls = $lsx; if ($opt_e) { $ls = $lse; }
	print GP "plot '$filehisto' using 1:4 title '$singlekey' ls $ls\n";
    }
    
    close (GP);

    if ($seeplots) { system ("$evince $outplot&\n"); }

}

sub plot_jointly_d1 {
    my ($ndir, $filehisto_ref, $title_ref, $xlabel, $key_ref, $seeplots,
	$m_ref, $merr_ref, $a_ref, $aerr_ref) = @_;
    
    my $ylabel  = "TREE  DISTANCE   (nBSD)";

    my $histotype;
    if ($filehisto_ref->[0] =~ /\/([^\/]+)$/) { $histotype = $1; }

    my @histoname;
    for (my $n = 0; $n < $ndir; $n ++) { if ($filehisto_ref->[$n] =~ /\/([^\/]+)\/[^\/]+$/) { $histoname[$n] = $1; } }

    my @aexp;
    if ($opt_f) {
	for (my $n = 0; $n < $ndir; $n ++) { $aexp[$n] = exp($a_ref->[$n]); }
    }

    my $outplot = "$outdir/$histoname[0]";
    for (my $n = 1; $n < $ndir; $n ++) { $outplot .= "_$histoname[$n]"; }

    my $max_diff = 0;
    my $max_diff_at = 0;
    calculate_max_diff($ndir, $m_ref, $merr_ref, $a_ref, $aerr_ref, \$max_diff, \$max_diff_at);
    my $key = "max_diff $max_diff at $max_diff_at\n";

    $outplot .= ".$histotype.d1";
    if    ($opt_x) { $outplot .= ".dnaml";        }
    elsif ($opt_y) { $outplot .= ".dnaml-erate";  }
    elsif ($opt_w) { $outplot .= ".dnaml_dnaml-erate";  }
    elsif ($opt_z) { $outplot .= ".dnaml_dnaml-erate4";  }
    else           { $outplot .= ".dnaml-erate4"; }
    $outplot .= ".ps";

    open(GP,'|'.GNUPLOT) || die "Gnuplot: $!";

    print GP "set terminal postscript color solid 14\n";
    print GP "set style line 1 lt 1 lw 4\n";
    print GP "set style line 2 lt 2 lw 4\n";
    print GP "set style line 3  lt 3 lw 4\n";
    print GP "set style line 4 lt 4 lw 4\n";
    print GP "set style line 5 lt 5 lw 4\n";
    print GP "set style line 6 lt 6 lw 4\n";
    print GP "set style line 7 lt 7 lw 4\n";
    print GP "set style line 8 lt 8 lw 4\n";
    print GP "set style line 9 lt 9 lw 4\n";
    print GP "set style line 10 lt 10 lw 4\n";
    print GP "set style line 11 lt 11 lw 4\n";
    print GP "set style line 12 lt 12 lw 4\n";
    print GP "set style line 13 lt 13 lw 4\n";
    print GP "set style line 14 lt 14 lw 4\n";
    print GP "set style line 15 lt 15 lw 5\n";

    print GP "set nokey\n";
    print GP "set output '$outplot'\n";
    print GP "set xlabel '$xlabel'\n";
    print GP "set ylabel '$ylabel'\n";
    #print GP "set yrange [0:9]\n";
    print GP "set yrange [0.8:2.8]\n";

    my $title = "100 RANDOM 8-TAXON TREES";
    print GP "set title '$title'\n";
    #print GP "set title \"$title\\n\\n$key\"\n";

    my $which  = "23";
    my $whiche = "24";
    if ($opt_x) { $which =  "5"; $whiche =   "6"; }
    if ($opt_y) { $which = "14"; $whiche =  "15"; }
    if ($opt_w) { $which = "32"; $whiche =  "33"; }
    if ($opt_z) { $which = "39"; $whiche =  "40"; }

    my $string = "";
    my $color = 1;
    for (my $n = 0; $n < $ndir; $n ++) {
	if ($opt_f) {
	    my $key = "fit\[$aexp[$n]($aerr_ref->[$n]) * x ^ $m_ref->[$n]($merr_ref->[$n])\]";
	    print GP "fe$n(x) = $aexp[$n] * x ** $m_ref->[$n]\n"; 
	    $string .= "fe$n(x) title '$key' ls $color\, ";
	}

	if ($n < $ndir-1) { 
	    $string .= "'$filehisto_ref->[$n]' using 1:$which title '$key_ref->[$n]' ls $color\, "; 
	}
	else { 
	    $string .= "'$filehisto_ref->[$n]' using 1:$which title '$key_ref->[$n]' ls $color";    
	}
	$color += 2; if ($color > 8) { $color = 1; }
    }
    print GP "plot $string\n";

    # with errorbars
    if (0) {
	my $string = "";
	my $color = 1;
	for (my $n = 0; $n < $ndir; $n ++) {
	    if ($opt_f) {
		my $key = "fit\[$aexp[$n]($aerr_ref->[$n]) * x ^ $m_ref->[$n]($merr_ref->[$n])\]";
		print GP "fe(x) = $aexp[$n] * x ** $m_ref->[$n]\n"; 
		$string .= "fe(x) title '$key' ls $color\, ";
	    }
	    
	    if ($n < $ndir-1) { 
		$string .= "'$filehisto_ref->[$n]' using 1:$which:$whiche  with yerrorbars title '$key_ref->[$n]' ls $color\, "; 
	    }
	    else { 
		$string .= "'$filehisto_ref->[$n]' using 1:$which:$whiche  with yerrorbars title '$key_ref->[$n]' ls $color";    
	    }
	$color += 2; if ($color > 8) { $color = 1; }
	}
	print GP "plot $string\n";
    }

    close (GP);
    if ($seeplots) { system ("$evince $outplot&\n"); }

}

sub plot_d1 {
    my ($filehisto, $title, $xlabel, $key, $seeplots) = @_;

    my $outplot = "$filehisto.d1.ps";
    my $ylabel  = "TREE  DISTANCE   (nBSD)";

    my $newkeyx = "$dnamlx";
    my $newkeye = "$dnamle";
    my $newkeyo = "$dnamlo";

    my $lsx = "3";
    my $lse = "2";
    my $lso = "1";

    open(GP,'|'.GNUPLOT) || die "Gnuplot: $!";

    print GP "set terminal postscript color solid 14\n";
    print GP "set style line 1 lt 1 lw 4\n";
    print GP "set style line 2 lt 2 lw 4\n";
    print GP "set style line 3  lt 3 lw 4\n";
    print GP "set style line 4 lt 4 lw 4\n";
    print GP "set style line 5 lt 5 lw 4\n";
    print GP "set style line 6 lt 6 lw 4\n";
    print GP "set style line 7 lt 7 lw 4\n";
    print GP "set style line 8 lt 8 lw 4\n";
    print GP "set style line 9 lt 9 lw 4\n";
    print GP "set style line 10 lt 10 lw 4\n";
    print GP "set style line 11 lt 11 lw 4\n";
    print GP "set style line 12 lt 12 lw 4\n";
    print GP "set style line 13 lt 13 lw 4\n";
    print GP "set style line 14 lt 14 lw 4\n";
    print GP "set style line 15 lt 15 lw 5\n";

    print GP "set output '$outplot'\n";
    print GP "set xlabel '$xlabel'\n";
    print GP "set ylabel '$ylabel'\n";
    print GP "set title \"$title\\n\\n$key\"\n";

    if ($opt_b || $opt_o) {
	print GP "plot '$filehisto' using 1:5 title '$newkeyx' ls $lsx, '$filehisto' using 1:14 title '$newkeye' ls $lse\n";
	print GP "plot '$filehisto' using 1:5:6 with yerrorbars title '$newkeyx' ls $lsx, '$filehisto' using 1:14:15  with yerrorbars title '$newkeye' ls $lse\n";
   
	if ($opt_o) {
	    print GP "plot '$filehisto' using 1:5 title '$newkeyx' ls $lsx, '$filehisto' using 1:23  title '$newkeyo' ls $lso\n";
	    print GP "plot '$filehisto' using 1:5:6 with yerrorbars title '$newkeyx' ls $lsx, '$filehisto' using 1:23:24  with yerrorbars title '$newkeyo' ls $lso\n";
	    print GP "plot '$filehisto' using 1:5 title '$newkeyx' ls $lsx, '$filehisto' using 1:14 title '$newkeye' ls $lse, '$filehisto' using 1:23 title '$newkeyo' ls $lso\n";
	    print GP "plot '$filehisto' using 1:5:6 with yerrorbars title '$newkeyx' ls $lsx, '$filehisto' using 1:14:15  with yerrorbars title '$newkeye' ls $lse, '$filehisto' using 1:23:24  with yerrorbars title '$newkeyo' ls $lso\n";
	    
	}
    }
    else {
	my $singlekey = "$dnamlx";
	my $ls = $lsx; if ($opt_e) { $ls = $lse; }

	print GP "plot '$filehisto' using 1:5 title '$singlekey' ls $ls\n";
	print GP "plot '$filehisto' using 1:5:6  with yerrorbars title '$singlekey' ls $ls\n";
    }

    close (GP);
    if ($seeplots) { system ("$evince $outplot&\n"); }

}

sub plot_jointly_d2 {
    my ($ndir, $filehisto_ref, $title_ref, $xlabel, $key_ref, $seeplots,
	$m_ref, $merr_ref, $a_ref, $aerr_ref) = @_;

    my $ylabel  = "TREE  DISTANCE   (SDD)";

    my $histotype;
    if ($filehisto_ref->[0] =~ /\/([^\/]+)$/) { $histotype = $1; }

    my @histoname;
    for (my $n = 0; $n < $ndir; $n ++) { if ($filehisto_ref->[$n] =~ /\/([^\/]+)\/[^\/]+$/) { $histoname[$n] = $1; } }

    my @aexp;
    if ($opt_f) {
	for (my $n = 0; $n < $ndir; $n ++) { $aexp[$n] = exp($a_ref->[$n]); }
    }

    my $outplot = "$outdir/$histoname[0]";
    for (my $n = 1; $n < $ndir; $n ++) { $outplot .= "_$histoname[$n]"; }
    $outplot .= ".$histotype.d2";
    if    ($opt_x) { $outplot .= ".dnaml";        }
    elsif ($opt_y) { $outplot .= ".dnaml-erate";  }
    elsif ($opt_w) { $outplot .= ".dnaml_dnaml-erate";  }
    elsif ($opt_z) { $outplot .= ".dnaml_dnaml-erate4";  }
    else           { $outplot .= ".dnaml-erate4"; }
    $outplot .= ".ps";

    my $max_diff = 0;
    my $max_diff_at = 0;
    calculate_max_diff($ndir, $m_ref, $merr_ref, $a_ref, $aerr_ref, \$max_diff, \$max_diff_at);
    my $key = "max_diff $max_diff at $max_diff_at\n";

    my $string;
    my $color;

    open(GP,'|'.GNUPLOT) || die "Gnuplot: $!";

    print GP "set terminal postscript color solid 14\n";
    print GP "set style line 1 lt 1 lw 4\n";
    print GP "set style line 2 lt 2 lw 4\n";
    print GP "set style line 3  lt 3 lw 4\n";
    print GP "set style line 4 lt 4 lw 4\n";
    print GP "set style line 5 lt 5 lw 4\n";
    print GP "set style line 6 lt 6 lw 4\n";
    print GP "set style line 7 lt 7 lw 4\n";
    print GP "set style line 8 lt 8 lw 4\n";
    print GP "set style line 9 lt 9 lw 4\n";
    print GP "set style line 10 lt 10 lw 4\n";
    print GP "set style line 11 lt 11 lw 4\n";
    print GP "set style line 12 lt 12 lw 4\n";
    print GP "set style line 13 lt 13 lw 4\n";
    print GP "set style line 14 lt 14 lw 4\n";
    print GP "set style line 15 lt 15 lw 5\n";

    print GP "set nokey\n";
    print GP "set output '$outplot'\n";
    print GP "set xlabel '$xlabel'\n";
    print GP "set ylabel '$ylabel'\n";
    #print GP "set yrange [0:10]\n";
    print GP "set yrange [0:4.5]\n";

    my $title = "100 RANDOM 8-TAXON TREES";
    print GP "set title '$title'\n";
    #print GP "set title \"$title\\n\\n$key\"\n";

    my $which  = "25";
    my $whiche = "26";
    if ($opt_x) { $which =  "7"; $whiche =   "8"; }
    if ($opt_y) { $which = "16"; $whiche =  "17"; }
    if ($opt_w) { $which = "34"; $whiche =  "35"; }
    if ($opt_z) { $which = "41"; $whiche =  "42"; }

    $string = "";
    $color = 1;
    for (my $n = 0; $n < $ndir; $n ++) {
	if ($opt_f) {
	    my $key = "fit\[$aexp[$n]($aerr_ref->[$n]) * x ^ $m_ref->[$n]($merr_ref->[$n])\]";
	    print GP "fe$n(x) = $aexp[$n] * x ** $m_ref->[$n]\n"; 
	    $string .= "fe$n(x) title '$key' ls $color\, ";
	}

	if ($n < $ndir-1) { 
	    $string .= "'$filehisto_ref->[$n]' using 1:$which title '$key_ref->[$n]' ls $color\, "; 
	}
	else { 
	    $string .= "'$filehisto_ref->[$n]' using 1:$which title '$key_ref->[$n]' ls $color";    
	}
	$color += 2; if ($color > 8) { $color = 1; }
    }
    print GP "plot $string\n";

    # with errorbars
    if (0) {
	$string = "";
	$color = 1;
	for (my $n = 0; $n < $ndir; $n ++) {
	    if ($n < $ndir-1) { 
		$string .= "'$filehisto_ref->[$n]' using 1:$which:$whiche  with yerrorbars title '$key_ref->[$n]' ls $color\, "; 
	    }
	    else { 
		$string .= "'$filehisto_ref->[$n]' using 1:$which:$whiche with yerrorbars title '$key_ref->[$n]' ls $color";    
	    }
	    $color += 2; if ($color > 8) { $color = 1; }
	}
	print GP "plot $string\n";
    }

    close (GP);
    if ($seeplots) { system ("$evince $outplot&\n"); }
}

sub plot_d2 {
    my ($filehisto, $title, $xlabel, $key, $seeplots) = @_;

    my $outplot = "$filehisto.d2.ps";
    my $ylabel  = "TREE  DISTANCE   (SDD)";

    my $newkeyx = "$dnamlx";
    my $newkeye = "$dnamle";
    my $newkeyo = "$dnamlo";

    my $lsx = "3";
    my $lse = "2";
    my $lso = "1";

    open(GP,'|'.GNUPLOT) || die "Gnuplot: $!";

    print GP "set terminal postscript color solid 14\n";
    print GP "set style line 1 lt 1 lw 4\n";
    print GP "set style line 2 lt 2 lw 4\n";
    print GP "set style line 3  lt 3 lw 4\n";
    print GP "set style line 4 lt 4 lw 4\n";
    print GP "set style line 5 lt 5 lw 4\n";
    print GP "set style line 6 lt 6 lw 4\n";
    print GP "set style line 7 lt 7 lw 4\n";
    print GP "set style line 8 lt 8 lw 4\n";
    print GP "set style line 9 lt 9 lw 4\n";
    print GP "set style line 10 lt 10 lw 4\n";
    print GP "set style line 11 lt 11 lw 4\n";
    print GP "set style line 12 lt 12 lw 4\n";
    print GP "set style line 13 lt 13 lw 4\n";
    print GP "set style line 14 lt 14 lw 4\n";
    print GP "set style line 15 lt 15 lw 5\n";

    print GP "set output '$outplot'\n";
    print GP "set xlabel '$xlabel'\n";
    print GP "set ylabel '$ylabel'\n";
    print GP "set title \"$title\\n\\n$key\"\n";

   if ($opt_b || $opt_o) {
	print GP "plot '$filehisto' using 1:7 title '$newkeyx' ls $lsx, '$filehisto' using 1:16 title '$newkeye' ls $lse\n";
	print GP "plot '$filehisto' using 1:7:8 with yerrorbars title '$newkeyx' ls $lsx, '$filehisto' using 1:16:17  with yerrorbars title '$newkeye' ls $lse\n";

	if ($opt_o) {
	    print GP "plot '$filehisto' using 1:7 title '$newkeyx' ls $lsx, '$filehisto' using 1:25 title '$newkeyo' ls $lso\n";
	    print GP "plot '$filehisto' using 1:7:8 with yerrorbars title '$newkeyx' ls $lsx, '$filehisto' using 1:25:26  with yerrorbars title '$newkeyo' ls $lso\n";
	    print GP "plot '$filehisto' using 1:7 title '$newkeyx' ls $lsx, '$filehisto' using 1:16 title '$newkeye' ls $lse, '$filehisto' using 1:25 title '$newkeyo' ls $lso\n";
	    print GP "plot '$filehisto' using 1:7:8 with yerrorbars title '$newkeyx' ls $lsx, '$filehisto' using 1:16:17  with yerrorbars title '$newkeye' ls $lse, '$filehisto' using 1:25:26  with yerrorbars title '$newkeyo' ls $lso\n";
	}
    }
    else {
	my $singlekey = "$dnamlx";
	my $ls = $lsx; if ($opt_e) { $ls = $lse; }

	print GP "plot '$filehisto' using 1:7:8  with yerrorbars title '$singlekey' ls $ls\n";
    }
 
    close (GP);
    if ($seeplots) { system ("$evince $outplot&\n"); }

}

sub plot_jointly_time {
    my ($ndir, $filehisto_ref, $title_ref, $xlabel, $key_ref, $seeplots) = @_;
   
    my $ylabel  = "RUN TIME  (sec)";

    my $histotype;
    if ($filehisto_ref->[0] =~ /\/([^\/]+)$/) { $histotype = $1; }

    my @histoname;
    for (my $n = 0; $n < $ndir; $n ++) { if ($filehisto_ref->[$n] =~ /\/([^\/]+)\/[^\/]+$/) { $histoname[$n] = $1; } }

    my $outplot = "$outdir/$histoname[0]";
    for (my $n = 1; $n < $ndir; $n ++) { $outplot .= "_$histoname[$n]"; }
    $outplot .= ".$histotype.time";
    if    ($opt_x) { $outplot .= ".dnaml";        }
    elsif ($opt_y) { $outplot .= ".dnaml-erate";  }
    else           { $outplot .= ".dnaml-erate4"; }
    $outplot .= ".ps";

    open(GP,'|'.GNUPLOT) || die "Gnuplot: $!";

    print GP "set terminal postscript color solid 14\n";
    print GP "set style line 1 lt 1 lw 4\n";
    print GP "set style line 2 lt 2 lw 4\n";
    print GP "set style line 3  lt 3 lw 4\n";
    print GP "set style line 4 lt 4 lw 4\n";
    print GP "set style line 5 lt 5 lw 4\n";
    print GP "set style line 6 lt 6 lw 4\n";
    print GP "set style line 7 lt 7 lw 4\n";
    print GP "set style line 8 lt 8 lw 4\n";
    print GP "set style line 9 lt 9 lw 4\n";
    print GP "set style line 10 lt 10 lw 4\n";
    print GP "set style line 11 lt 11 lw 4\n";
    print GP "set style line 12 lt 12 lw 4\n";
    print GP "set style line 13 lt 13 lw 4\n";
    print GP "set style line 14 lt 14 lw 4\n";
    print GP "set style line 15 lt 15 lw 5\n";
 
    print GP "set nokey\n";
    print GP "set output '$outplot'\n";
    print GP "set xlabel '$xlabel'\n";
    print GP "set ylabel '$ylabel'\n";
    #print GP "set yrange [0:50]\n";
    print GP "set yrange [0:13]\n";

    my $title = "100 RANDOM 8-TAXON TREES";
    print GP "set title '$title'\n";
    #print GP "set title \"$title\\n\\n$key\"\n";

    my $which  = "27";
    my $whiche = "28";
    if ($opt_x) { $which =  "9"; $whiche =  "10"; }
    if ($opt_y) { $which = "18"; $whiche =  "19"; }
    
    if (0) {
	my $string = "";
	my $color = 1;
	for (my $n = 0; $n < $ndir; $n ++) {
	    if ($n < $ndir-1) { 
		$string .= "'$filehisto_ref->[$n]' using 1:$which title '$key_ref->[$n]' ls $color\, "; 
	    }
	    else { 
		$string .= "'$filehisto_ref->[$n]' using 1:$which title '$key_ref->[$n]' ls $color";    
	    }
	    $color += 2; if ($color > 8) { $color = 1; }
	}
	print GP "plot $string\n";
    }

    # with errorbars
    if (1) {
	my $string = "";
	my $color = 1;
	for (my $n = 0; $n < $ndir; $n ++) {
	    if ($n < $ndir-1) { 
		$string .= "'$filehisto_ref->[$n]' using 1:$which:$whiche  with yerrorbars title '$key_ref->[$n]' ls $color\, "; 
	    }
	    else { 
		$string .= "'$filehisto_ref->[$n]' using 1:$which:$whiche  with yerrorbars title '$key_ref->[$n]' ls $color";    
	    }
	    $color += 2; if ($color > 8) { $color = 1; }
	}
	print GP "plot $string\n";	
    }

    close (GP);
    if ($seeplots) { system ("$evince $outplot&\n"); }
}

sub plot_time {
    my ($filehisto, $title, $xlabel, $key, $seeplots) = @_;

    my $outplot = "$filehisto.time.ps";
    my $ylabel  = "RUN TIME (sec)";

    my $newkeyx = "$dnamlx";
    my $newkeye = "$dnamle";
    my $newkeyo = "$dnamlo";

    my $lsx = "3";
    my $lse = "2";
    my $lso = "1";

    open(GP,'|'.GNUPLOT) || die "Gnuplot: $!";

    print GP "set terminal postscript color solid 14\n";
    print GP "set style line 1 lt 1 lw 4\n";
    print GP "set style line 2 lt 2 lw 4\n";
    print GP "set style line 3  lt 3 lw 4\n";
    print GP "set style line 4 lt 4 lw 4\n";
    print GP "set style line 5 lt 5 lw 4\n";
    print GP "set style line 6 lt 6 lw 4\n";
    print GP "set style line 7 lt 7 lw 4\n";
    print GP "set style line 8 lt 8 lw 4\n";
    print GP "set style line 9 lt 9 lw 4\n";
    print GP "set style line 10 lt 10 lw 4\n";
    print GP "set style line 11 lt 11 lw 4\n";
    print GP "set style line 12 lt 12 lw 4\n";
    print GP "set style line 13 lt 13 lw 4\n";
    print GP "set style line 14 lt 14 lw 4\n";
    print GP "set style line 15 lt 15 lw 5\n";

    print GP "set output '$outplot'\n";
    print GP "set xlabel '$xlabel'\n";
    print GP "set ylabel '$ylabel'\n";
    print GP "set title \"$title\\n\\n$key\"\n";

    if ($opt_b || $opt_o) {
	#print GP "plot '$filehisto' using 1:9:10 with yerrorbars title '$newkeyx' ls $lsx, '$filehisto' using 1:18:19  with yerrorbars title '$newkeye' ls $lse\n";
    
	if ($opt_o) {
	    print GP "plot '$filehisto' using 1:9:10 with yerrorbars title '$newkeyx' ls $lsx, '$filehisto' using 1:18:19  with yerrorbars title '$newkeye' ls $lse\n";
	    print GP "plot '$filehisto' using 1:9:10 with yerrorbars title '$newkeyx' ls $lsx, '$filehisto' using 1:27:28  with yerrorbars title '$newkeyo' ls $lso\n";
	    #print GP "plot '$filehisto' using 1:9:10 with yerrorbars title '$newkeyx' ls $lsx, '$filehisto' using 1:18:19  with yerrorbars title '$newkeye' ls $lse, '$filehisto' using 1:27:28  with yerrorbars title '$newkeyo' ls $lso\n";
	}
    }
    else {
	my $singlekey = "$dnamlx";

	my $ls = $lsx;
	if ($opt_e) { $ls = $lse; }

	print GP "plot '$filehisto' using 1:9:10  with yerrorbars title '$singlekey' ls $ls\n";
    }
 
   close (GP);
    if ($seeplots) { system ("$evince $outplot&\n"); }

}

sub plot_jointly_like {
    my ($ndir, $filehisto_ref, $title_ref, $xlabel, $key_ref, $seeplots) = @_;
}

sub plot_like {

    my ($filehisto, $title, $xlabel, $key, $seeplots) = @_;

    my $outplot = "$filehisto.like.ps";
    my $ylabel  = "LN LIKELIHOOD";

    my $newkeyx = "$dnamlx";
    my $newkeye = "$dnamle";
    my $newkeyo = "$dnamlo";
    
    my $lsx = "3";
    my $lse = "2";
    my $lso = "1";

    open(GP,'|'.GNUPLOT) || die "Gnuplot: $!";

    print GP "set terminal postscript color solid 14\n";
    print GP "set style line 1 lt 1 lw 4\n";
    print GP "set style line 2 lt 2 lw 4\n";
    print GP "set style line 3  lt 3 lw 4\n";
    print GP "set style line 4 lt 4 lw 4\n";
    print GP "set style line 5 lt 5 lw 4\n";
    print GP "set style line 6 lt 6 lw 4\n";
    print GP "set style line 7 lt 7 lw 4\n";
    print GP "set style line 8 lt 8 lw 4\n";
    print GP "set style line 9 lt 9 lw 4\n";
    print GP "set style line 10 lt 10 lw 4\n";
    print GP "set style line 11 lt 11 lw 4\n";
    print GP "set style line 12 lt 12 lw 4\n";
    print GP "set style line 13 lt 13 lw 4\n";
    print GP "set style line 14 lt 14 lw 4\n";
    print GP "set style line 15 lt 15 lw 5\n";

    print GP "set output '$outplot'\n";
    print GP "set xlabel '$xlabel'\n";
    print GP "set ylabel '$ylabel'\n";
    print GP "set title \"$title\\n\\n$key\"\n";
    
     if ($opt_b || $opt_o) {
	 print GP "plot '$filehisto' using 1:3:4  with yerrorbars title '$newkeyx' ls $lsx, '$filehisto' using 1:20:21  with yerrorbars title '$newkeye' ls $lse\n";
     }
    if ($opt_o) {
	print GP "plot '$filehisto' using 1:3:4  with yerrorbars title '$newkeyx' ls $lsx, '$filehisto' using 1:20:21  with yerrorbars title '$newkeye' ls $lse, '$filehisto' using 1:37:38  with yerrorbars title '$newkeyo' ls $lso\n";
    }
    
    if ($opt_e) { 
	my $singlekey .= "$dnamlx"; 
	my $ls = $lsx; if ($opt_e) { $ls = $lse; }
	
	print GP "plot '$filehisto' using 1:3:4  with yerrorbars title '$singlekey' ls $ls\n";
    }
    
    close (GP);
    
    if ($seeplots) { system ("$evince $outplot&\n"); }
}

sub plot_jointly_abl {
    my ($ndir, $filehisto_ref, $title_ref, $xlabel, $key_ref, $seeplots) = @_;
    my $ylabel  = "AVERAGE BRANCH LENGTH";

    my $histotype;
    if ($filehisto_ref->[0] =~ /\/([^\/]+)$/) { $histotype = $1; }

    my @histoname;
    for (my $n = 0; $n < $ndir; $n ++) { if ($filehisto_ref->[$n] =~ /\/([^\/]+)\/[^\/]+$/) { $histoname[$n] = $1; } }

    my $outplot = "$outdir/$histoname[0]";
    for (my $n = 1; $n < $ndir; $n ++) { $outplot .= "_$histoname[$n]"; }
    $outplot .= ".$histotype.abl";
    if    ($opt_x) { $outplot .= ".dnaml";        }
    elsif ($opt_y) { $outplot .= ".dnaml-erate";  }
    else           { $outplot .= ".dnaml-erate4"; }
    $outplot .= ".ps";

    open(GP,'|'.GNUPLOT) || die "Gnuplot: $!";

    print GP "set terminal postscript color solid 14\n";
    print GP "set style line 1 lt 1 lw 4\n";
    print GP "set style line 2 lt 2 lw 4\n";
    print GP "set style line 3  lt 3 lw 4\n";
    print GP "set style line 4 lt 4 lw 4\n";
    print GP "set style line 5 lt 5 lw 4\n";
    print GP "set style line 6 lt 6 lw 4\n";
    print GP "set style line 7 lt 7 lw 4\n";
    print GP "set style line 8 lt 8 lw 4\n";
    print GP "set style line 9 lt 9 lw 4\n";
    print GP "set style line 10 lt 10 lw 4\n";
    print GP "set style line 11 lt 11 lw 4\n";
    print GP "set style line 12 lt 12 lw 4\n";
    print GP "set style line 13 lt 13 lw 4\n";
    print GP "set style line 14 lt 14 lw 4\n";
    print GP "set style line 15 lt 15 lw 5\n";
 
    print GP "set nokey\n";
    print GP "set output '$outplot'\n";
    print GP "set xlabel '$xlabel'\n";
    print GP "set ylabel '$ylabel'\n";
    #print GP "set yrange [0:8]\n";
    print GP "set yrange [0.04:0.1]\n";

    my $title = "100 RANDOM 8-TAXON TREES";
    print GP "set title '$title'\n";
    #print GP "set title \"$title\\n\\n$key\"\n";

    my $which  = "39";
    my $whiche = "40";
    if ($opt_x) { $which =  "5"; $whiche =  "6"; }
    if ($opt_y) { $which = "22"; $whiche =  "23"; }

    if (0) {
	my $string = "";
	my $color = 1;
	for (my $n = 0; $n < $ndir; $n ++) {
	    if ($n < $ndir-1) { 
		$string .= "'$filehisto_ref->[$n]' using 1:$which title '$key_ref->[$n]' ls $color\, "; 
	    }
	    else { 
		$string .= "'$filehisto_ref->[$n]' using 1:$which title '$key_ref->[$n]' ls $color";    
	    }
	    $color += 2; if ($color > 8) { $color = 1; }
	}
	print GP "plot $string\n";
    }
    
    # with errorbars
    if (1) {
	my $string = "";
	my $color = 1;
	for (my $n = 0; $n < $ndir; $n ++) {
	    if ($n < $ndir-1) { 
		$string .= "'$filehisto_ref->[$n]' using 1:$which:$whiche  with yerrorbars title '$key_ref->[$n]' ls $color\, "; 
	    }
	    else { 
		$string .= "'$filehisto_ref->[$n]' using 1:$which:$whiche  with yerrorbars title '$key_ref->[$n]' ls $color";    
	    }
	    $color += 2; if ($color > 8) { $color = 1; }
	}
	print GP "plot $string\n";	
    }

    close (GP);
    if ($seeplots) { system ("$evince $outplot&\n"); }
}

sub plot_abl {

    my ($filehisto, $title, $xlabel, $key, $seeplots) = @_;

    my $outplot = "$filehisto.abl.ps";
    my $ylabel  = "AVERAGE BRANCH LENGTH";

    my $newkeyx = "$dnamlx";
    my $newkeye = "$dnamle";
    my $newkeyo = "$dnamlo";
    
    my $lsx = "3";
    my $lse = "2";
    my $lso = "1";

    open(GP,'|'.GNUPLOT) || die "Gnuplot: $!";

    print GP "set terminal postscript color solid 14\n";
    print GP "set style line 1 lt 1 lw 4\n";
    print GP "set style line 2 lt 2 lw 4\n";
    print GP "set style line 3  lt 3 lw 4\n";
    print GP "set style line 4 lt 4 lw 4\n";
    print GP "set style line 5 lt 5 lw 4\n";
    print GP "set style line 6 lt 6 lw 4\n";
    print GP "set style line 7 lt 7 lw 4\n";
    print GP "set style line 8 lt 8 lw 4\n";
    print GP "set style line 9 lt 9 lw 4\n";
    print GP "set style line 10 lt 10 lw 4\n";
    print GP "set style line 11 lt 11 lw 4\n";
    print GP "set style line 12 lt 12 lw 4\n";
    print GP "set style line 13 lt 13 lw 4\n";
    print GP "set style line 14 lt 14 lw 4\n";
    print GP "set style line 15 lt 15 lw 5\n";
 
    print GP "set output '$outplot'\n";
    print GP "set xlabel '$xlabel'\n";
    print GP "set ylabel '$ylabel'\n";
    print GP "set title \"$title\\n\\n$key\"\n";
    
    if ($opt_b || $opt_o) {
	print GP "plot '$filehisto' using 1:5:6  with yerrorbars title '$newkeyx' ls $lsx, '$filehisto' using 1:22:23  with yerrorbars title '$newkeye' ls $lse\n";
	
	if ($opt_o) {
	    print GP "plot '$filehisto' using 1:5:6  with yerrorbars title '$newkeyx' ls $lsx, '$filehisto' using 1:39:40  with yerrorbars title '$newkeyo' ls $lso\n";
	    #print GP "plot '$filehisto' using 1:5:6  with yerrorbars title '$newkeyx' ls $lsx, '$filehisto' using 1:22:23  with yerrorbars title '$newkeye' ls $lse, '$filehisto' using 1:39:40  with yerrorbars title '$newkeyo' ls $lso\n";
	}
    }
    
    if ($opt_e) { 
	my $singlekey .= "$dnamlx"; 
	my $ls = $lsx; if ($opt_e) { $ls = $lse; }
	
	print GP "plot '$filehisto' using 1:5:6  with yerrorbars title '$singlekey' ls $ls\n";
    }
    
    close (GP);
    
    if ($seeplots) { system ("$evince $outplot&\n"); }
}

sub plot_jointly_param {
    my ($ndir, $filehisto_ref, $title_ref, $xlabel, $key_ref, $seeplots) = @_;
}

sub plot_param {

    my ($filehisto, $title, $xlabel, $key, $seeplots) = @_;

    my $outplot = "$filehisto.param.ps";
    my $ylabel  = "RATE PARAMETERS";

    my $newkeyx = "$dnamlx";
    my $newkeye = "$dnamle";
    my $newkeyo = "$dnamlo";
    
    my $lsx = "3";
    my $lse = "2";
    my $lso = "1";

    open(GP,'|'.GNUPLOT) || die "Gnuplot: $!";

    print GP "set terminal postscript color solid 14\n";
    print GP "set style line 1 lt 1 lw 4\n";
    print GP "set style line 2 lt 2 lw 4\n";
    print GP "set style line 3  lt 3 lw 4\n";
    print GP "set style line 4 lt 4 lw 4\n";
    print GP "set style line 5 lt 5 lw 4\n";
    print GP "set style line 6 lt 6 lw 4\n";
    print GP "set style line 7 lt 7 lw 4\n";
    print GP "set style line 8 lt 8 lw 4\n";
    print GP "set style line 9 lt 9 lw 4\n";
    print GP "set style line 10 lt 10 lw 4\n";
    print GP "set style line 11 lt 11 lw 4\n";
    print GP "set style line 12 lt 12 lw 4\n";
    print GP "set style line 13 lt 13 lw 4\n";
    print GP "set style line 14 lt 14 lw 4\n";
    print GP "set style line 15 lt 15 lw 5\n";

    print GP "set output '$outplot'\n";
    print GP "set xlabel '$xlabel'\n";
    print GP "set title \"$title\\n\\n$key\"\n";
    
     if ($opt_b || $opt_o) {
	$ylabel = "RATE PARAMETERS  ** $newkeye";
	print GP "set ylabel '$ylabel'\n";
	print GP "plot '$filehisto' using 1:24 title 'ALPHA' ls 1, '$filehisto' using 1:26 title 'BETA' ls 2, '$filehisto' using 1:28 title 'INS' ls 3, '$filehisto' using 1:30 title 'DEL' ls 4\n";
	print GP "plot '$filehisto' using 1:24:25  with yerrorbars title 'ALPHA' ls 1, '$filehisto' using 1:26:27  with yerrorbars title 'BETA' ls 2, '$filehisto' using 1:28:29  with yerrorbars title 'INS' ls 3, '$filehisto' using 1:30:31  with yerrorbars title 'DEL' ls 4\n";
    }
    if ($opt_o) {
	$ylabel = "RATE PARAMETERS ** $newkeyo";
	print GP "set ylabel '$ylabel'\n";
	print GP "plot '$filehisto' using 1:41 title 'ALPHA' ls 1, '$filehisto' using 1:43 title 'BETA' ls 2, '$filehisto' using 1:45 title 'INS' ls 3, '$filehisto' using 1:47 title 'DEL' ls 4\n";
	print GP "plot '$filehisto' using 1:41:42  with yerrorbars title 'ALPHA' ls 1, '$filehisto' using 1:43:44  with yerrorbars title 'BETA' ls 2, '$filehisto' using 1:45:46  with yerrorbars title 'INS' ls 3, '$filehisto' using 1:47:48  with yerrorbars title 'DEL' ls 4\n";
    }
    
    if ($opt_e) { 
	my $singlekey .= "$dnamlx"; 
	my $ls = $lsx; if ($opt_e) { $ls = $lse; }
	
	$ylabel = "RATE PARAMETERS ** $singlekey";
	print GP "set ylabel '$ylabel'\n";
	print GP "plot '$filehisto' using 1:7 title 'ALPHA' ls 1, '$filehisto' using 1:9 title 'BETA' ls 2, '$filehisto' using 1:11 title 'INS' ls 3, '$filehisto' using 1:13 title 'DEL' ls 4\n";
	print GP "plot '$filehisto' using 1:7:8  with yerrorbars title 'ALPHA' ls 1, '$filehisto' using 1:9:10  with yerrorbars title 'BETA' ls 2, '$filehisto' using 1:11:12  with yerrorbars title 'INS' ls 3, '$filehisto' using 1:13:14  with yerrorbars title 'DEL' ls 4\n";
    }
    
    close (GP);
    
    if ($seeplots) { system ("$evince $outplot&\n"); }
}

sub plot_jointly_ttr {
    my ($ndir, $filehisto_ref, $title_ref, $xlabel, $key_ref, $seeplots) = @_;
}

sub plot_ttr {

    my ($filehisto, $title, $xlabel, $key, $seeplots) = @_;

    my $outplot = "$filehisto.ttr.ps";
    my $ylabel  = "RATE PARAMETERS";

    my $newkeyx = "$dnamlx";
    my $newkeye = "$dnamle";
    my $newkeyo = "$dnamlo";
    
    my $ymin = -2;
    my $ymax = 10;

    my $lsx = "3";
    my $lse = "2";
    my $lso = "1";

    open(GP,'|'.GNUPLOT) || die "Gnuplot: $!";

    print GP "set terminal postscript color solid 14\n";
    print GP "set style line 1 lt 1 lw 4\n";
    print GP "set style line 2 lt 2 lw 4\n";
    print GP "set style line 3  lt 3 lw 4\n";
    print GP "set style line 4 lt 4 lw 4\n";
    print GP "set style line 5 lt 5 lw 4\n";
    print GP "set style line 6 lt 6 lw 4\n";
    print GP "set style line 7 lt 7 lw 4\n";
    print GP "set style line 8 lt 8 lw 4\n";
    print GP "set style line 9 lt 9 lw 4\n";
    print GP "set style line 10 lt 10 lw 4\n";
    print GP "set style line 11 lt 11 lw 4\n";
    print GP "set style line 12 lt 12 lw 4\n";
    print GP "set style line 13 lt 13 lw 4\n";
    print GP "set style line 14 lt 14 lw 4\n";
    print GP "set style line 15 lt 15 lw 5\n";

    print GP "set output '$outplot'\n";
    print GP "set xlabel '$xlabel'\n";
    print GP "set title \"$title\\n\\n$key\"\n";
    
    print GP "set yrange [$ymin:$ymax]\n";

     if ($opt_b || $opt_o) {
	$ylabel = "RATE PARAMETERS  ** $newkeye";
	print GP "plot '$filehisto' using 1:32:33  with yerrorbars title 'TTR' ls 1, '$filehisto' using 1:34:35  with yerrorbars title 'ALPHA + BETA' ls 2\n";

    }
    if ($opt_o) {
	$ylabel = "RATE PARAMETERS  ** $newkeyo";
	print GP "plot '$filehisto' using 1:49:50  with yerrorbars title 'TTR' ls 1, '$filehisto' using 1:51:52  with yerrorbars title 'ALPHA + BETA' ls 2\n";
    }
    
    if ($opt_e) { 
	my $singlekey .= "$dnamlx"; 
	my $ls = $lsx; if ($opt_e) { $ls = $lse; }
	
	$ylabel = "RATE PARAMETERS ** $singlekey";
	print GP "plot '$filehisto' using 1:15:16  with yerrorbars title 'TTR' ls 1, '$filehisto' using 1:17:18  with yerrorbars title 'ALPHA + BETA' ls 2\n";
    }
    
    close (GP);
    
    if ($seeplots) { system ("$evince $outplot&\n"); }
}

