#!/usr/bin/perl -w
#fromtree2rosemsa2phyliptree.pl

use strict; 
use Class::Struct;

use vars qw ($opt_A $opt_B $opt_C $opt_D $opt_G $opt_R $opt_b $opt_c $opt_e $opt_E $opt_h $opt_i $opt_I  $opt_k $opt_l $opt_m $opt_M $opt_n $opt_o $opt_O $opt_p $opt_P $opt_r $opt_s $opt_t $opt_T $opt_v);  # required if strict used
use Getopt::Std;
use constant GNUPLOT => '/usr/bin/gnuplot';
getopts ('ABCDGRbceE:h:i:I:kl:m:M:n:oOp:P:r:st:T:v');


# Print a helpful message if the user provides no input file.
if (!@ARGV) {
        print "usage:  fromtree2rosemsa2phyliptree.pl [options] outdir\n\n";
        print "options:\n";
        print "-A           :  option for second test: branch opt only a la dnaml [default is CG_FULLMIN]\n";
	print "-B           :  option for second test: branch opt only using cg [default is CG_FULLMIN]\n";
	print "-C           :  option for third test: CG_FULLMIN [default is CG_FULLMIN with all parameters optimization]\n";
	print "-D           :  use dnaml-erate-dev [ default dnaml-erate]\n";
	print "-G           :  use erate to evolve seqs and generate alignments [ default rose]\n";
	print "-R           :  realign the sequences generated with rose/erate\n";
        print "-b           :  use both dnaml and dnaml-erate in the same msa\n";
        print "-c           :  do NOT clean up intermediate files with input alignments\n";
        print "-e           :  use dnaml-erate    [ default dnaml]\n";
        print "-E <num>     :  c/t parameter value for SIMPROT indel length distribution [ default 100]\n";
        print "-h <hisfile> :  give a histogram file for ploting only\n";
        print "-i <num>     :  inc len [ default 25]\n";
        print "-I <name>    :  rose indel distribution: 'oneindel' or 'flat' or 'poisson' or 'empirical' [default: poison]\n";
        print "-k           :  maintain the output files for each case\n";
        print "-l <num>     :  number of leaves in the tree [ default 8]\n";
        print "-m <num>     :  min len [ default 50]\n";
        print "-M <num>     :  max len [ default 1000]\n";
        print "-n <num>     :  number of msa per len and tree [ default 20]\n";
        print "-o           :  use 3 programs: dnaml and dnaml-erate and dnaml-erate optimizing all rate paramters in the same msa\n";
        print "-O           :  use dnamle v 0.0\n";
        print "-p <num>     :  insertionper [ default 0.0, from 0 to 100]\n";
        print "-P <num>     :  deletionper  [ default 0.0, from 0 to 100]\n";
        print "-r <num>     :  ratio of branch lengths on a 4-taxon tree (twobranchratio) [ default 1.0]\n";
        print "-s           :  simulate trees [default use a fixed tree]\n";
        print "-t <num>     :  rooted tree's average branch length (tbl) [ default 0.1]\n";
        print "-T <num>     :  number of different trees generated [ default 1]\n";
        print "-v           :  verbose\n";
	exit;
}
my $outdir = shift;

my $evince = "evince";

# $ERATEDIR (shell env variable)
my $eratedir =  $ENV{'ERATEDIR'};

# erate-generate
#
my $erate = "$eratedir/src/erate-generate/src/erate-generate";

# Rose version 1.3
#
my $rose = "$eratedir/src/rose-1.3/src/rose";

#Realign
#
my $realignwith = "/usr/local/clustalw/clustalw";
my $realign   = 0;
if ($opt_R) { $realign = 1;}

# PHYLIP version 3.66
#
my $phylipdir = "$eratedir/src/phylip3.66-erate/src/";
if ($opt_O) { $phylipdir = "$eratedir/versions/erate-v.0.0/phylip3.66-erate/src/"; }

my $phylipv = "PHYLIP_3.66";
my $phylip;
if ($opt_e) { 
    if ($opt_D) { $phylip = $phylipdir."dnaml-erate-dev"; $phylipv .= "-dnaml-erate-dev"; }
    else        { $phylip = $phylipdir."dnaml-erate";     $phylipv .= "-dnaml-erate";     }
}
else            { $phylip = $phylipdir."dnaml";           $phylipv .= "-dnaml";           }

my $treedist = $phylipdir."treedist";

my $phylip2;
my $phylip2v;
if ($opt_b || $opt_o) {
    if ($opt_D) {
	$phylip2  = $phylipdir."dnaml-erate-dev";
	$phylip2v = "PHYLIP_3.66-dnaml-erate-dev";
    }
    else {
	$phylip2  = $phylipdir."dnaml-erate";
	$phylip2v = "PHYLIP_3.66-dnaml-erate";
    }
}

my $testdir = "$eratedir/$outdir";
my $tempdir = "/tmp/$outdir";
system("mkdir $testdir\n");
system("mkdir -p $tempdir\n");
system("rm $testdir/*\n");
system("rm $tempdir/*\n");
my $plotonly;
if ($opt_h) { $plotonly = "$opt_h"; }

# a given tree or simulate trees
my $simtree = 0;
if ($opt_s) { $simtree = 1; }

# nubmer of taxa in the tree
my $sqnum = 8;
if ($opt_l) { 
    $sqnum = $opt_l; 
    if ($simtree == 0) {
	if ($sqnum != 8 && $sqnum != 4) {
	    print "sorry the fixed tree has to have 4 or 8 taxa\n"; die;
	}
    }
} 
# verbose options
my $seeplots = 0;
my $verbose = 0; if ($opt_v) { $verbose = 1; }
my $clean   = 1;
if ($opt_c) { $clean = 0; }

srand;

#
# A given case
#
my $casename;
my $casedir;
my $casetmpdir;

my $abl;            # tree's average branch length (abl)
my $twobranchratio; # for a fix 4-taxon tree with 2 types of branch lengths tb/ta

my $ntree;     # number of different trees considered
my $nmsa;      # number of msas generated per tree and len
my $sqlen;     # the msa generated by rose

my $model;     # the substitution model used by rose
my $freqa;
my $freqc;
my $freqg;
my $freqt;
my $ttratio;
my $meansubs;

my $indeldist; # the indel model used by rose
my $maxindel;
my $lambda;
my $c_over_t;
my $insertionper;
my $deletionper;
my $simcoding;
my $treetp2;


struct Histogram => {
histofile   => '$',
histofilewc => '$',
N           => '$',
k           => '$',
totx        => '@',
tote        => '@',
toto        => '@',
totxe       => '@',
totxo       => '@',
tpx         => '@',  # true positives
tpe         => '@',
tpo         => '@',
tpxe        => '@',
tpxo        => '@',
dist1x_ave  => '@', # distance 1
dist1e_ave  => '@',
dist1o_ave  => '@',
dist1xe_ave => '@',
dist1xo_ave => '@',
dist1x_std  => '@',
dist1e_std  => '@',
dist1o_std  => '@',
dist1xe_std => '@',
dist1xo_std => '@',
dist2x_ave  => '@', # distance 2
dist2e_ave  => '@',
dist2o_ave  => '@',
dist2xe_ave => '@',
dist2xo_ave => '@',
dist2x_std  => '@',
dist2e_std  => '@',
dist2o_std  => '@',
dist2xe_std => '@',
dist2xo_std => '@',
runtmx_ave  => '@', # running time
runtme_ave  => '@',
runtmo_ave  => '@',
runtmx_std  => '@',
runtme_std  => '@',
runtmo_std  => '@',
};

struct ParamHisto => {
histofile   => '$',
histofilewc => '$',
N           => '$',
k           => '$',
totx        => '@',
tote        => '@',
toto        => '@',
likex_ave   => '@', # lnlikelihood
likee_ave   => '@',
likeo_ave   => '@',
likex_std   => '@',
likee_std   => '@',
likeo_std   => '@',
ablx_ave    => '@', # average branch length
able_ave    => '@',
ablo_ave    => '@',
ablx_std    => '@',
able_std    => '@',
ablo_std    => '@',
alpx_ave    => '@', # alpha
alpe_ave    => '@',
alpo_ave    => '@',
alpx_std    => '@',
alpe_std    => '@',
alpo_std    => '@',
betx_ave    => '@', # beta
bete_ave    => '@',
beto_ave    => '@',
betx_std    => '@',
bete_std    => '@',
beto_std    => '@',
insx_ave    => '@',
inse_ave    => '@', # insertions rate
inso_ave    => '@',
insx_std    => '@',
inse_std    => '@',
inso_std    => '@',
delx_ave    => '@', # deletions rate
dele_ave    => '@',
delo_ave    => '@',
delx_std    => '@',
dele_std    => '@',
delo_std    => '@',
ttrx_ave    => '@', # transitions/transversion ratio
ttre_ave    => '@',
ttro_ave    => '@',
ttrx_std    => '@',
ttre_std    => '@',
ttro_std    => '@',
apbx_ave    => '@', # alpha + beta
apbe_ave    => '@',
apbo_ave    => '@',
apbx_std    => '@',
apbe_std    => '@',
apbo_std    => '@',
};

struct ParamGlobal => {
totx      => '$',
tote      => '$',
toto      => '$',
ablx_ave  => '$', # average branch length
able_ave  => '$',
ablo_ave  => '$',
ablx_std  => '$',
able_std  => '$',
ablo_std  => '$',
alpx_ave  => '$', # alpha
alpe_ave  => '$',
alpo_ave  => '$',
alpx_std  => '$',
alpe_std  => '$',
alpo_std  => '$',
betx_ave  => '$', # beta
bete_ave  => '$',
beto_ave  => '$',
betx_std  => '$',
bete_std  => '$',
beto_std  => '$',
insx_ave  => '$', # insertions rate
inse_ave  => '$',
inso_ave  => '$',
insx_std  => '$',
inse_std  => '$',
inso_std  => '$',
delx_ave  => '$', # deletions rate
dele_ave  => '$',
delo_ave  => '$',
delx_std  => '$',
dele_std  => '$',
delo_std  => '$',
ttrx_ave  => '$',  # transitions/transversion ratio
ttre_ave  => '$',
ttro_ave  => '$',
ttrx_std  => '$',
ttre_std  => '$',
ttro_std  => '$',
apbx_ave  => '$', # alpha + beta
apbe_ave  => '$',
apbo_ave  => '$',
apbx_std  => '$',
apbe_std  => '$',
apbo_std  => '$',
};

struct Tree => {
ntaxa  => '$',
parent => '@',
left   => '@',
right  => '@',
ld     => '@',
rd     => '@',
taxaparent    => '@', 
};


struct Case => {
name             => '$',
dir              => '$',
tmpdir           => '$',
dnadefaultsfile  => '$',
roseinputfile    => '$',
outfile          => '$',
ntree            => '$', # number of trees
nmsa             => '$', # number of msa's per tree
meanpairid       => '$', # average pairwise % identify of all msa's
stdpairid        => '$', # standard deviation % pairwise identify of all msa's
meanpairmut      => '$', # average pairwise % mutations of all msa's
stdpairmut       => '$', # standard deviation % mutations identify of all msa's
meanpairindl     => '$', # average pairwise % indels of all msa's
stdpairindl      => '$', # standard deviation % indels identify of all msa's
meanresfreq      => '@', # average residue frequency for all alignments
stdresfreq       => '@', # standard deviation residue frequency for all alignments
meanindelfreq    => '$', # average indel frequency for all alignments
stdindelfreq     => '$', # standard deviation indel frequency for all alignments
meanmsalen       => '$', # average len of all alignments
stdmsalen        => '$', # standard deviation for len of all alignments
sqlen            => '$', # the msa generate by rose
sqnum            => '$',
sqhash           => '%', # hash table to store the names of the sequences
ablu             => '$', # average unnormalized branch length
abln             => '$', # average normalized   branch length
twobranchratio   => '$', # for a fixed 4-taxon tree with 2 different brach lengths
treeunh          => '$', # the unnormalized tree in nh format
treennh          => '$', # the normalized   tree in nh format
treeu            => '$', # the unnormalized tree structure
treen            => '$', # the normalized   tree structure
model            => '$', # the substitution model for rose. JC' or 'F84' or 'F82' or 'HKY' are rose's defaults
freq             => '@',
alpha            => '$', # phylip rate parameter
beta             => '$', # phylip rate parameter
ttratio          => '$', # ttratio definde a la phylip
rosettratio      => '$', # corresponding rose ttratio
rosemeansubs     => '$', # corresponding rose meansubs
fracchange       => '$', # overall rate of substitutions per site
insertionper     => '$', # the indel model for rose
deletionper      => '$',
simcoding        => '$', # true if simulating coding indels
indeldist        => '$', # can be flat or poisson or empirical
maxindel         => '$', # maximum indel length
lambda           => '$', # parameter for the poisson distribution
c_over_t         => '$', # parameter for the empirical distribution
hfo              => '$', # histogram structure respect to frequency of indels
hmsalen          => '$', # histogram structure respect to msalen
hsqleng          => '$', # histogram structure respect to geometrical mean of sequences
hsqlena          => '$', # histogram structure respect to arithmetic mean of sequences
hpfo             => '$', # param histogram structure respect to frequency of indels
hpmsalen         => '$', # param histogram structure respect to msalen
hpsqleng         => '$', # param histogram structure respect to geometrical mean of sequences
hpsqlena         => '$', # param histogram structure respect to arithmetic mean of sequences
};

struct Globalstats => {
    treesofar       => '$',
    param           => '$', # structure for global stats on rate paramters
    meanpairid      => '$', # average pairwise % identify of all msa's
    stdpairid       => '$', # standard deviation % pairwise identify of all msa's
    meanpairmut     => '$', # average pairwise % mutations of all msa's
    stdpairmut      => '$', # standard deviation % mutations  identify of all msa's
    meanpairindl    => '$', # average pairwise % indels of all msa's
    stdpairindl     => '$', # standard deviation % indels  identify of all msa's
    meanindelfreq   => '$', # average indel frequency for all alignments and all cases
    stdindelfreq    => '$', # standard deviation indel frequency for all alignments and all cases
    meanresfreq     => '@', # average residue frequency for all alignments and all cases
    stdresfreq      => '@', # standard deviation residue frequency for all alignments and all cases
};

# averages for all alignments in all cases
my $globals;

# a given case
my $case;
$casedir    = "$testdir";
$casetmpdir = "$tempdir";
$nmsa = 20; if ($opt_n) { $nmsa = $opt_n };
$ntree = 1; if ($opt_T) { $ntree = $opt_T };
$sqlen = 300;
$abl = 0.1;            if ($opt_t) { $abl = $opt_t; } # average branch length
$twobranchratio = 1.0; if ($opt_r) { $twobranchratio = $opt_r; } # ratio of braches on a 4-taxon tree
$model = "F84";
$freqa = 0.25;
$freqc = 0.25;
$freqg = 0.25;
$freqt = 0.25;
$ttratio = "2.0"; # 2.0 is PHYLIP;s default - it has to be a string
$insertionper = "0.00";  # its a %  and it has to be a string
if ($opt_p) { $insertionper = "$opt_p"; }
$deletionper  = "0.00";  # its a %  and it has to be a string
if ($opt_P) { $deletionper = "$opt_P"; }
$indeldist = "poisson"; 
# this is an empirical distribution to simulate indels
# in a exon 
$simcoding = 0; 
if ($opt_I) { $indeldist = "$opt_I"; }
if ($indeldist =~/^empirical$/) { $simcoding = 1; }
$maxindel = 100;
$lambda = 0.5;             # parameter for the poisson distribution
$c_over_t = 100.0; # c/t parameter for the empirical distribution
if ($opt_E) { $c_over_t = $opt_E; if ( $c_over_t < 0) { print "c/t has to be positive\n"; die; } }
if ($simcoding == 1) { print "\nc/t=$c_over_t\n"; }

#set up variable
my $variable = "sqlen";
my $minval;
my $maxval;
my $valinc;

if ($variable =~ /sqlen/) {
    $minval = 50;   if ($opt_m) { $minval = $opt_m; }
    $maxval = 1000; if ($opt_M) { $maxval = $opt_M; }
    $valinc = 5;    if ($opt_i) { $valinc = $opt_i; }
    if ($maxval < $minval) {  print "Please, provide a positive range\n"; die; }
}
elsif ($variable =~ /time/) {
    $minval = 0.0;
    $maxval = 30;
    $valinc = 0.001;
}

my $nlcases = int(($maxval-$minval)/$valinc) + 1 ;
my $ntotalcases = $ntree * $nlcases;
my $ncase;
setup(\$globals, \$case, $casedir, $casetmpdir, $abl, $twobranchratio, $ntree, $nmsa, $sqlen, $sqnum, $phylipv, 
      $model, $freqa, $freqc, $freqg, $freqt, $ttratio, 
      $insertionper, $deletionper, $simcoding, $indeldist, $maxindel, $lambda, $c_over_t);

# do all the stuff for each case
#
if (!$plotonly){
    for (my $tree = 0, my $ct = 1; $tree < $ntree; $tree ++, $ct ++) {
	
	setup_tree(\$case, $simtree);
	
	# now proceed generating msa at different lenghts
	# and infering the tree from them
	for (my $val = $minval, my $cl = 1; $val <= $maxval; $val += $valinc, $cl ++) {
	    $casename = "C\.$ct\.$cl";
	    $ncase = $cl + $nlcases*($ct-1);
	    docase($ncase, $variable, $val, \$globals, \$case, $casename, $casedir, $casetmpdir);
	}
    }
}          

#
# calculate stats,
# write the histogram and plot
#
finishupallcases($variable, $minval, $case, \$globals, $ntotalcases);

##########################
# routines
#########################
sub accumulate_averages {
    my ($val, $meanval_ref, $meansquareval_ref) = @_;

    $$meanval_ref       += $val;
    $$meansquareval_ref += $val*$val;
}

sub calculate_averages {
    my ($meanval_ref, $stdval_ref, $number) = @_;

    my $mean = $$meanval_ref;
    my $std  = $$stdval_ref;

    if ($number > 1) {
	$mean /= $number;
	
	$std -= $mean*$mean*$number;
	$std /= ($number-1);

	if ($std < 0.) {
	    if ($std > -0.01) { $std = 0.0; }
	    else { print "cannot take average: number=$number sum=$$meanval_ref sqsum=$$stdval_ref mean=$mean sigma=$std\n"; die; }
	}
	$std  = sqrt($std);
    }
    elsif ($number == 1) {
	$mean /= $number;
	$std   = 0.0;
    }
    else {
	$mean = 0.0;
	$std  = 0.0;
    }

    $$meanval_ref = $mean;
    $$stdval_ref  = $std;

}

sub calculate_global_stats {
    my ($globals_ref, $ncases, $nmsa, $sqnum) = @_;
    
    my $param = $$globals_ref->param;
    my $tnmsa     = $nmsa*$ncases;             
    my $treesofar = int($ncases/$nlcases) + 1; # total number of trees analysed so far

    # total number of msa analysed so far
    my $tnmsax =  $nmsa*$param->{"ParamGlobal::totx"};
    my $tnmsae =  $nmsa*$param->{"ParamGlobal::tote"};
    my $tnmsao =  $nmsa*$param->{"ParamGlobal::toto"};

    $$globals_ref->{"Globalstats::treesofar"} = $treesofar;
    
    calculate_averages(\$param->{"ParamGlobal::ablx_ave"},   \$param->{"ParamGlobal::ablx_std"},  $tnmsax);
    calculate_averages(\$param->{"ParamGlobal::alpx_ave"},   \$param->{"ParamGlobal::alpx_std"},  $tnmsax);
    calculate_averages(\$param->{"ParamGlobal::betx_ave"},   \$param->{"ParamGlobal::betx_std"},  $tnmsax);
    calculate_averages(\$param->{"ParamGlobal::insx_ave"},   \$param->{"ParamGlobal::insx_std"},  $tnmsax);
    calculate_averages(\$param->{"ParamGlobal::delx_ave"},   \$param->{"ParamGlobal::delx_std"},  $tnmsax);
    calculate_averages(\$param->{"ParamGlobal::ttrx_ave"},   \$param->{"ParamGlobal::ttrx_std"},  $tnmsax);
    calculate_averages(\$param->{"ParamGlobal::apbx_ave"},   \$param->{"ParamGlobal::apbx_std"},  $tnmsax);
    
    calculate_averages(\$param->{"ParamGlobal::able_ave"},   \$param->{"ParamGlobal::able_std"},  $tnmsae);
    calculate_averages(\$param->{"ParamGlobal::alpe_ave"},   \$param->{"ParamGlobal::alpe_std"},  $tnmsae);
    calculate_averages(\$param->{"ParamGlobal::bete_ave"},   \$param->{"ParamGlobal::bete_std"},  $tnmsae);
    calculate_averages(\$param->{"ParamGlobal::inse_ave"},   \$param->{"ParamGlobal::inse_std"},  $tnmsae);
    calculate_averages(\$param->{"ParamGlobal::dele_ave"},   \$param->{"ParamGlobal::dele_std"},  $tnmsae);
    calculate_averages(\$param->{"ParamGlobal::ttre_ave"},   \$param->{"ParamGlobal::ttre_std"},  $tnmsae);
    calculate_averages(\$param->{"ParamGlobal::apbe_ave"},   \$param->{"ParamGlobal::apbe_std"},  $tnmsae);
    
    calculate_averages(\$param->{"ParamGlobal::ablo_ave"},   \$param->{"ParamGlobal::ablo_std"},  $tnmsao);
    calculate_averages(\$param->{"ParamGlobal::alpo_ave"},   \$param->{"ParamGlobal::alpo_std"},  $tnmsao);
    calculate_averages(\$param->{"ParamGlobal::beto_ave"},   \$param->{"ParamGlobal::beto_std"},  $tnmsao);
    calculate_averages(\$param->{"ParamGlobal::inso_ave"},   \$param->{"ParamGlobal::inso_std"},  $tnmsao);
    calculate_averages(\$param->{"ParamGlobal::delo_ave"},   \$param->{"ParamGlobal::delo_std"},  $tnmsao);
    calculate_averages(\$param->{"ParamGlobal::ttro_ave"},   \$param->{"ParamGlobal::ttro_std"},  $tnmsao);
    calculate_averages(\$param->{"ParamGlobal::apbo_ave"},   \$param->{"ParamGlobal::apbo_std"},  $tnmsao);

   # calculate stats for indel/residue frequencies
    calculate_averages(\$$globals_ref->{"Globalstats::meanindelfreq"},  
		       \$$globals_ref->{"Globalstats::stdindelfreq"},  $tnmsa);
    calculate_averages(\$$globals_ref->{"Globalstats::meanresfreq"}[0], 
		       \$$globals_ref->{"Globalstats::stdresfreq"}[0], $tnmsa);
    calculate_averages(\$$globals_ref->{"Globalstats::meanresfreq"}[1], 
		       \$$globals_ref->{"Globalstats::stdresfreq"}[1], $tnmsa);
    calculate_averages(\$$globals_ref->{"Globalstats::meanresfreq"}[2], 
		       \$$globals_ref->{"Globalstats::stdresfreq"}[2], $tnmsa);
    calculate_averages(\$$globals_ref->{"Globalstats::meanresfreq"}[3], 
		       \$$globals_ref->{"Globalstats::stdresfreq"}[3], $tnmsa);
    
    #calculate stats for pairwise identities/mutations/indels
    my $npairs = $tnmsa*$sqnum*($sqnum-1)/2;
    calculate_averages(\$$globals_ref->{"Globalstats::meanpairid"},   
		       \$$globals_ref->{"Globalstats::stdpairid"},   
		       $npairs);
    calculate_averages(\$$globals_ref->{"Globalstats::meanpairmut"},  
		       \$$globals_ref->{"Globalstats::stdpairmut"},  
		       $npairs);

    calculate_averages(\$$globals_ref->{"Globalstats::meanpairindl"}, 
		       \$$globals_ref->{"Globalstats::stdpairindl"}, 
		       $npairs);
}

sub check_tree_branches {
    my ($tree) = @_;

    my $ntaxa = $tree->{"Tree::ntaxa"};
    my $nnode = ($ntaxa > 1)? $ntaxa-1 : $ntaxa;
    my $haszerolen = 0;
    my $intl;
    my $intr;

    for (my $n = 0; $n < $nnode; $n ++) {
	$intl = int(${$tree->ld}[$n]*100000);
	$intr = int(${$tree->rd}[$n]*100000);

	if ($intl > 0. && $intr > 0.) { next; }
	else { return 1; }
    }

    return $haszerolen;

}

sub compare_between_trees {
    my ($case_ref, $globals_ref, $nali, 
	$phylipoutputfile_a, $phylipouttreefile_a, $phylipmode_a,   
	$phylipoutputfile_b, $phylipouttreefile_b, $phylipmode_b, 
	$msalen, $sqleng, $sqlena, $indelfrq) = @_;

    my $phyliptreenh_a;
    my $phyliptreenh_b;
    my $phyliptree_a;
    my $phyliptree_b;
    my $doesnotparse = 0;
    my $issametree = 0;

    my $hfo      = $$case_ref->hfo;
    my $hmsalen  = $$case_ref->hmsalen;
    my $hsqleng  = $$case_ref->hsqleng;
    my $hsqlena  = $$case_ref->hsqlena;
    
    my $Nfo      = $hfo->{"Histogram::N"};
    my $Nmsalen  = $hmsalen->{"Histogram::N"};
    my $Nsqleng  = $hsqleng->{"Histogram::N"};
    my $Nsqlena  = $hsqlena->{"Histogram::N"};

    my $kfo      = $hfo->{"Histogram::k"};
    my $kmsalen  = $hmsalen->{"Histogram::k"};
    my $ksqleng  = $hsqleng->{"Histogram::k"};
    my $ksqlena  = $hsqlena->{"Histogram::k"};

    #extract phylip trees in nh format from outtree
    extract_nhtree_from_phylip_outtree($phylipouttreefile_a, \$phyliptreenh_a);
    extract_nhtree_from_phylip_outtree($phylipouttreefile_b, \$phyliptreenh_b);

    if ( ($phylipmode_a == 0 && !$phyliptreenh_a) ||
	 ($phylipmode_b == 0 && !$phyliptreenh_b)   ) {
	$doesnotparse = 1;
    }
    else {
    # the phylip tree has the first sequence as right from root node
    # my convention is first sequence left from root node.
    # this function reverses the order of the tree in  nh format
	reverse_tree(\$phyliptreenh_a);
	reverse_tree(\$phyliptreenh_b);

	#phylip tree structure
	$phyliptree_a = Tree->new();
	nh2tree(\%{$$case_ref->{"Case::sqhash"}}, $phyliptreenh_a, \$phyliptree_a, \$doesnotparse);
	$phyliptree_b = Tree->new();
	nh2tree(\%{$$case_ref->{"Case::sqhash"}}, $phyliptreenh_b, \$phyliptree_b, \$doesnotparse);
    }
    
    # there was something wrong here, do not parse this result
    if ($doesnotparse) {
	open(OUT, ">>".$$case_ref->{"Case::outfile"});
	printf OUT "\nbetween_tree comparison: FAILED\n";
	close (OUT);
	printf "\nbetween_tree comparison: FAILED\n"; 
	return;
    }

    # extract the trees average branch length
    my $abl_a        = tree_abl($phyliptree_a);
    my $abl_b        = tree_abl($phyliptree_b);
    my $ablsq_a = $abl_a * $abl_a;
    my $ablsq_b = $abl_b * $abl_b;

    my $scale = 1.0;
    tree_rescale_dnaml(\$phyliptree_a, \$phyliptreenh_a, $phylipoutputfile_a, 
		       $phylipmode_a, 
		       $$case_ref->{"Case::alpha"}, $$case_ref->{"Case::beta"}, 
		       $scale);
    tree_rescale_dnaml(\$phyliptree_b, \$phyliptreenh_b, $phylipoutputfile_b, 
		       $phylipmode_b, 
		       $$case_ref->{"Case::alpha"}, $$case_ref->{"Case::beta"}, 
		       $scale);

    if ($verbose) { 
	print "\nphylip tree A:\n$phyliptreenh_a\n";
	print_tree($phyliptree_a);	
	print "\nphylip tree B:\n$phyliptreenh_b\n";
	print_tree($phyliptree_b);	
    }

    ###############################
    #
    # do the actual tree comparison 
    #
    ###############################
    $issametree = compare_trees_really($phyliptree_a, $phyliptree_b); 

    if ($phylipmode_a == 0 && $phylipmode_b == 1) {
	fill_histo_array(1, $indelfrq, $Nfo,      $kfo,      $hfo->{"Histogram::totxe"});
	fill_histo_array(1, $msalen,   $Nmsalen,  $kmsalen,  $hmsalen->{"Histogram::totxe"});
	fill_histo_array(1, $sqleng,   $Nsqleng,  $ksqleng,  $hsqleng->{"Histogram::totxe"});
	fill_histo_array(1, $sqlena,   $Nsqlena,  $ksqlena,  $hsqlena->{"Histogram::totxe"});

	fill_histo_array($issametree, $indelfrq, $Nfo,     $kfo,     $hfo->{"Histogram::tpxe"});
	fill_histo_array($issametree, $msalen,   $Nmsalen, $kmsalen, $hmsalen->{"Histogram::tpxe"});
	fill_histo_array($issametree, $sqleng,   $Nsqleng, $ksqleng, $hsqleng->{"Histogram::tpxe"});
	fill_histo_array($issametree, $sqlena,   $Nsqlena, $ksqlena, $hsqlena->{"Histogram::tpxe"});
    }
    elsif ($phylipmode_a == 0 && $phylipmode_b == 2) {
	fill_histo_array(1, $indelfrq, $Nfo,      $kfo,      $hfo->{"Histogram::totxo"});
	fill_histo_array(1, $msalen,   $Nmsalen,  $kmsalen,  $hmsalen->{"Histogram::totxo"});
	fill_histo_array(1, $sqleng,   $Nsqleng,  $ksqleng,  $hsqleng->{"Histogram::totxo"});
	fill_histo_array(1, $sqlena,   $Nsqlena,  $ksqlena,  $hsqlena->{"Histogram::totxo"});

	fill_histo_array($issametree, $indelfrq, $Nfo,     $kfo,     $hfo->{"Histogram::tpxo"});
	fill_histo_array($issametree, $msalen,   $Nmsalen, $kmsalen, $hmsalen->{"Histogram::tpxo"});
	fill_histo_array($issametree, $sqleng,   $Nsqleng, $ksqleng, $hsqleng->{"Histogram::tpxo"});
	fill_histo_array($issametree, $sqlena,   $Nsqlena, $ksqlena, $hsqlena->{"Histogram::tpxo"});
    }
    
    # compare using tree distance: Branch Score Distance
    my $dist1 = compare_trees_TREEDIST($case_ref, $nali, $phyliptreenh_a, $phyliptreenh_b, 1); 
    my $dist1sq = $dist1*$dist1;

    if ($phylipmode_a == 0 && $phylipmode_b == 1) {
	fill_histo_array($dist1,   $indelfrq, $Nfo,     $kfo,     $hfo->{"Histogram::dist1xe_ave"});
	fill_histo_array($dist1,   $msalen,   $Nmsalen, $kmsalen, $hmsalen->{"Histogram::dist1xe_ave"});
	fill_histo_array($dist1,   $sqleng,   $Nsqleng, $ksqleng, $hsqleng->{"Histogram::dist1xe_ave"});
	fill_histo_array($dist1,   $sqlena,   $Nsqlena, $ksqlena, $hsqlena->{"Histogram::dist1xe_ave"});
	fill_histo_array($dist1sq, $indelfrq, $Nfo,     $kfo,     $hfo->{"Histogram::dist1xe_std"});
	fill_histo_array($dist1sq, $msalen,   $Nmsalen, $kmsalen, $hmsalen->{"Histogram::dist1xe_std"});
	fill_histo_array($dist1sq, $sqleng,   $Nsqleng, $ksqleng, $hsqleng->{"Histogram::dist1xe_std"});
	fill_histo_array($dist1sq, $sqlena,   $Nsqlena, $ksqlena, $hsqlena->{"Histogram::dist1xe_std"});
    }
    elsif ($phylipmode_a == 0 && $phylipmode_b == 2) {
	fill_histo_array($dist1,   $indelfrq, $Nfo,     $kfo,     $hfo->{"Histogram::dist1xo_ave"});
	fill_histo_array($dist1,   $msalen,   $Nmsalen, $kmsalen, $hmsalen->{"Histogram::dist1xo_ave"});
	fill_histo_array($dist1,   $sqleng,   $Nsqleng, $ksqleng, $hsqleng->{"Histogram::dist1xo_ave"});
	fill_histo_array($dist1,   $sqlena,   $Nsqlena, $ksqlena, $hsqlena->{"Histogram::dist1xo_ave"});
	fill_histo_array($dist1sq, $indelfrq, $Nfo,     $kfo,     $hfo->{"Histogram::dist1xo_std"});
	fill_histo_array($dist1sq, $msalen,   $Nmsalen, $kmsalen, $hmsalen->{"Histogram::dist1xo_std"});
	fill_histo_array($dist1sq, $sqleng,   $Nsqleng, $ksqleng, $hsqleng->{"Histogram::dist1xo_std"});
	fill_histo_array($dist1sq, $sqlena,   $Nsqlena, $ksqlena, $hsqlena->{"Histogram::dist1xo_std"});
    }

    # compare using tree distance: Symmetric Difference 
    my $dist2 = compare_trees_TREEDIST($case_ref, $nali, $phyliptreenh_a, $phyliptreenh_b, 2); 
    my $dist2sq = $dist2*$dist2;

    if ($phylipmode_a == 0 && $phylipmode_b == 1) {
	fill_histo_array($dist2,   $indelfrq, $Nfo,     $kfo,     $hfo->{"Histogram::dist2xe_ave"});
	fill_histo_array($dist2,   $msalen,   $Nmsalen, $kmsalen, $hmsalen->{"Histogram::dist2xe_ave"});
	fill_histo_array($dist2,   $sqleng,   $Nsqleng, $ksqleng, $hsqleng->{"Histogram::dist2xe_ave"});
	fill_histo_array($dist2,   $sqlena,   $Nsqlena, $ksqlena, $hsqlena->{"Histogram::dist2xe_ave"});
	fill_histo_array($dist2sq, $indelfrq, $Nfo,     $kfo,     $hfo->{"Histogram::dist2xe_std"});
	fill_histo_array($dist2sq, $msalen,   $Nmsalen, $kmsalen, $hmsalen->{"Histogram::dist2xe_std"});
	fill_histo_array($dist2sq, $sqleng,   $Nsqleng, $ksqleng, $hsqleng->{"Histogram::dist2xe_std"});
	fill_histo_array($dist2sq, $sqlena,   $Nsqlena, $ksqlena, $hsqlena->{"Histogram::dist2xe_std"});
    }
    elsif ($phylipmode_a == 0 && $phylipmode_b == 2) {
	fill_histo_array($dist2,   $indelfrq, $Nfo,     $kfo,     $hfo->{"Histogram::dist2xo_ave"});
	fill_histo_array($dist2,   $msalen,   $Nmsalen, $kmsalen, $hmsalen->{"Histogram::dist2xo_ave"});
	fill_histo_array($dist2,   $sqleng,   $Nsqleng, $ksqleng, $hsqleng->{"Histogram::dist2xo_ave"});
	fill_histo_array($dist2,   $sqlena,   $Nsqlena, $ksqlena, $hsqlena->{"Histogram::dist2xo_ave"});
	fill_histo_array($dist2sq, $indelfrq, $Nfo,     $kfo,     $hfo->{"Histogram::dist2xo_std"});
	fill_histo_array($dist2sq, $msalen,   $Nmsalen, $kmsalen, $hmsalen->{"Histogram::dist2xo_std"});
	fill_histo_array($dist2sq, $sqleng,   $Nsqleng, $ksqleng, $hsqleng->{"Histogram::dist2xo_std"});
	fill_histo_array($dist2sq, $sqlena,   $Nsqlena, $ksqlena, $hsqlena->{"Histogram::dist2xo_std"});
    }

    # add to outputfile
    open(OUT, ">>".$$case_ref->{"Case::outfile"});
    printf OUT "rearranged tree A:\n%s\n", $phyliptreenh_a;
    printf OUT "rearranged tree B:\n%s\n", $phyliptreenh_b;
    printf OUT "tree comparison: %d dist1 %f dist2 %d\n", $issametree, $dist1, $dist2;	 
    printf OUT "average branch length: A %f B %f\n", $abl_a, $abl_b;	 
    close(OUT);
    printf "tree comparison: %d dist1 %f dist2 %d abl_a %f abl_b %f\n", 
    $issametree, $dist1, $dist2, $abl_a, $abl_b;	 

    # check consistency of results
    if ($dist2 == 0 && !$issametree) {
	print "\nphylip tree A:\n$phyliptreenh_a\n";
	print_tree($phyliptree_a);
	print "\nphylip tree B:\n$phyliptreenh_b\n";
	print_tree($phyliptree_b);
	die;
    }

    if ($verbose) { 
	print "phylip tree A\n";
	print_tree($phyliptree_a); 
	print "phylip tree B\n";
	print_tree($phyliptree_b); 
    }

}

sub compare_to_giventrees {
    my ($case_ref, $globals_ref, $nali, 
	$phylipoutputfile, $phylipouttreefile, $phyliptimefile, 
	$msalen, $sqleng, $sqlena, $indelfrq, $phylipmode) = @_;

    my $phyliptreenh;
    my $phyliptree;
    my $issametree = 0;
    my $doesnotparse = 0;

    my $treen   = $$case_ref->treen;
    my $treennh = $$case_ref->treennh;
    my $param   = $$globals_ref->param;
    
    my $hfo      = $$case_ref->hfo;
    my $hmsalen  = $$case_ref->hmsalen;
    my $hsqleng  = $$case_ref->hsqleng;
    my $hsqlena  = $$case_ref->hsqlena;
    
    my $hpfo     = $$case_ref->hpfo;
    my $hpmsalen = $$case_ref->hpmsalen;
    my $hpsqleng = $$case_ref->hpsqleng;
    my $hpsqlena = $$case_ref->hpsqlena;

    my $Nfo      = $hfo->{"Histogram::N"};
    my $Nmsalen  = $hmsalen->{"Histogram::N"};
    my $Nsqleng  = $hsqleng->{"Histogram::N"};
    my $Nsqlena  = $hsqlena->{"Histogram::N"};

    my $Npfo     = $hpfo->{"ParamHisto::N"};
    my $Npmsalen = $hpmsalen->{"ParamHisto::N"};
    my $Npsqleng = $hpsqleng->{"ParamHisto::N"};
    my $Npsqlena = $hpsqlena->{"ParamHisto::N"};

    my $kfo      = $hfo->{"Histogram::k"};
    my $kmsalen  = $hmsalen->{"Histogram::k"};
    my $ksqleng  = $hsqleng->{"Histogram::k"};
    my $ksqlena  = $hsqlena->{"Histogram::k"};

    my $kpfo     = $hpfo->{"ParamHisto::k"};
    my $kpmsalen = $hpmsalen->{"ParamHisto::k"};
    my $kpsqleng = $hpsqleng->{"ParamHisto::k"};
    my $kpsqlena = $hpsqlena->{"ParamHisto::k"};

    #extract phylip tree in nh format from outtree
    extract_nhtree_from_phylip_outtree($phylipouttreefile, \$phyliptreenh);
    
    if ($phylipmode == 0 && !$phyliptreenh) {
	$doesnotparse = 1;
    }
    else {
    # the phylip tree has the first sequence as right from root node
    # my convention is first sequence left from root node.
    # this function reverses the order of the tree in  nh format
	reverse_tree(\$phyliptreenh);

	#phylip tree structure
	$phyliptree = Tree->new();
	nh2tree(\%{$$case_ref->{"Case::sqhash"}}, $phyliptreenh, \$phyliptree, \$doesnotparse);
    }
    
    
    # there was something wrong here, do not parse this result
    if ($doesnotparse == 1) {
	open(OUT, ">>".$$case_ref->{"Case::outfile"});
	printf OUT "\ntree comparison: FAILED\n";
	close (OUT);
	printf "\ntree comparison: FAILED\n"; 
	return;
    }

    # extract the average branch length
    my $abl        = tree_abl($phyliptree);
    my $target_abl = tree_abl($treen);
    my $ablsq = $abl * $abl;
    # paranoia
    if (abs($target_abl - $$case_ref->{"Case::abln"}) > 0.00001) {
	print "bad tree? abl should be $$case_ref->{\"Case::abln\"} not $target_abl\n"; 
	print "given tree\n";
	print_tree($treen); 
	print "phylip tree\n";
	print_tree($phyliptree); 
	die;
    }
    
    # Lookup runtime
    fill_runtime_histo($phyliptimefile, $case_ref, $msalen, $sqleng, $sqlena, $indelfrq, $phylipmode);
    
    # fillup histogram for parameters
    fill_param_histo($phylipoutputfile, $case_ref, $globals_ref, $msalen, $sqleng, $sqlena, $indelfrq, $phylipmode);
    
    if ($phylipmode == 0) {
	# accumulate for global stats
	$param->{"ParamGlobal::totx"} += 1;
	accumulate_averages($abl, \$param->{"ParamGlobal::ablx_ave"}, \$param->{"ParamGlobal::ablx_std"});
	
	fill_histo_array($abl,   $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::ablx_ave"});
	fill_histo_array($abl,   $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::ablx_ave"});
	fill_histo_array($abl,   $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::ablx_ave"});
	fill_histo_array($abl,   $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::ablx_ave"});
	fill_histo_array($ablsq, $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::ablx_std"});
	fill_histo_array($ablsq, $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::ablx_std"});
	fill_histo_array($ablsq, $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::ablx_std"});
	fill_histo_array($ablsq, $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::ablx_std"});
    }
    elsif ($phylipmode == 1) {
	# accumulate for global stats
	$param->{"ParamGlobal::tote"} += 1;
	accumulate_averages($abl, \$param->{"ParamGlobal::able_ave"}, \$param->{"ParamGlobal::able_std"});
	
	fill_histo_array($abl,   $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::able_ave"});
	fill_histo_array($abl,   $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::able_ave"});
	fill_histo_array($abl,   $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::able_ave"});
	fill_histo_array($abl,   $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::able_ave"});
	fill_histo_array($ablsq, $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::able_std"});
	fill_histo_array($ablsq, $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::able_std"});
	fill_histo_array($ablsq, $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::able_std"});
	fill_histo_array($ablsq, $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::able_std"});
    }
    elsif ($phylipmode == 2) {
	# accumulate for global stats
	$param->{"ParamGlobal::toto"} += 1;
	accumulate_averages($abl, \$param->{"ParamGlobal::ablo_ave"}, \$param->{"ParamGlobal::ablo_std"});
	
	fill_histo_array($abl,   $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::ablo_ave"});
	fill_histo_array($abl,   $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::ablo_ave"});
	fill_histo_array($abl,   $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::ablo_ave"});
	fill_histo_array($abl,   $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::ablo_ave"});
	fill_histo_array($ablsq, $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::ablo_std"});
	fill_histo_array($ablsq, $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::ablo_std"});
	fill_histo_array($ablsq, $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::ablo_std"});
	fill_histo_array($ablsq, $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::ablo_std"});
    }
    
    # Before we compare the trees, we need to rescale the
    # branches of the trees produced with dnaml-erate.
    #
    # By using rose to add indels, we have increased the
    # degree of divergence of each branch by a unknown amount.
    #
    # plain dnaml does not care, since it replaces indels
    # with all possible nucleotides.
    #
    # dnaml-erate tries to model that extra divergence,
    # so the tree reflects that, and for alignments with indels
    # in general it produces trees with a larger abl than
    # the one we started with.
    #
    # I correct for this by rescaling the branch lengths
    # so that they have the target abl (average branch lenght)
    #
    # This effect affects dnaml-erate a lot more than
    # it does dnaml
    #
    # of course, this will ONLY affect the distance
    # that uses branch lenghts, 
    # and it is not just based on tree topology.
    #    
    # The above trick still does not allow us to compare BSD (branch score distances)
    # for different experiments with different target abl.
    #
    # In order to be able to compare across the bord, both the given tree, and
    # the found tree are rescaled to having an abl= 1.0 (an arbitrary number, but
    # the same for all cases.

    my $scale = 1.0;
    my $treen_scale = $treen; #the given tree rescaled
    my $treennh_scale;
    tree_copy(\$treen_scale, $treen);
    tree_rescale(\$treen_scale, \$treennh_scale, $scale);
    tree_rescale_dnaml(\$phyliptree, \$phyliptreenh, $phylipoutputfile, 
		       $phylipmode, 
		       $$case_ref->{"Case::alpha"}, $$case_ref->{"Case::beta"}, 
		       $scale);

    if ($verbose) { 
	print "\ntreen:\n$treennh\n";
	print_tree($treen);
	print "\nphylip tree:\n$phyliptreenh\n";
	print_tree($phyliptree);	
    }

    ###############################
    #
    # do the actual tree comparison 
    # (comparison are to the tree with normalized branch lengths)
    #
    ###############################
    $issametree = compare_trees_really($treen, $phyliptree); 
    
    if ($phylipmode == 0) {
	fill_histo_array(1, $indelfrq, $Nfo,      $kfo,      $hfo->{"Histogram::totx"});
	fill_histo_array(1, $msalen,   $Nmsalen,  $kmsalen,  $hmsalen->{"Histogram::totx"});
	fill_histo_array(1, $sqleng,   $Nsqleng,  $ksqleng,  $hsqleng->{"Histogram::totx"});
	fill_histo_array(1, $sqlena,   $Nsqlena,  $ksqlena,  $hsqlena->{"Histogram::totx"});

	fill_histo_array(1, $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::totx"});
	fill_histo_array(1, $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::totx"});
	fill_histo_array(1, $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::totx"});
	fill_histo_array(1, $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::totx"});
	
	fill_histo_array($issametree, $indelfrq, $Nfo,     $kfo,     $hfo->{"Histogram::tpx"});
	fill_histo_array($issametree, $msalen,   $Nmsalen, $kmsalen, $hmsalen->{"Histogram::tpx"});
	fill_histo_array($issametree, $sqleng,   $Nsqleng, $ksqleng, $hsqleng->{"Histogram::tpx"});
	fill_histo_array($issametree, $sqlena,   $Nsqlena, $ksqlena, $hsqlena->{"Histogram::tpx"});
    }
    elsif ($phylipmode == 1) {
	fill_histo_array(1, $indelfrq, $Nfo,     $kfo,     $hfo->{"Histogram::tote"});
	fill_histo_array(1, $msalen,   $Nmsalen, $kmsalen, $hmsalen->{"Histogram::tote"});
	fill_histo_array(1, $sqleng,   $Nsqleng, $ksqleng, $hsqleng->{"Histogram::tote"});
	fill_histo_array(1, $sqlena,   $Nsqlena, $ksqlena, $hsqlena->{"Histogram::tote"});

	fill_histo_array(1, $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::tote"});
	fill_histo_array(1, $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::tote"});
	fill_histo_array(1, $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::tote"});
	fill_histo_array(1, $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::tote"});
	
	fill_histo_array($issametree, $indelfrq, $Nfo,     $kfo,     $hfo->{"Histogram::tpe"});
	fill_histo_array($issametree, $msalen,   $Nmsalen, $kmsalen, $hmsalen->{"Histogram::tpe"});
	fill_histo_array($issametree, $sqleng,   $Nsqleng, $ksqleng, $hsqleng->{"Histogram::tpe"});
	fill_histo_array($issametree, $sqlena,   $Nsqlena, $ksqlena, $hsqlena->{"Histogram::tpe"});
    }
    elsif ($phylipmode == 2) {
	fill_histo_array(1, $indelfrq, $Nfo,     $kfo,     $hfo->{"Histogram::toto"});
	fill_histo_array(1, $msalen,   $Nmsalen, $kmsalen, $hmsalen->{"Histogram::toto"});
	fill_histo_array(1, $sqleng,   $Nsqleng, $ksqleng, $hsqleng->{"Histogram::toto"});
	fill_histo_array(1, $sqlena,   $Nsqlena, $ksqlena, $hsqlena->{"Histogram::toto"});

	fill_histo_array(1, $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::toto"});
	fill_histo_array(1, $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::toto"});
	fill_histo_array(1, $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::toto"});
	fill_histo_array(1, $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::toto"});
	
	fill_histo_array($issametree, $indelfrq, $Nfo,     $kfo,     $hfo->{"Histogram::tpo"});
	fill_histo_array($issametree, $msalen,   $Nmsalen, $kmsalen, $hmsalen->{"Histogram::tpo"});
	fill_histo_array($issametree, $sqleng,   $Nsqleng, $ksqleng, $hsqleng->{"Histogram::tpo"});
	fill_histo_array($issametree, $sqlena,   $Nsqlena, $ksqlena, $hsqlena->{"Histogram::tpo"});
    }
    
    # compare using tree distance: Branch Score Distance
    my $dist1 = compare_trees_TREEDIST($case_ref, $nali, $treennh_scale, $phyliptreenh, 1); # use the scaled target tree
    my $dist1sq = $dist1*$dist1;

    if ($phylipmode == 0) {
	fill_histo_array($dist1,   $indelfrq, $Nfo,     $kfo,     $hfo->{"Histogram::dist1x_ave"});
	fill_histo_array($dist1,   $msalen,   $Nmsalen, $kmsalen, $hmsalen->{"Histogram::dist1x_ave"});
	fill_histo_array($dist1,   $sqleng,   $Nsqleng, $ksqleng, $hsqleng->{"Histogram::dist1x_ave"});
	fill_histo_array($dist1,   $sqlena,   $Nsqlena, $ksqlena, $hsqlena->{"Histogram::dist1x_ave"});
	fill_histo_array($dist1sq, $indelfrq, $Nfo,     $kfo,     $hfo->{"Histogram::dist1x_std"});
	fill_histo_array($dist1sq, $msalen,   $Nmsalen, $kmsalen, $hmsalen->{"Histogram::dist1x_std"});
	fill_histo_array($dist1sq, $sqleng,   $Nsqleng, $ksqleng, $hsqleng->{"Histogram::dist1x_std"});
	fill_histo_array($dist1sq, $sqlena,   $Nsqlena, $ksqlena, $hsqlena->{"Histogram::dist1x_std"});
    }
    elsif ($phylipmode == 1) {
	fill_histo_array($dist1,   $indelfrq, $Nfo,     $kfo,     $hfo->{"Histogram::dist1e_ave"});
	fill_histo_array($dist1,   $msalen,   $Nmsalen, $kmsalen, $hmsalen->{"Histogram::dist1e_ave"});
	fill_histo_array($dist1,   $sqleng,   $Nsqleng, $ksqleng, $hsqleng->{"Histogram::dist1e_ave"});
	fill_histo_array($dist1,   $sqlena,   $Nsqlena, $ksqlena, $hsqlena->{"Histogram::dist1e_ave"});
	fill_histo_array($dist1sq, $indelfrq, $Nfo,     $kfo,     $hfo->{"Histogram::dist1e_std"});
	fill_histo_array($dist1sq, $msalen,   $Nmsalen, $kmsalen, $hmsalen->{"Histogram::dist1e_std"});
	fill_histo_array($dist1sq, $sqleng,   $Nsqleng, $ksqleng, $hsqleng->{"Histogram::dist1e_std"});
	fill_histo_array($dist1sq, $sqlena,   $Nsqlena, $ksqlena, $hsqlena->{"Histogram::dist1e_std"});
    }
    elsif ($phylipmode == 2) {
	fill_histo_array($dist1,   $indelfrq, $Nfo,     $kfo,     $hfo->{"Histogram::dist1o_ave"});
	fill_histo_array($dist1,   $msalen,   $Nmsalen, $kmsalen, $hmsalen->{"Histogram::dist1o_ave"});
	fill_histo_array($dist1,   $sqleng,   $Nsqleng, $ksqleng, $hsqleng->{"Histogram::dist1o_ave"});
	fill_histo_array($dist1,   $sqlena,   $Nsqlena, $ksqlena, $hsqlena->{"Histogram::dist1o_ave"});
	fill_histo_array($dist1sq, $indelfrq, $Nfo,     $kfo,     $hfo->{"Histogram::dist1o_std"});
	fill_histo_array($dist1sq, $msalen,   $Nmsalen, $kmsalen, $hmsalen->{"Histogram::dist1o_std"});
	fill_histo_array($dist1sq, $sqleng,   $Nsqleng, $ksqleng, $hsqleng->{"Histogram::dist1o_std"});
	fill_histo_array($dist1sq, $sqlena,   $Nsqlena, $ksqlena, $hsqlena->{"Histogram::dist1o_std"});
    }
    
    # compare using tree distance: Symmetric Difference 
    my $dist2 = compare_trees_TREEDIST($case_ref, $nali, $treennh, $phyliptreenh, 2); 
    my $dist2sq = $dist2*$dist2;
    if ($phylipmode == 0) {
	fill_histo_array($dist2,   $indelfrq, $Nfo,     $kfo,     $hfo->{"Histogram::dist2x_ave"});
	fill_histo_array($dist2,   $msalen,   $Nmsalen, $kmsalen, $hmsalen->{"Histogram::dist2x_ave"});
	fill_histo_array($dist2,   $sqleng,   $Nsqleng, $ksqleng, $hsqleng->{"Histogram::dist2x_ave"});
	fill_histo_array($dist2,   $sqlena,   $Nsqlena, $ksqlena, $hsqlena->{"Histogram::dist2x_ave"});
	fill_histo_array($dist2sq, $indelfrq, $Nfo,     $kfo,     $hfo->{"Histogram::dist2x_std"});
	fill_histo_array($dist2sq, $msalen,   $Nmsalen, $kmsalen, $hmsalen->{"Histogram::dist2x_std"});
	fill_histo_array($dist2sq, $sqleng,   $Nsqleng, $ksqleng, $hsqleng->{"Histogram::dist2x_std"});
	fill_histo_array($dist2sq, $sqlena,   $Nsqlena, $ksqlena, $hsqlena->{"Histogram::dist2x_std"});
    }
    elsif ($phylipmode == 1) {
	fill_histo_array($dist2,   $indelfrq, $Nfo,     $kfo,     $hfo->{"Histogram::dist2e_ave"});
	fill_histo_array($dist2,   $msalen,   $Nmsalen, $kmsalen, $hmsalen->{"Histogram::dist2e_ave"});
	fill_histo_array($dist2,   $sqleng,   $Nsqleng, $ksqleng, $hsqleng->{"Histogram::dist2e_ave"});
	fill_histo_array($dist2,   $sqlena,   $Nsqlena, $ksqlena, $hsqlena->{"Histogram::dist2e_ave"});
	fill_histo_array($dist2sq, $indelfrq, $Nfo,     $kfo,     $hfo->{"Histogram::dist2e_std"});
	fill_histo_array($dist2sq, $msalen,   $Nmsalen, $kmsalen, $hmsalen->{"Histogram::dist2e_std"});
	fill_histo_array($dist2sq, $sqleng,   $Nsqleng, $ksqleng, $hsqleng->{"Histogram::dist2e_std"});
	fill_histo_array($dist2sq, $sqlena,   $Nsqlena, $ksqlena, $hsqlena->{"Histogram::dist2e_std"});
    }
    elsif ($phylipmode == 2) {
	fill_histo_array($dist2,   $indelfrq, $Nfo,     $kfo,     $hfo->{"Histogram::dist2o_ave"});
	fill_histo_array($dist2,   $msalen,   $Nmsalen, $kmsalen, $hmsalen->{"Histogram::dist2o_ave"});
	fill_histo_array($dist2,   $sqleng,   $Nsqleng, $ksqleng, $hsqleng->{"Histogram::dist2o_ave"});
	fill_histo_array($dist2,   $sqlena,   $Nsqlena, $ksqlena, $hsqlena->{"Histogram::dist2o_ave"});
	fill_histo_array($dist2sq, $indelfrq, $Nfo,     $kfo,     $hfo->{"Histogram::dist2o_std"});
	fill_histo_array($dist2sq, $msalen,   $Nmsalen, $kmsalen, $hmsalen->{"Histogram::dist2o_std"});
	fill_histo_array($dist2sq, $sqleng,   $Nsqleng, $ksqleng, $hsqleng->{"Histogram::dist2o_std"});
	fill_histo_array($dist2sq, $sqlena,   $Nsqlena, $ksqlena, $hsqlena->{"Histogram::dist2o_std"});
    }
 
    # add to outputfile
    open(OUT, ">>".$$case_ref->{"Case::outfile"});
    printf OUT "rearranged tree:\n%s\n", $phyliptreenh;
    printf OUT "original   tree:\n%s\n", $$case_ref->{"Case::treennh"};
    printf OUT "tree comparison: %d dist1 %f dist2 %d\n", $issametree, $dist1, $dist2;	 
    printf OUT "average branch length: given %f found %f\n", $$case_ref->{"Case::abln"}, $abl;	 
    close(OUT);
    printf "tree comparison: %d dist1 %f dist2 %d abl %f target abl %f\n", 
    $issametree, $dist1, $dist2, $abl, $target_abl;	 

    # check consistency of results
    if ($dist2 == 0 && !$issametree) {
	print "\ngiven:\n$treennh\n";
	print_tree($treen);
	print "\nphylip tree:\n$phyliptreenh\n";
	print_tree($phyliptree);
	die;
    }

    if ($verbose) { 
	print "given tree\n";
	print_tree($treen); 
	print "phylip tree\n";
	print_tree($phyliptree); 
    }
}

sub compare_trees_TREEDIST {

    my ($case_ref, $nali, $tree1, $tree2, $method) = @_;

    my $treefile              = $$case_ref->{"Case::tmpdir"}."/treefile.".$$case_ref->{"Case::name"}.".n$nali";
    my $treedistinputfile     = $$case_ref->{"Case::tmpdir"}."/treedistinput.".$$case_ref->{"Case::name"}.".n$nali";
    my $treedistoutputfile    = $$case_ref->{"Case::tmpdir"}."/treedistoutput.".$$case_ref->{"Case::name"}.".n$nali";
    my $treedistscreenoutfile = $$case_ref->{"Case::tmpdir"}."/treedistscreenout.".$$case_ref->{"Case::name"}.".n$nali";

    my $distance = 0;

    # put both trees in one file
    open(FILE, ">$treefile");
    print FILE "$tree1\n";
    print FILE "$tree2\n";
    close (FILE);
    if ($verbose) { 
	print "the tredist input:\n";
	system ("more $treefile\n"); 
 	print "end tredist input\n\n";
   }

    # make treedist inputfile

    my $treefile_short = $treefile; # phylip has a limit on name length
    #if ($treefile_shor =~ /\/([^\/]+)$/) { $treefile_short = $1; }
    if ($verbose) { printf("NAME:$treefile\nSHORT:$treefile_short\n"); }

    open(IN, ">$treedistinputfile");
    print IN "$treefile_short\n";
    if ($method == 1) { print IN "Y\n"; }
    if ($method == 2) { print IN "D\n"; print IN "Y\n"; }
    close (IN);

    if ($verbose) { system ("more $treedistinputfile\n"); }

    # run TREEDIST
    # 
    system("$treedist < $treedistinputfile > $treedistscreenoutfile\n"); 
    system("mv outfile $treedistoutputfile\n");

    if ($verbose) { 
	print "the output:\n";
	system ("more $treedistoutputfile\n"); 
	print "end output\n\n";
    }

    # parse the output
    open(OUT, "$treedistoutputfile");
    while(<OUT>) {
	if (/^Trees\s+1\s+and\s+2:\s+(\S+)\s*$/) {
	    $distance = $1;
	    decimal(\$distance);
	}
    }
    close (OUT);


    if ($clean) {
	system("rm $treefile\n");
	system("rm $treedistinputfile\n");
	system("rm $treedistscreenoutfile\n");
	system("rm $treedistoutputfile\n");
    }

    return $distance;
}

# copy of easel funcion
#
# Function:  esl_tree_Compare()
# Incept:    SRE, Fri Sep 22 14:05:09 2006 [Janelia]
#
# Purpose:   Given two trees <T1> and <T2> for the same
#            set of <N> taxa (represented in the trees by the same
#            indices, <0..N-1>), compare the topologies of the
#            two trees.
#
#            For comparing unrooted topologies, be sure that <T1> and
#            <T2> both obey the unrooted tree convention that the
#            "root" is placed on the branch to taxon 0. (That is,
#            <T->left[0] = 0>.)
#
sub compare_trees_really {
    my ($tree1, $tree2) = @_;
    
    my $issametree = 1;
    my @Mg;
    my $child;
    my $a;
    my $b;

    # We're going to use the tree mapping function M(g) [Goodman79]:
    # M[g] for node g in T1 is the index of the lowest node in T2
    # that contains the same children taxa as the subtree
    # under g in T1.
       
    # We use the SDI algorithm [ZmasekEddy01] to construct M(g),
    # by postorder traversal of T1
    #
    for (my $g = $tree1->{"Tree::ntaxa"}-2; $g >= 0; $g--)
    {
      $child = ${$tree1->left}[$g];
      if ($child <= 0) { $a = ${$tree2->taxaparent}[-$child]; }
      else             { $a = ${$tree2->parent}[$Mg[$child]]; }

      $child = ${$tree1->right}[$g];
      if ($child <= 0) { $b = ${$tree2->taxaparent}[-$child]; }
      else             { $b = ${$tree2->parent}[$Mg[$child]];   }

      if ($a != $b) {  # a shortcut in SDI: special case for exact tree comparison 
	  return 0; 
      }
      $Mg[$g] = $a;
    }

    if ($verbose) { print "tree comparison: $issametree\n"; }

    return $issametree;
}

sub copy_paramhisto {
    my ($p_ref, $param) = @_;

    my $p = ParamGlobal->new();

    $p->totx($param->totx);
    $p->tote($param->tote);
    $p->toto($param->toto);

    $p->ablx_ave($param->ablx_ave);
    $p->ablx_std($param->ablx_std);
    $p->able_ave($param->able_ave);
    $p->able_std($param->able_std);
    $p->ablo_ave($param->ablo_ave);
    $p->ablo_std($param->ablo_std);

    $p->alpx_ave($param->alpx_ave);
    $p->alpx_std($param->alpx_std);
    $p->alpe_ave($param->alpe_ave);
    $p->alpe_std($param->alpe_std);
    $p->alpo_ave($param->alpo_ave);
    $p->alpo_std($param->alpo_std);

    $p->betx_ave($param->betx_ave);
    $p->betx_std($param->betx_std);
    $p->bete_ave($param->bete_ave);
    $p->bete_std($param->bete_std);
    $p->beto_ave($param->beto_ave);
    $p->beto_std($param->beto_std);

    $p->insx_ave($param->insx_ave);
    $p->insx_std($param->insx_std);
    $p->inse_ave($param->inse_ave);
    $p->inse_std($param->inse_std);
    $p->inso_ave($param->inso_ave);
    $p->inso_std($param->inso_std);

    $p->delx_ave($param->delx_ave);
    $p->delx_std($param->delx_std);
    $p->dele_ave($param->dele_ave);
    $p->dele_std($param->dele_std);
    $p->delo_ave($param->delo_ave);
    $p->delo_std($param->delo_std);

    $p->ttrx_ave($param->ttrx_ave);
    $p->ttrx_std($param->ttrx_std);
    $p->ttre_ave($param->ttre_ave);
    $p->ttre_std($param->ttre_std);
    $p->ttro_ave($param->ttro_ave);
    $p->ttro_std($param->ttro_std);

    $p->apbx_ave($param->apbx_ave);
    $p->apbx_std($param->apbx_std);
    $p->apbe_ave($param->apbe_ave);
    $p->apbe_std($param->apbe_std);
    $p->apbo_ave($param->apbo_ave);
    $p->apbo_std($param->apbo_std);


    $$p_ref = $p;

}

sub create_phylip_tree {
    my ($case_ref, $globals_ref, $nali, 
	$rosemsafile, $phylipoutputfile, $phylipouttreefile, $phyliptimefile, 
	$msalen, $sqleng, $sqlena, $indelfrq, $phylipmode) = @_;

    my $phylipinputfile     = $$case_ref->{"Case::tmpdir"}."/phylipinput.".$$case_ref->{"Case::name"}.".n$nali";
    my $phylipscreenoutfile = $$case_ref->{"Case::tmpdir"}."/phylipscreenout.".$$case_ref->{"Case::name"}.".n$nali";

    my $which_phylip;
    if    ($phylipmode == 0) { $which_phylip = $phylip;  }
    elsif ($phylipmode == 1) { $which_phylip = $phylip2; }
    elsif ($phylipmode == 2) { $which_phylip = $phylip2; }

    # run PHYLIP
    #
    # with default parameters: 
    # F84, ttratio = 2.0, a+b = 1
    # get distribution from the alignment
    #

    # make phylip inputfile

    my $rosemsafile_short = $rosemsafile; # phylip has a limit on name length
    #if ($rosemsafile_short =~ /\/([^\/]+)$/) { $rosemsafile_short = $1; }

    open(IN, ">$phylipinputfile");
    print IN "$rosemsafile_short\n";
    if ($phylipmode == 1) {
	if    ($opt_A) { print IN "6\n"; }
	elsif ($opt_B) { print IN "6\nB\n"; }
    }
    if ($phylipmode == 2) {
	if (!$opt_C) { print IN "7\n"; } # dnaml-erate option to optimize alpha y beta
    }
    print IN "Y\n";
    close (IN);
    if ($verbose) {system ("more $phylipinputfile\n"); }

    # run PHYLIP
    # 
    my $cmd = "$which_phylip < $phylipinputfile > $phylipscreenoutfile";
    system("(time -p $cmd) 2> $phyliptimefile\n"); 

    system("mv outfile $phylipoutputfile\n");
    system("mv outtree $phylipouttreefile\n");

    # add the testrun info to the summary file
    #
    system("more $phylipinputfile >> ".$$case_ref->{"Case::outfile"}."\n");
    system("more $phylipouttreefile >> ".$$case_ref->{"Case::outfile"}."\n");

    if ($verbose) { 
	system ("more $phylipoutputfile\n"); 
	print "the tree:\n";
	system ("more $phylipouttreefile\n"); 
    }

    if ($clean) {
	system("rm $phylipinputfile\n");
	system("rm $phylipscreenoutfile\n");
    }
}

sub create_tree_structures {
    my ($case_ref) = @_;

    my $doesnotparse = 0;

    my $treeu = Tree->new();
    my $treen = Tree->new();

    $treeu->ntaxa($$case_ref->{"Case::sqnum"});
    $treen->ntaxa($$case_ref->{"Case::sqnum"});

    nh2tree(\%{$$case_ref->{"Case::sqhash"}}, $$case_ref->{"Case::treeunh"}, \$treeu, \$doesnotparse);
    nh2tree(\%{$$case_ref->{"Case::sqhash"}}, $$case_ref->{"Case::treennh"}, \$treen, \$doesnotparse);

    if ($doesnotparse) {
	print "given tree does not parse\n"; die;
	print OUT "given tree does not parse\n"; die;
    }

    $$case_ref->{"Case::treeu"} = $treeu;
    $$case_ref->{"Case::treen"} = $treen;

}

sub create_msa {
    my ($case_ref, $globals_ref, $nali, $msafile, $msalen_ref, $sqleng_ref, $sqlena_ref, $indelfreq_ref) = @_;
    
    my $outputfile  = $$case_ref->{"Case::tmpdir"}."/output.".$$case_ref->{"Case::name"}.".n$nali";
    
    if ($opt_G) {
	run_erate($case_ref, $nali, $msafile);
    }
    else {
	run_rose($case_ref, $outputfile);
	rosemsa2phylip($outputfile, $msafile);
    }
    
    if ($verbose) { system("more $msafile\n"); }

    if ($realign) {
	realign($msafile);
    }
    if ($verbose) { system("more $msafile\n"); }

    # get stats from the imput alignment (in phylip format)
    my $meanpairid;
    my $stdpairid;
    my $meanpairmut;
    my $stdpairmut;
    my $meanpairindl;
    my $stdpairindl;
    my $indelfreq;
    my $afreq;
    my $cfreq;
    my $gfreq;
    my $tfreq;
    get_stats_msa($case_ref, $globals_ref, $msafile, 
		  \$meanpairid, \$stdpairid, \$meanpairmut, \$stdpairmut, \$meanpairindl, \$stdpairindl, 
		  $msalen_ref, $sqleng_ref, $sqlena_ref, \$indelfreq, \$afreq, \$cfreq, \$gfreq, \$tfreq);
    
    # add the testrun info to the summary file
    #
    header_for_testrun($case_ref, $nali, $meanpairid, $stdpairid, $meanpairmut, $stdpairmut, 
		       $meanpairindl, $stdpairindl, $$msalen_ref, 
		       $indelfreq, $afreq, $cfreq, $gfreq, $tfreq);
    system("more $msafile >> ".$$case_ref->{"Case::outfile"}."\n");
    
    $$indelfreq_ref = $indelfreq;

    if ($clean) {
	system("rm $outputfile\n");
    }
}

sub create_TheMutationProbability {
    my ($sqlen) = @_;

    my $prob;
    my $n = 0;
	
    $prob = "[";
	while ($n++ < $sqlen) {
	    $prob .= "1.0,";
	}
    $prob .= "]";
    
    return $prob;
}

#
# Indels can be created with the following distributions
#
# flat: 1/maxlen, up to maxlen
#
# poisson:
#
#
sub create_TheInsFunc {
    my ($outfile, $type, $maxlen, $lambda, $c_over_t) = @_;

    my $func;

    if ($type eq "oneindel") {
	$func = oneindel_IndelDist($maxlen);
    }
    elsif ($type eq "flat") {
	$func = flat_IndelDist($maxlen);
    }
    elsif ($type eq "poisson") {
	$func = poisson_IndelDist($outfile, $maxlen, $lambda);
    }
    elsif ($type eq "empirical") {
	$func = empirical_IndelDist($maxlen, $c_over_t);
    }
    else { print "indel distribution has to be either 'flat' or 'poisson' or 'empirical'\n"; die; }

    return $func;
}
sub oneindel_IndelDist {
    my ($maxlen) = @_;
    
    my $func;
    my $n = 0;
    
     $func = "[";
    while ($n++ <= $maxlen) {
	if ($n == 1) { $func .= "1.0, "; }
	else         { $func .= "0.0, "; }
    }
    $func .= "]";
    
    return $func;
}
sub flat_IndelDist {
    my ($maxlen) = @_;
    
    my $val;
    my $func;
    my $n = 0;
    
    $val = 1.0/$maxlen;
    
    $func = "[";
    while ($n++ <= $maxlen) {
	$func .= "$val,";
    }
    $func .= "]";
    
    return $func;
}
sub poisson_IndelDist {
    my ($outfile, $maxlen, $lambda) = @_;
    
    my $foofile = "$outfile.foo";
    my $val;
    my $func;
    my $n = 0;
    
    $val = exp(-$lambda);
    $func = "[";
    while ($n <= $maxlen) {
	my $sval;
	open(FOO, ">$foofile");
	    printf FOO  "%.20f\n", $val;
	close(FOO);
	open(FOO, "$foofile");
	while(<FOO>) {
	    if (/^(\S+)/) { $sval = $1; }
	}
	close(FOO);
	$func .= "$sval,"; 
	$n ++;
	$val *= $lambda/$n;
    }
    $func .= "]";
    
    system("rm $foofile\n");
    return $func;
}
sub empirical_IndelDist {
    my ($maxlen, $c_over_t) = @_;
    
    my $val;
    my $func;
    my $n = 1;
    my $norm = 0.0;
    my $steep = 3;
    
    my $m1 = 0.01027;
    my $m2 = 0.003031;
    my $m3 = 0.0006141;
    my $m4 = 0.00002090;

    my $alpha1 = $c_over_t/0.96;
    my $alpha2 = $c_over_t/3.13;
    my $alpha3 = $c_over_t/14.3;
    my $alpha4 = $c_over_t/81.7;

    while ($n <= $maxlen) {
	if ($n%$steep == 0) {
	    my $aa = $n/$steep;
	    $norm += $m1*exp(-$aa*$alpha1);
	    $norm += $m2*exp(-$aa*$alpha2);
	    $norm += $m3*exp(-$aa*$alpha3);
	    $norm += $m4*exp(-$aa*$alpha4);
	}
	$n ++;
    }

    $n = 1;
    $func = "[";
    while ($n <= $maxlen) {
	if ($n%$steep == 0) {
	    $val = 0.0;
	    my $aa = $n/$steep;
	    $val += $m1*exp(-$aa*$alpha1);
	    $val += $m2*exp(-$aa*$alpha2);
	    $val += $m3*exp(-$aa*$alpha3);
	    $val += $m4*exp(-$aa*$alpha4);
	    $val /= $norm;
	    decimal(\$val);

	    $func .= "$val,"; 
	}
	else {
	    $func .= "0.0,"; 
	}
	$n ++;
    }
    $func .= "]";
    
    return $func;
}

sub decimal {
    my ($val_ref) =@_;

    my $val = $$val_ref;
    my $newval;
    my $root;
    my $power;
    my $tol = 0.000001;

    if ($val =~ /^(\S+)e-[0]*(\d+)$/) {
	$root = $1;
	$power = $2;
	
	while ($root >= 1) { $power --; $root /= 10.; }

	if ($root =~ /^0\.(\S+)/) { $newval = "0."; $root = $1; }
	else { print "decimal(): something went wrong val=$val newval=$newval root=$root power=$power\n"; die; }

	my $n = 0;
	while ($n++<$power) { $newval .= "0"; }

	$newval .= $root;
    }
    elsif ($val =~ /^(\S+)e\+[0]+$/) {
	$newval = $1;
    }
    elsif ($val =~ /^(\S+)e\+[0]+(\d+)$/) {
	$root = $1;
	$power = $2;

	$newval = $root;
	while ($power > 0) { $power --; $newval *= 10.; }

    }
    else {
	$newval = $val;
    }

    # check
    if (abs($val-$newval) > $tol){ 
	printf "decimal(): bad value newval %f val %f diff %f tol %f\n", $newval, $val, abs($val-$newval), $tol; 
	die; 
    }

    $$val_ref = $newval;
}

#
# F84 has two parameters alpha and beta [FelsensteinChurchill96]
#
# dnaml allows to set R (the average ratio of transitions to transversion given by
#
#  R = [ p_a*p_g + p_c*p_t + alpha/beta * (p_a*p_g/p_R + p_c*p_t/p_Y) ]/ (p_R*p_Y)
#
#  introduce the quantities  aa = p_R * p_Y * R - p_a*p_g - p_c*p_t
#
#                            bb = p_a*p_g/p_R + p_c*p_t/p_Y
#
# then alpha/beta = aa/bb
#
# phylip introduces also the condition alpha+beta=1
#
# and then sets alpha = aa / (aa+bb)
#               beta  = bb / (aa+bb)
# 
sub dnaml_param {
    my ($alpha_ref, $beta_ref, $ttratio, $freqa, $freqc, $freqg, $freqt) = @_;
 
    my $alpha;
    my $beta;

    my $aa = ($freqa + $freqg) * ($freqc + $freqt) * $ttratio - $freqa * $freqg - $freqc * $freqt;
    my $bb = $freqa * $freqg / ($freqa + $freqg) + $freqc * $freqt / ($freqc + $freqt);

    if ($aa + $bb > 0.) {
	$alpha = $aa / ($aa+$bb);
	$beta  = $bb / ($aa+$bb);
    }

    $$alpha_ref = $alpha;
    $$beta_ref  = $beta;
}

sub docase {
    my ($ncase, $variable, $val, $globals_ref, $case_ref, $casename, $casedir, $casetmpdir) = @_;

    if ($variable =~ /sqlen/) { $sqlen = $val; }

    #set up variables
    setupcase ($case_ref, $casename, $casedir, $casetmpdir, $sqlen);

    # go through all required msa's
    for (my $n = 0; $n < $$case_ref->nmsa; $n ++) { 
	doallforcase($variable, $val, $case_ref, $globals_ref, $n); 
    }

    # write to files
    finishupcase ($case_ref);
    finishuppartially($variable, $minval, $$case_ref, $globals_ref, $ncase);   
 
    #compress the output file or remove it
    if ($opt_k) {
	system("gzip ".$$case_ref->{"Case::outfile"}."\n");
    }
    else {
	system("rm ".$$case_ref->{"Case::outfile"}."\n");
    }
}

sub doallforcase {
    my ($variable, $val, $case_ref, $globals_ref, $nali) = @_;

    my $msalen;
    my $sqleng;
    my $sqlena;
    my $indelfrq;
    my $phylipmode;

    my $msafile = $$case_ref->{"Case::tmpdir"}."/msa.".$$case_ref->{"Case::name"}.".n$nali";

    printf "%s-%d>variable %s = %d\n", $$case_ref->{"Case::name"}, $nali+1, $variable, $val;

    # from the given tree, create the MSA using rose or erate
    #
    create_msa($case_ref, $globals_ref, $nali, $msafile, \$msalen, \$sqleng, \$sqlena, \$indelfrq);
 
    # From the MSA, infer the tree 
    #
    $phylipmode = 0; # dnaml mode
    my $phylipouttreefile0 = $$case_ref->{"Case::tmpdir"}."/phyliptreefile.".$$case_ref->{"Case::name"}.".n$nali.m$phylipmode";
    my $phylipoutputfile0  = $$case_ref->{"Case::tmpdir"}."/phylipoutfile.".$$case_ref->{"Case::name"}.".n$nali.m$phylipmode";
    phylip_analysis($case_ref, $globals_ref, $nali, $msafile, $msalen, $sqleng, $sqlena, $indelfrq, $phylipmode, $phylipoutputfile0, $phylipouttreefile0); 

    $phylipmode = 1; # dnaml-erate mode (the optimization options are selected later)
    my $phylipouttreefile1 = $$case_ref->{"Case::tmpdir"}."/phyliptreefile.".$$case_ref->{"Case::name"}.".n$nali.m$phylipmode";
    my $phylipoutputfile1  = $$case_ref->{"Case::tmpdir"}."/phylipoutfile.".$$case_ref->{"Case::name"}.".n$nali.m$phylipmode";
    if ($opt_b || $opt_o) { 
	phylip_analysis($case_ref, $globals_ref, $nali, $msafile, $msalen, $sqleng, $sqlena, $indelfrq, $phylipmode, $phylipoutputfile1, $phylipouttreefile1); 
    }
    
    $phylipmode = 2; # dnaml-erate mode (the optimization options are selected later)
    my $phylipouttreefile2 = $$case_ref->{"Case::tmpdir"}."/phyliptreefile.".$$case_ref->{"Case::name"}.".n$nali.m$phylipmode";
    my $phylipoutputfile2  = $$case_ref->{"Case::tmpdir"}."/phylipoutfile.".$$case_ref->{"Case::name"}.".n$nali.m$phylipmode";
    if ($opt_o) { 
	phylip_analysis($case_ref, $globals_ref, $nali, $msafile, $msalen, $sqleng, $sqlena, $indelfrq, $phylipmode, $phylipoutputfile2, $phylipouttreefile2); 
    }
    
    compare_between_trees($case_ref, $globals_ref, $nali, 
			  $phylipoutputfile0, $phylipouttreefile0, 0,   
			  $phylipoutputfile1, $phylipouttreefile1, 1, 
			  $msalen, $sqleng, $sqlena, $indelfrq);
    compare_between_trees($case_ref, $globals_ref, $nali, 
			  $phylipoutputfile0, $phylipouttreefile0, 0,   
			  $phylipoutputfile2, $phylipouttreefile2, 2, 
			  $msalen, $sqleng, $sqlena, $indelfrq);
    
    if ($clean) {
	system("rm $msafile\n");
	system("rm $phylipoutputfile0\n");
	system("rm $phylipoutputfile1\n");
	system("rm $phylipoutputfile2\n");
	system("rm $phylipouttreefile0\n");
	system("rm $phylipouttreefile1\n");
	system("rm $phylipouttreefile2\n");
    }

    $$case_ref = $case;
}

sub extract_nhtree_from_phylip_outtree {
    my ($phylipouttreefile, $treenh_ref) = @_;

    my $treenh = "";

    open(FILE, "$phylipouttreefile");
    while(<FILE>) {
	if (/^\s+$/)      { next; }
	elsif (/^(\S+)$/) { $treenh .= "$1"; }
	else              { $treenh = ""; last; }
    }
    close (FILE);

    # if phylip cannot calculate times, it writes \s+nan, and treenh will 
    # end up empty
    if (!$treenh) { 
	print "phylip3.66 did not generate any tree\n"; 
    }

    $$treenh_ref = $treenh;
}

sub fill_histo_array {
    my ($val, $len, $N, $k, $his_ref) = @_;
    my $dim = $N * $k;
    
    if ($len >=  $N) { $his_ref->[$dim] += 1; return; }

    for (my $i=0; $i<=$dim; $i++) { 
	if ( $i/$k <= $len && $len < ($i+1)/$k) { 
	    $his_ref->[$i] += $val; 
	    last; 
	} 
    }
}

sub fill_runtime_histo {
    my ($phyliptimefile, $case_ref, $msalen, $sqleng, $sqlena, $indelfrq, $phylipmode) = @_;

    my $runtime = parse_phylipscreenout($phyliptimefile);
    my $runtimesq = $runtime * $runtime;
    my $hfo     = $$case_ref->hfo;
    my $hmsalen = $$case_ref->hmsalen;
    my $hsqleng = $$case_ref->hsqleng;
    my $hsqlena = $$case_ref->hsqlena; 
    my $Nfo     = $hfo->{"Histogram::N"};
    my $Nmsalen = $hmsalen->{"Histogram::N"};
    my $Nsqleng = $hsqleng->{"Histogram::N"};
    my $Nsqlena = $hsqlena->{"Histogram::N"};

    my $kfo     = $hfo->{"Histogram::k"};
    my $kmsalen = $hmsalen->{"Histogram::k"};
    my $ksqleng = $hsqleng->{"Histogram::k"};
    my $ksqlena = $hsqlena->{"Histogram::k"};

    if ($phylipmode == 0) {
	fill_histo_array($runtime,   $indelfrq, $Nfo,     $kfo,     $hfo->{"Histogram::runtmx_ave"});
	fill_histo_array($runtime,   $msalen,   $Nmsalen, $kmsalen, $hmsalen->{"Histogram::runtmx_ave"});
	fill_histo_array($runtime,   $sqleng,   $Nsqleng, $ksqleng, $hsqleng->{"Histogram::runtmx_ave"});
	fill_histo_array($runtime,   $sqlena,   $Nsqlena, $ksqlena, $hsqlena->{"Histogram::runtmx_ave"});
	fill_histo_array($runtimesq, $indelfrq, $Nfo,     $kfo,     $hfo->{"Histogram::runtmx_std"});
	fill_histo_array($runtimesq, $msalen,   $Nmsalen, $kmsalen, $hmsalen->{"Histogram::runtmx_std"});
	fill_histo_array($runtimesq, $sqleng,   $Nsqleng, $ksqleng, $hsqleng->{"Histogram::runtmx_std"});
	fill_histo_array($runtimesq, $sqlena,   $Nsqlena, $ksqlena, $hsqlena->{"Histogram::runtmx_std"});
    }
    elsif ($phylipmode == 1) {
	fill_histo_array($runtime,   $indelfrq, $Nfo,     $kfo,     $hfo->{"Histogram::runtme_ave"});
	fill_histo_array($runtime,   $msalen,   $Nmsalen, $kmsalen, $hmsalen->{"Histogram::runtme_ave"});
	fill_histo_array($runtime,   $sqleng,   $Nsqleng, $ksqleng, $hsqleng->{"Histogram::runtme_ave"});
	fill_histo_array($runtime,   $sqlena,   $Nsqlena, $ksqlena, $hsqlena->{"Histogram::runtme_ave"});
	fill_histo_array($runtimesq, $indelfrq, $Nfo,     $kfo,     $hfo->{"Histogram::runtme_std"});
	fill_histo_array($runtimesq, $msalen,   $Nmsalen, $kmsalen, $hmsalen->{"Histogram::runtme_std"});
	fill_histo_array($runtimesq, $sqleng,   $Nsqleng, $ksqleng, $hsqleng->{"Histogram::runtme_std"});
	fill_histo_array($runtimesq, $sqlena,   $Nsqlena, $ksqlena, $hsqlena->{"Histogram::runtme_std"});
    }
    elsif ($phylipmode == 2) {
	fill_histo_array($runtime,   $indelfrq, $Nfo,     $kfo,     $hfo->{"Histogram::runtmo_ave"});
	fill_histo_array($runtime,   $msalen,   $Nmsalen, $kmsalen, $hmsalen->{"Histogram::runtmo_ave"});
	fill_histo_array($runtime,   $sqleng,   $Nsqleng, $ksqleng, $hsqleng->{"Histogram::runtmo_ave"});
	fill_histo_array($runtime,   $sqlena,   $Nsqlena, $ksqlena, $hsqlena->{"Histogram::runtmo_ave"});
	fill_histo_array($runtimesq, $indelfrq, $Nfo,     $kfo,     $hfo->{"Histogram::runtmo_std"});
	fill_histo_array($runtimesq, $msalen,   $Nmsalen, $kmsalen, $hmsalen->{"Histogram::runtmo_std"});
	fill_histo_array($runtimesq, $sqleng,   $Nsqleng, $ksqleng, $hsqleng->{"Histogram::runtmo_std"});
	fill_histo_array($runtimesq, $sqlena,   $Nsqlena, $ksqlena, $hsqlena->{"Histogram::runtmo_std"});
    }

}

sub fill_param_histo {
    my ($phylipoutfile, $case_ref, $globals_ref, $msalen, $sqleng, $sqlena, $indelfrq, $phylipmode) = @_;

    my $param = $$globals_ref->param;

    my $like = 0.0;
    my $alp  = 0.0;
    my $bet  = 0.0;
    my $ins  = 0.0;
    my $del  = 0.0;
    my $ttr  = 0.0;
    my $apb  = 0.0;
    my $frqa = 0.0;
    my $frqc = 0.0;
    my $frqg = 0.0;
    my $frqt = 0.0;
    my $frqo = 0.0;

    parse_phylipoutfile($phylipmode, $phylipoutfile, \$like, \$frqa, \$frqc, \$frqg, \$frqt, \$frqo, 
			\$alp, \$bet, \$ins, \$del, \$ttr, \$apb, 1);

    my $likesq = $like * $like;
    my $alpsq  = $alp * $alp;
    my $betsq  = $bet * $bet;
    my $inssq  = $ins * $ins;
    my $delsq  = $del * $del;
    my $ttrsq  = $ttr * $ttr;
    my $apbsq  = $apb * $apb;

    my $hpfo     = $$case_ref->hpfo;
    my $hpmsalen = $$case_ref->hpmsalen;
    my $hpsqleng = $$case_ref->hpsqleng;
    my $hpsqlena = $$case_ref->hpsqlena;

    my $Npfo     = $hpfo->{"ParamHisto::N"};
    my $Npmsalen = $hpmsalen->{"ParamHisto::N"};
    my $Npsqleng = $hpsqleng->{"ParamHisto::N"};
    my $Npsqlena = $hpsqlena->{"ParamHisto::N"};

    my $kpfo     = $hpfo->{"ParamHisto::k"};
    my $kpmsalen = $hpmsalen->{"ParamHisto::k"};
    my $kpsqleng = $hpsqleng->{"ParamHisto::k"};
    my $kpsqlena = $hpsqlena->{"ParamHisto::k"};


    if ($phylipmode == 0) {
	# accumulate for global stats
	accumulate_averages($alp,  \$param->{"ParamGlobal::alpx_ave"},  \$param->{"ParamGlobal::alpx_std"});
	accumulate_averages($bet,  \$param->{"ParamGlobal::betx_ave"},  \$param->{"ParamGlobal::betx_std"});
	accumulate_averages($ins,  \$param->{"ParamGlobal::insx_ave"},  \$param->{"ParamGlobal::insx_std"});
	accumulate_averages($del,  \$param->{"ParamGlobal::delx_ave"},  \$param->{"ParamGlobal::delx_std"});
	accumulate_averages($ttr,  \$param->{"ParamGlobal::ttrx_ave"},  \$param->{"ParamGlobal::ttrx_std"});
	accumulate_averages($apb,  \$param->{"ParamGlobal::apbx_ave"},  \$param->{"ParamGlobal::apbx_std"});

	fill_histo_array($like,   $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::likex_ave"});
	fill_histo_array($like,   $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::likex_ave"});
	fill_histo_array($like,   $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::likex_ave"});
	fill_histo_array($like,   $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::likex_ave"});
	fill_histo_array($likesq, $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::likex_std"});
	fill_histo_array($likesq, $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::likex_std"});
	fill_histo_array($likesq, $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::likex_std"});
	fill_histo_array($likesq, $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::likex_std"});

	fill_histo_array($alp,   $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::alpx_ave"});
	fill_histo_array($alp,   $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::alpx_ave"});
	fill_histo_array($alp,   $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::alpx_ave"});
	fill_histo_array($alp,   $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::alpx_ave"});
	fill_histo_array($alpsq, $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::alpx_std"});
	fill_histo_array($alpsq, $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::alpx_std"});
	fill_histo_array($alpsq, $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::alpx_std"});
	fill_histo_array($alpsq, $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::alpx_std"});

	fill_histo_array($bet,   $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::betx_ave"});
	fill_histo_array($bet,   $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::betx_ave"});
	fill_histo_array($bet,   $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::betx_ave"});
	fill_histo_array($bet,   $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::betx_ave"});
	fill_histo_array($betsq, $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::betx_std"});
	fill_histo_array($betsq, $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::betx_std"});
	fill_histo_array($betsq, $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::betx_std"});
	fill_histo_array($betsq, $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::betx_std"});

	fill_histo_array($ins,   $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::insx_ave"});
	fill_histo_array($ins,   $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::insx_ave"});
	fill_histo_array($ins,   $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::insx_ave"});
	fill_histo_array($ins,   $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::insx_ave"});
	fill_histo_array($inssq, $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::insx_std"});
	fill_histo_array($inssq, $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::insx_std"});
	fill_histo_array($inssq, $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::insx_std"});
	fill_histo_array($inssq, $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::insx_std"});
 
	fill_histo_array($del,   $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::delx_ave"});
	fill_histo_array($del,   $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::delx_ave"});
	fill_histo_array($del,   $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::delx_ave"});
	fill_histo_array($del,   $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::delx_ave"});
	fill_histo_array($delsq, $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::delx_std"});
	fill_histo_array($delsq, $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::delx_std"});
	fill_histo_array($delsq, $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::delx_std"});
	fill_histo_array($delsq, $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::delx_std"});

	fill_histo_array($ttr,   $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::ttrx_ave"});
	fill_histo_array($ttr,   $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::ttrx_ave"});
	fill_histo_array($ttr,   $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::ttrx_ave"});
	fill_histo_array($ttr,   $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::ttrx_ave"});
	fill_histo_array($ttrsq, $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::ttrx_std"});
	fill_histo_array($ttrsq, $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::ttrx_std"});
	fill_histo_array($ttrsq, $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::ttrx_std"});
	fill_histo_array($ttrsq, $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::ttrx_std"});

	fill_histo_array($apb,   $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::apbx_ave"});
	fill_histo_array($apb,   $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::apbx_ave"});
	fill_histo_array($apb,   $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::apbx_ave"});
	fill_histo_array($apb,   $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::apbx_ave"});
	fill_histo_array($apbsq, $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::apbx_std"});
	fill_histo_array($apbsq, $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::apbx_std"});
	fill_histo_array($apbsq, $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::apbx_std"});
	fill_histo_array($apbsq, $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::apbx_std"});
    }
    elsif ($phylipmode == 1) {
	accumulate_averages($alp,  \$param->{"ParamGlobal::alpe_ave"},  \$param->{"ParamGlobal::alpe_std"});
	accumulate_averages($bet,  \$param->{"ParamGlobal::bete_ave"},  \$param->{"ParamGlobal::bete_std"});
	accumulate_averages($ins,  \$param->{"ParamGlobal::inse_ave"},  \$param->{"ParamGlobal::inse_std"});
	accumulate_averages($del,  \$param->{"ParamGlobal::dele_ave"},  \$param->{"ParamGlobal::dele_std"});
	accumulate_averages($ttr,  \$param->{"ParamGlobal::ttre_ave"},  \$param->{"ParamGlobal::ttre_std"});
	accumulate_averages($apb,  \$param->{"ParamGlobal::apbe_ave"},  \$param->{"ParamGlobal::apbe_std"});

	fill_histo_array($like,   $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::likee_ave"});
	fill_histo_array($like,   $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::likee_ave"});
	fill_histo_array($like,   $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::likee_ave"});
	fill_histo_array($like,   $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::likee_ave"});
	fill_histo_array($likesq, $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::likee_std"});
	fill_histo_array($likesq, $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::likee_std"});
	fill_histo_array($likesq, $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::likee_std"});
	fill_histo_array($likesq, $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::likee_std"});

	fill_histo_array($alp,   $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::alpe_ave"});
	fill_histo_array($alp,   $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::alpe_ave"});
	fill_histo_array($alp,   $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::alpe_ave"});
	fill_histo_array($alp,   $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::alpe_ave"});
	fill_histo_array($alpsq, $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::alpe_std"});
	fill_histo_array($alpsq, $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::alpe_std"});
	fill_histo_array($alpsq, $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::alpe_std"});
	fill_histo_array($alpsq, $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::alpe_std"});

	fill_histo_array($bet,   $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::bete_ave"});
	fill_histo_array($bet,   $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::bete_ave"});
	fill_histo_array($bet,   $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::bete_ave"});
	fill_histo_array($bet,   $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::bete_ave"});
	fill_histo_array($betsq, $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::bete_std"});
	fill_histo_array($betsq, $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::bete_std"});
	fill_histo_array($betsq, $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::bete_std"});
	fill_histo_array($betsq, $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::bete_std"});

	fill_histo_array($ins,   $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::inse_ave"});
	fill_histo_array($ins,   $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::inse_ave"});
	fill_histo_array($ins,   $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::inse_ave"});
	fill_histo_array($ins,   $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::inse_ave"});
	fill_histo_array($inssq, $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::inse_std"});
	fill_histo_array($inssq, $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::inse_std"});
	fill_histo_array($inssq, $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::inse_std"});
	fill_histo_array($inssq, $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::inse_std"});

	fill_histo_array($del,   $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::dele_ave"});
	fill_histo_array($del,   $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::dele_ave"});
	fill_histo_array($del,   $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::dele_ave"});
	fill_histo_array($del,   $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::dele_ave"});
	fill_histo_array($delsq, $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::dele_std"});
	fill_histo_array($delsq, $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::dele_std"});
	fill_histo_array($delsq, $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::dele_std"});
	fill_histo_array($delsq, $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::dele_std"});

	fill_histo_array($ttr,   $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::ttre_ave"});
	fill_histo_array($ttr,   $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::ttre_ave"});
	fill_histo_array($ttr,   $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::ttre_ave"});
	fill_histo_array($ttr,   $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::ttre_ave"});
	fill_histo_array($ttrsq, $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::ttre_std"});
	fill_histo_array($ttrsq, $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::ttre_std"});
	fill_histo_array($ttrsq, $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::ttre_std"});
	fill_histo_array($ttrsq, $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::ttre_std"});

	fill_histo_array($apb,   $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::apbe_ave"});
	fill_histo_array($apb,   $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::apbe_ave"});
	fill_histo_array($apb,   $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::apbe_ave"});
	fill_histo_array($apb,   $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::apbe_ave"});
	fill_histo_array($apbsq, $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::apbe_std"});
	fill_histo_array($apbsq, $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::apbe_std"});
	fill_histo_array($apbsq, $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::apbe_std"});
	fill_histo_array($apbsq, $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::apbe_std"});
    }
    elsif ($phylipmode == 2) {
	# accumulate for global stats
	accumulate_averages($alp,  \$param->{"ParamGlobal::alpo_ave"},  \$param->{"ParamGlobal::alpo_std"});
	accumulate_averages($bet,  \$param->{"ParamGlobal::beto_ave"},  \$param->{"ParamGlobal::beto_std"});
	accumulate_averages($ins,  \$param->{"ParamGlobal::inso_ave"},  \$param->{"ParamGlobal::inso_std"});
	accumulate_averages($del,  \$param->{"ParamGlobal::delo_ave"},  \$param->{"ParamGlobal::delo_std"});
	accumulate_averages($ttr,  \$param->{"ParamGlobal::ttro_ave"},  \$param->{"ParamGlobal::ttro_std"});
	accumulate_averages($apb,  \$param->{"ParamGlobal::apbo_ave"},  \$param->{"ParamGlobal::apbo_std"});

	fill_histo_array($like,   $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::likeo_ave"});
	fill_histo_array($like,   $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::likeo_ave"});
	fill_histo_array($like,   $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::likeo_ave"});
	fill_histo_array($like,   $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::likeo_ave"});
	fill_histo_array($likesq, $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::likeo_std"});
	fill_histo_array($likesq, $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::likeo_std"});
	fill_histo_array($likesq, $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::likeo_std"});
	fill_histo_array($likesq, $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::likeo_std"});

	fill_histo_array($alp,   $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::alpo_ave"});
	fill_histo_array($alp,   $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::alpo_ave"});
	fill_histo_array($alp,   $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::alpo_ave"});
	fill_histo_array($alp,   $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::alpo_ave"});
	fill_histo_array($alpsq, $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::alpo_std"});
	fill_histo_array($alpsq, $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::alpo_std"});
	fill_histo_array($alpsq, $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::alpo_std"});
	fill_histo_array($alpsq, $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::alpo_std"});

	fill_histo_array($bet,   $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::beto_ave"});
	fill_histo_array($bet,   $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::beto_ave"});
	fill_histo_array($bet,   $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::beto_ave"});
	fill_histo_array($bet,   $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::beto_ave"});
	fill_histo_array($betsq, $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::beto_std"});
	fill_histo_array($betsq, $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::beto_std"});
	fill_histo_array($betsq, $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::beto_std"});
	fill_histo_array($betsq, $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::beto_std"});

	fill_histo_array($ins,   $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::inso_ave"});
	fill_histo_array($ins,   $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::inso_ave"});
	fill_histo_array($ins,   $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::inso_ave"});
	fill_histo_array($ins,   $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::inso_ave"});
	fill_histo_array($inssq, $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::inso_std"});
	fill_histo_array($inssq, $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::inso_std"});
	fill_histo_array($inssq, $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::inso_std"});
	fill_histo_array($inssq, $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::inso_std"});

	fill_histo_array($del,   $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::delo_ave"});
	fill_histo_array($del,   $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::delo_ave"});
	fill_histo_array($del,   $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::delo_ave"});
	fill_histo_array($del,   $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::delo_ave"});
	fill_histo_array($delsq, $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::delo_std"});
	fill_histo_array($delsq, $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::delo_std"});
	fill_histo_array($delsq, $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::delo_std"});
	fill_histo_array($delsq, $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::delo_std"});

	fill_histo_array($ttr,   $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::ttro_ave"});
	fill_histo_array($ttr,   $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::ttro_ave"});
	fill_histo_array($ttr,   $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::ttro_ave"});
	fill_histo_array($ttr,   $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::ttro_ave"});
	fill_histo_array($ttrsq, $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::ttro_std"});
	fill_histo_array($ttrsq, $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::ttro_std"});
	fill_histo_array($ttrsq, $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::ttro_std"});
	fill_histo_array($ttrsq, $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::ttro_std"});

	fill_histo_array($apb,   $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::apbo_ave"});
	fill_histo_array($apb,   $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::apbo_ave"});
	fill_histo_array($apb,   $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::apbo_ave"});
	fill_histo_array($apb,   $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::apbo_ave"});
	fill_histo_array($apbsq, $indelfrq, $Npfo,     $kpfo,     $hpfo->{"ParamHisto::apbo_std"});
	fill_histo_array($apbsq, $msalen,   $Npmsalen, $kpmsalen, $hpmsalen->{"ParamHisto::apbo_std"});
	fill_histo_array($apbsq, $sqleng,   $Npsqleng, $kpsqleng, $hpsqleng->{"ParamHisto::apbo_std"});
	fill_histo_array($apbsq, $sqlena,   $Npsqlena, $kpsqlena, $hpsqlena->{"ParamHisto::apbo_std"});
    }

}

sub finishupcase { 
    my ($case_ref) = @_;

    # calculate stats for len
    calculate_averages(\$$case_ref->{"Case::meanmsalen"}, \$$case_ref->{"Case::stdmsalen"}, $$case_ref->{"Case::nmsa"});

    # calculate stats for indel/residue frequencies
    calculate_averages(\$$case_ref->{"Case::meanindelfreq"}, 
		       \$$case_ref->{"Case::stdindelfreq"}, $$case_ref->{"Case::nmsa"});
    calculate_averages(\$$case_ref->{"Case::meanresfreq"}[0], 
		       \$$case_ref->{"Case::stdresfreq"}[0], $$case_ref->{"Case::nmsa"});
    calculate_averages(\$$case_ref->{"Case::meanresfreq"}[1], 
		       \$$case_ref->{"Case::stdresfreq"}[1], $$case_ref->{"Case::nmsa"});
    calculate_averages(\$$case_ref->{"Case::meanresfreq"}[2], 
		       \$$case_ref->{"Case::stdresfreq"}[2], $$case_ref->{"Case::nmsa"});
    calculate_averages(\$$case_ref->{"Case::meanresfreq"}[3], 
		       \$$case_ref->{"Case::stdresfreq"}[3], $$case_ref->{"Case::nmsa"});

    #calculate stats for pairwise identities/mutations/indels
    my $npairwise = $$case_ref->{"Case::nmsa"}*$$case_ref->{"Case::sqnum"}*($$case_ref->{"Case::sqnum"}-1)/2;
    calculate_averages(\$$case_ref->{"Case::meanpairid"},   \$$case_ref->{"Case::stdpairid"},  $npairwise);
    calculate_averages(\$$case_ref->{"Case::meanpairmut"},  \$$case_ref->{"Case::stdpairmut"}, $npairwise);
    calculate_averages(\$$case_ref->{"Case::meanpairindl"}, \$$case_ref->{"Case::stdpairindl"}, $npairwise);

    print_case($$case_ref);

    #clean up
    if ($clean == 1) {
	system ("rm ".$$case_ref->{"Case::dnadefaultsfile"}."\n"); 

	system ("rm ".$$case_ref->{"Case::roseinputfile"}."\n"); 
    }

}

sub finishupallcases {
    my ($variable, $init, $case, $globals_ref, $c) = @_;
    
    # calculate global stats for indel/residue frequencies and  pairwise identities
    calculate_global_stats($globals_ref, $c, $case->nmsa, $case->sqnum);
 
    # write the final histogram
    my $iswc = 0;
    write_histograms($case, $globals_ref, $iswc);

    # remove the working copies of the histograms
    my $hfohisto     = $case->{"Case::hfo"};
    my $hmsalenhisto = $case->{"Case::hmsalen"};
    my $hsqlenghisto = $case->{"Case::hsqleng"};
    my $hsqlenahisto = $case->{"Case::hsqlena"};

    my $hpfohisto     = $case->{"Case::hpfo"};
    my $hpmsalenhisto = $case->{"Case::hpmsalen"};
    my $hpsqlenghisto = $case->{"Case::hpsqleng"};
    my $hpsqlenahisto = $case->{"Case::hpsqlena"};

    system ("rm ".$hfohisto->{"Histogram::histofilewc"}."\n"); 
    system ("rm ".$hmsalenhisto->{"Histogram::histofilewc"}."\n"); 
    system ("rm ".$hsqlenghisto->{"Histogram::histofilewc"}."\n"); 
    system ("rm ".$hsqlenahisto->{"Histogram::histofilewc"}."\n"); 

    system ("rm ".$hfohisto->{"Histogram::histofilewc"}.".ps\n"); 
    system ("rm ".$hmsalenhisto->{"Histogram::histofilewc"}.".ps\n"); 
    system ("rm ".$hsqlenghisto->{"Histogram::histofilewc"}.".ps\n"); 
    system ("rm ".$hsqlenahisto->{"Histogram::histofilewc"}.".ps\n"); 

    system ("rm ".$hpfohisto->{"ParamHisto::histofilewc"}."\n"); 
    system ("rm ".$hpmsalenhisto->{"ParamHisto::histofilewc"}."\n"); 
    system ("rm ".$hpsqlenghisto->{"ParamHisto::histofilewc"}."\n"); 
    system ("rm ".$hpsqlenahisto->{"ParamHisto::histofilewc"}."\n"); 

    system ("rm ".$hpfohisto->{"ParamHisto::histofilewc"}.".ps\n"); 
    system ("rm ".$hpmsalenhisto->{"ParamHisto::histofilewc"}.".ps\n"); 
    system ("rm ".$hpsqlenghisto->{"ParamHisto::histofilewc"}.".ps\n"); 
    system ("rm ".$hpsqlenahisto->{"ParamHisto::histofilewc"}.".ps\n"); 
}

sub finishuppartially {
    my ($variable, $init, $case, $globals_ref, $c) = @_;
 
    my $partialstats;
    setupglobals(\$partialstats);

    # copy global stats into partialstats
    copy_paramhisto(\$partialstats->{"Globalstats::param"}, $$globals_ref->{"Globalstats::param"});

    $partialstats->{"Globalstats::meanindelfreq"}  = $$globals_ref->{"Globalstats::meanindelfreq"};
    $partialstats->{"Globalstats::stdindelfreq"}   = $$globals_ref->{"Globalstats::stdindelfreq"};
    $partialstats->{"Globalstats::meanresfreq"}[0] = $$globals_ref->{"Globalstats::meanresfreq"}[0];
    $partialstats->{"Globalstats::stdresfreq"}[0]  = $$globals_ref->{"Globalstats::stdresfreq"}[0];
    $partialstats->{"Globalstats::meanresfreq"}[1] = $$globals_ref->{"Globalstats::meanresfreq"}[1];
    $partialstats->{"Globalstats::stdresfreq"}[1]  = $$globals_ref->{"Globalstats::stdresfreq"}[1];
    $partialstats->{"Globalstats::meanresfreq"}[2] = $$globals_ref->{"Globalstats::meanresfreq"}[2];
    $partialstats->{"Globalstats::stdresfreq"}[2]  = $$globals_ref->{"Globalstats::stdresfreq"}[2];
    $partialstats->{"Globalstats::meanresfreq"}[3] = $$globals_ref->{"Globalstats::meanresfreq"}[3];
    $partialstats->{"Globalstats::stdresfreq"}[3]  = $$globals_ref->{"Globalstats::stdresfreq"}[3];
    $partialstats->{"Globalstats::meanpairid"}     = $$globals_ref->{"Globalstats::meanpairid"};
    $partialstats->{"Globalstats::stdpairid"}      = $$globals_ref->{"Globalstats::stdpairid"};
    $partialstats->{"Globalstats::meanpairmut"}    = $$globals_ref->{"Globalstats::meanpairmut"};
    $partialstats->{"Globalstats::stdpairmut"}     = $$globals_ref->{"Globalstats::stdpairmut"};
    $partialstats->{"Globalstats::meanpairindl"}   = $$globals_ref->{"Globalstats::meanpairindl"};
    $partialstats->{"Globalstats::stdpairindl"}    = $$globals_ref->{"Globalstats::stdpairindl"};
    $partialstats->{"Globalstats::runtime_ave"}    = $$globals_ref->{"Globalstats::runtime_ave"};
    $partialstats->{"Globalstats::runtime_std"}    = $$globals_ref->{"Globalstats::runtime_std"};
    $partialstats->{"Globalstats::runtime_e_ave"}  = $$globals_ref->{"Globalstats::runtime_e_ave"};
    $partialstats->{"Globalstats::runtime_e_std"}  = $$globals_ref->{"Globalstats::runtime_e_std"};
    
    # calculate global stats for indel/residue frequencies and  pairwise identities
    calculate_global_stats(\$partialstats, $c, $case->nmsa, $case->sqnum);

    # write the working copy histogram
    # in case we want to check before the whole job finishes
    my $iswc = 1;
    write_histograms($case, \$partialstats, $iswc);
    
}

# the overal rate of substitutions per site = fracchange 
#
# 2.0* alpha * bb + beta * (1 - p_a*p_a- p_c*p_c- p_g*p_g- p_t*p_t)
#
sub fracchange
{
    my ($xi, $xv, $frqa, $frqc, $frqg, $frqt) = @_;
    
    my $fracchange;
    my $frqgr;
    my $frqty;
    my $tol = 0.0001;
    
    # Check that the stationary probabilities add up to one 
    
    if (abs($frqa + $frqc + $frqg + $frqt - 1.0) > $tol) {
	printf "A %f C %f G %f T %f tol %f %f\n", $frqa, $frqc, $frqg, $frqt, abs($frqa + $frqc + $frqg + $frqt - 1.0), $tol;
	print "fracchange(): bad frequencies\n"; die;
    }
    
    $frqgr = ($frqa + $frqg > 0.)? $frqg/($frqa + $frqg) : 0.0;
    $frqty = ($frqc + $frqt > 0.)? $frqt/($frqc + $frqt) : 0.0;
    
    # Average change of mutations 
    $fracchange = 
	+ $xi * (2.0 * $frqa * $frqgr + 2.0 * $frqc * $frqty) 
	+ $xv * (1.0 - $frqa * $frqa - $frqc * $frqc - $frqg * $frqg - $frqt * $frqt);
    
    
    if ($verbose) {
	printf "\nfracchange %f xi %f xv %f\n", $fracchange, $xi, $xv;
    }
    
    return $fracchange;
}

# the overal rate of substitutions and indels per site = fracchange 
#
# fracchange = (1- frqo) * (frachcange_mutations + del_rate) + frqo * ins_rate
#
sub fracchange_erate
{
    my ($xi, $xv, $xl, $xm, $frqa, $frqc, $frqg, $frqt, $freqo) = @_;
    
    my $fracchange;
    my $fracchange_mutations;
   
    # Average change of mutations 
    $fracchange_mutations = fracchange($xi, $xv, $frqa, $frqc, $frqg, $frqt);    
    
    # Average change including indels 
    $fracchange = (1.0 - $freqo) * ($fracchange_mutations + $xm) + $freqo * $xl;
    
    if ($verbose) {
	printf "\nfracchange %f mut %f fo %f xi %f xv %f xl %f xm %f\n", 
	$fracchange, $fracchange_mutations, $freqo, $xi, $xv, $xl, $xm;
    }
    
    return $fracchange;
}

sub get_stats_msa {
    my ($case_ref, $globals_ref, $phylipmsafile, 
	$meanpairid_ref,   $stdpairid_ref,
	$meanpairmut_ref,  $stdpairmut_ref, 
	$meanpairindl_ref, $stdpairindl_ref, 
	$msalen_ref, $seqlen_geomean_ref, $seqlen_arimean_ref, 
	$indelfreq_ref, $afreq_ref, $cfreq_ref, $gfreq_ref, $tfreq_ref) = @_;

    my $seqlen_geomean = 0.0;
    my $seqlen_arimean = 0.0;
    my $seqlen_geostd  = 0.0;
    my $seqlen_aristd  = 0.0;

    my $meanpairid   = 0.0;
    my $stdpairid    = 0.0;
    my $meanpairmut  = 0.0;
    my $stdpairmut   = 0.0;
    my $meanpairindl = 0.0;
    my $stdpairindl  = 0.0;
    my $indelfreq    = 0.0;
    my $afreq  = 0.0;
    my $cfreq  = 0.0;
    my $gfreq  = 0.0;
    my $tfreq  = 0.0;
    my $counts;

    my @msa;
    my $line;
    my $aseq;
    my $alen;
    my $n = 0;

    if ($verbose) {
	system("more $phylipmsafile\n");
    }

    open(MSA, $phylipmsafile);
    while(<MSA>) {
	if (/\s*(\d+)\s+(\d+)/) {
	    $aseq = $1;
	    $alen = $2;

	    #initialize
	    for (my $s = 0; $s < $aseq; $s ++) { $msa[$s] = ""; }
	}
	elsif (/^\S+\s+(.+)/) {
	    $line = $1; $line =~ s/ //g;
	    $msa[$n++] .= $line;
	}
	elsif (/^\s+$/) {
	    if ($n != $aseq) { print "bad alignment\n"; die; }
	    $n = 0;
	}
	elsif (/\s+(.+)/) {
	    $line = $1; $line =~ s/ //g;
	    $msa[$n++] .= $line;
	}
    }
    close (MSA);

    #check the alignment is complete
    for (my $s = 0; $s < $aseq; $s ++) { 
	if (length($msa[$s]) != $alen) { 
	    print "bad alignment alen=$alen real=", length($msa[$s]), "\n"; die;
	} 
    }

    # indel/residue statistics
    # indels use - only
    my $aseqg = 0;
    for (my $s = 0; $s < $aseq; $s ++) { 
	my $indel  = ($msa[$s] =~ tr/\-/\-/);
	my $acount = ($msa[$s] =~ tr/A/A/);
	my $ccount = ($msa[$s] =~ tr/C/C/);
	my $gcount = ($msa[$s] =~ tr/G/G/);
	my $tcount = ($msa[$s] =~ tr/T/T/);
 
	#calculate the geometric and arithmetic means of seqlen
	if ($indel > $alen) { print "bad sequence! alen = $alen indels = $indel\n"; die; }
	my $seqlen = $alen - $indel;
	accumulate_averages($seqlen, \$seqlen_arimean, \$seqlen_aristd);	    
	if ($seqlen > 0) { 
	    accumulate_averages(log($seqlen), \$seqlen_geomean, \$seqlen_geostd); 
	    $aseqg ++; 
	}	    

	$indelfreq += $indel; 

	$afreq += $acount; 
	$cfreq += $ccount; 
	$gfreq += $gcount; 
	$tfreq += $tcount; 

    }   

    $counts = $aseq * $alen;

    calculate_averages(\$seqlen_arimean, \$seqlen_aristd, $aseq);
    calculate_averages(\$seqlen_geomean, \$seqlen_geostd, $aseqg);
    $seqlen_geomean = exp($seqlen_geomean);

    if ($counts > 0) { 
	$indelfreq /= $counts; 
	$afreq     /= $counts;  
	$cfreq     /= $counts; 
	$gfreq     /= $counts; 
	$tfreq     /= $counts; 
    }

    if (1.0-$indelfreq > 0 ) {
	$afreq /= (1.0-$indelfreq);  
	$cfreq /= (1.0-$indelfreq); 
	$gfreq /= (1.0-$indelfreq); 
	$tfreq /= (1.0-$indelfreq); 
    }

    #accumulate lengths
    accumulate_averages($alen, \$$case_ref->{"Case::meanmsalen"}, \$$case_ref->{"Case::stdmsalen"});

    #add for this case
    accumulate_averages($indelfreq, \$$case_ref->{"Case::meanindelfreq"}, \$$case_ref->{"Case::stdindelfreq"});
    accumulate_averages($afreq, \$$case_ref->{"Case::meanresfreq"}[0], \$$case_ref->{"Case::stdresfreq"}[0]);
    accumulate_averages($cfreq, \$$case_ref->{"Case::meanresfreq"}[1], \$$case_ref->{"Case::stdresfreq"}[1]);
    accumulate_averages($gfreq, \$$case_ref->{"Case::meanresfreq"}[2], \$$case_ref->{"Case::stdresfreq"}[2]);
    accumulate_averages($tfreq, \$$case_ref->{"Case::meanresfreq"}[3], \$$case_ref->{"Case::stdresfreq"}[3]);

    # add for global stats
    accumulate_averages($indelfreq, \$$globals_ref->{"Globalstats::meanindelfreq"}, 
			\$$globals_ref->{"Globalstats::stdindelfreq"});
    accumulate_averages($afreq, \$$globals_ref->{"Globalstats::meanresfreq"}[0], 
			\$$globals_ref->{"Globalstats::stdresfreq"}[0]);
    accumulate_averages($cfreq, \$$globals_ref->{"Globalstats::meanresfreq"}[1], 
			\$$globals_ref->{"Globalstats::stdresfreq"}[1]);
    accumulate_averages($gfreq, \$$globals_ref->{"Globalstats::meanresfreq"}[2], 
			\$$globals_ref->{"Globalstats::stdresfreq"}[2]);
    accumulate_averages($tfreq, \$$globals_ref->{"Globalstats::meanresfreq"}[3], 
			\$$globals_ref->{"Globalstats::stdresfreq"}[3]);

    # pairwise identity statistics
    my $ncases = 0;
    for (my $s1 = 0; $s1 < $aseq; $s1 ++) { 
	for (my $s2 = $s1+1; $s2 < $aseq; $s2 ++) {
	    my $len = $alen;
	    my $pairid   = 0.0;
	    my $pairmut  = 0.0;
	    my $pairindl = 0.0;
	    $ncases ++;

	    for (my $x = 0.; $x < $alen; $x ++) {
		my $a = substr($msa[$s1], $x, 1);
		my $b = substr($msa[$s2], $x, 1);
		if ($a eq $b) {
		    if ($a eq "-") { $len --;    }
		    else           { $pairid ++; }
		}
		else {
		    if ($a eq "-" || $b eq "-") { $pairindl ++; }
		    else                        { $pairmut ++;  }
		}
	    }
	    if ($len > 0) { 
		$pairid   /= 0.01*$len; 
		$pairmut  /= 0.01*$len; 
		$pairindl /= 0.01*$len; 
	    }
	    accumulate_averages($pairid,   \$meanpairid,   \$stdpairid);	    
	    accumulate_averages($pairmut,  \$meanpairmut,  \$stdpairmut);	    
	    accumulate_averages($pairindl, \$meanpairindl, \$stdpairindl);	    

	    accumulate_averages($pairid,   \$$case_ref->{"Case::meanpairid"},   \$$case_ref->{"Case::stdpairid"});
	    accumulate_averages($pairmut,  \$$case_ref->{"Case::meanpairmut"},  \$$case_ref->{"Case::stdpairmut"});
	    accumulate_averages($pairindl, \$$case_ref->{"Case::meanpairindl"}, \$$case_ref->{"Case::stdpairindl"});

	    accumulate_averages($pairid,   \$$globals_ref->{"Globalstats::meanpairid"},  
				\$$globals_ref->{"Globalstats::stdpairid"});
	    accumulate_averages($pairmut,  \$$globals_ref->{"Globalstats::meanpairmut"},  
				\$$globals_ref->{"Globalstats::stdpairmut"});
	    accumulate_averages($pairindl, \$$globals_ref->{"Globalstats::meanpairindl"}, 
				\$$globals_ref->{"Globalstats::stdpairindl"});
	}  
    }
    calculate_averages(\$meanpairid,   \$stdpairid,   $ncases);
    calculate_averages(\$meanpairmut,  \$stdpairmut,  $ncases);
    calculate_averages(\$meanpairindl, \$stdpairindl, $ncases);

    $$seqlen_geomean_ref = $seqlen_geomean;
    $$seqlen_arimean_ref = $seqlen_arimean;
    $$meanpairid_ref     = $meanpairid;
    $$stdpairid_ref      = $stdpairid;
    $$meanpairmut_ref    = $meanpairmut;
    $$stdpairmut_ref     = $stdpairmut;
    $$meanpairindl_ref   = $meanpairindl;
    $$stdpairindl_ref    = $stdpairindl;
    $$msalen_ref         = $alen;
    $$indelfreq_ref      = $indelfreq;
    $$afreq_ref          = $afreq;
    $$cfreq_ref          = $cfreq;
    $$gfreq_ref          = $gfreq;
    $$tfreq_ref          = $tfreq;
}


sub gnuplot_histo {

    my ($filehisto, $title, $xlabel, $key) = @_;

    my $outplot = "$filehisto.ps";
    my $ylabel;

    open(GP,'|'.GNUPLOT) || die "Gnuplot: $!";

    print GP "set terminal postscript color solid 14\n";
    print GP "set style line 1 lt 1 lw 4\n";
    print GP "set style line 2 lt 2 lw 4\n";
    print GP "set style line 3  lt 3 lw 4\n";
    print GP "set style line 4 lt 4 lw 4\n";+
    print GP "set style line 5 lt 5 lw 4\n";
    print GP "set style line 6 lt 6 lw 4\n";
    print GP "set style line 7 lt 7 lw 4\n";
    print GP "set style line 8 lt 8 lw 4\n";
    print GP "set style line 9 lt 9 lw 4\n";
    print GP "set style line 11 lt 1 lw 2 pt 1 ps 1.2\n";
    print GP "set style line 22 lt 2 lw 2 pt 2 ps 1.2\n";
    print GP "set style line 33 lt 3 lw 2 pt 3 ps 1.2\n";
    print GP "set style line 44 lt 4 lw 2 pt 4 ps 1.2\n";
    print GP "set style line 55 lt 5 lw 2 pt 5 ps 1.2\n";
    print GP "set style line 66 lt 6 lw 2 pt 6 ps 1.2\n";
    print GP "set style line 77 lt 7 lw 2 pt 7 ps 1.2\n";
    print GP "set style line 88 lt 8 lw 2 pt 8 ps 1.2\n";
    print GP "set style line 99 lt 9 lw 2 pt 9 ps 1.2\n";

    print GP "set output '$outplot'\n";
    #print GP "set nokey\n";
    print GP "set xlabel '$xlabel'\n";

    print GP "set title \"$title\\n\\n$key\"\n";
    my $newkeyx = "dnaml";
    my $newkeye = "dnaml-erate";
    my $newkeyo = "dnaml-erate4";

    if ($opt_b || $opt_o) {
	$ylabel = "fraction TRUE POSITIVES";
	print GP "set ylabel '$ylabel'\n";
	print GP "plot '$filehisto' using 1:4 title '$newkeyx' ls 3, '$filehisto' using 1:13 title '$newkeye' ls 2\n";

	$ylabel = "TREE DISTANCE";
	print GP "set ylabel '$ylabel'\n";
	print GP "plot '$filehisto' using 1:5:6 with yerrorbars title '$newkeyx' ls 3, '$filehisto' using 1:14:15  with yerrorbars title '$newkeye' ls 2\n";

	$ylabel = "RUN TIME (sec)";
	print GP "set ylabel '$ylabel'\n";
	print GP "plot '$filehisto' using 1:9:10 with yerrorbars title '$newkeyx' ls 3, '$filehisto' using 1:18:19  with yerrorbars title '$newkeye' ls 2\n";
    
	if ($opt_o) {
	    $ylabel = "fraction TRUE POSITIVES";
	    print GP "set ylabel '$ylabel'\n";
	    print GP "plot '$filehisto' using 1:4 title '$newkeyx' ls 3, '$filehisto' using 1:13 title '$newkeye' ls 2, '$filehisto' using 1:22 title '$newkeyo' ls 1\n";
	    
	    $ylabel = "DISTANCE";
	    print GP "set ylabel '$ylabel'\n";
	    print GP "plot '$filehisto' using 1:5:6 with yerrorbars title '$newkeyx' ls 3, '$filehisto' using 1:14:15  with yerrorbars title '$newkeye' ls 2, '$filehisto' using 1:23:24  with yerrorbars title '$newkeyo' ls 1\n";
	    
	    $ylabel = "RUN TIME (sec)";
	    print GP "set ylabel '$ylabel'\n";
	    print GP "plot '$filehisto' using 1:9:10 with yerrorbars title '$newkeyx' ls 3, '$filehisto' using 1:18:19  with yerrorbars title '$newkeye' ls 2, '$filehisto' using 1:27:28  with yerrorbars title '$newkeyo' ls 1\n";
	}
    }
    else {
	my $singlekey = "dnaml";
	if ($opt_e) { $singlekey .= "-erate"; }

	$ylabel = "fraction TRUE POSITIVES";
	print GP "set ylabel '$ylabel'\n";
	print GP "plot '$filehisto' using 1:4 title '$singlekey' ls 3\n";

	$ylabel = "DISTANCE";
	print GP "set ylabel '$ylabel'\n";
	print GP "plot '$filehisto' using 1:5:6  with yerrorbars title '$singlekey' ls 3\n";

	$ylabel = "RUN TIME (sec)";
	print GP "set ylabel '$ylabel'\n";
	print GP "plot '$filehisto' using 1:9:10  with yerrorbars title '$singlekey' ls 3\n";
    }
    
    close (GP);

    if ($seeplots) { system ("$evince $outplot&\n"); }
}

sub gnuplot_paramhisto {

    my ($filehisto, $title, $xlabel, $key) = @_;

    my $outplot = "$filehisto.ps";
    my $ylabel;

    open(GP,'|'.GNUPLOT) || die "Gnuplot: $!";

    print GP "set terminal postscript color solid 14\n";
    print GP "set style line 1 lt 1 lw 4\n";
    print GP "set style line 2 lt 2 lw 4\n";
    print GP "set style line 3  lt 3 lw 4\n";
    print GP "set style line 4 lt 4 lw 4\n";
    print GP "set style line 5 lt 5 lw 4\n";
    print GP "set style line 6 lt 6 lw 4\n";
    print GP "set style line 7 lt 7 lw 4\n";
    print GP "set style line 8 lt 8 lw 4\n";
    print GP "set style line 9 lt 9 lw 4\n";
    print GP "set style line 11 lt 1 lw 2 pt 1 ps 1.2\n";
    print GP "set style line 22 lt 2 lw 2 pt 2 ps 1.2\n";
    print GP "set style line 33 lt 3 lw 2 pt 3 ps 1.2\n";
    print GP "set style line 44 lt 4 lw 2 pt 4 ps 1.2\n";
    print GP "set style line 55 lt 5 lw 2 pt 5 ps 1.2\n";
    print GP "set style line 66 lt 6 lw 2 pt 6 ps 1.2\n";
    print GP "set style line 77 lt 7 lw 2 pt 7 ps 1.2\n";
    print GP "set style line 88 lt 8 lw 2 pt 8 ps 1.2\n";
    print GP "set style line 99 lt 9 lw 2 pt 9 ps 1.2\n";

    print GP "set output '$outplot'\n";
    #print GP "set nokey\n";
    print GP "set xlabel '$xlabel'\n";

    print GP "set title \"$title\\n\\n$key\"\n";
    my $newkeyx = "dnaml";
    my $newkeye = "dnaml-erate";
    my $newkeyo = "dnaml-erate4";

    if ($opt_b || $opt_o) {
	$ylabel = "LN LIKELIHOOD";
	print GP "set ylabel '$ylabel'\n";
	print GP "plot '$filehisto' using 1:3:4  with yerrorbars title '$newkeyx' ls 1, '$filehisto' using 1:20:21  with yerrorbars title '$newkeye' ls 2\n";

	$ylabel = "AVERAGE BRANCH LENGTH";
	print GP "set ylabel '$ylabel'\n";
	print GP "plot '$filehisto' using 1:5:6  with yerrorbars title '$newkeyx' ls 1, '$filehisto' using 1:22:23  with yerrorbars title '$newkeye' ls 2\n";

	$ylabel = "RATE PARAMETERS ** $newkeye";
	print GP "set ylabel '$ylabel'\n";
	print GP "plot '$filehisto' using 1:24:25  with yerrorbars title 'ALPHA' ls 1, '$filehisto' using 1:26:27  with yerrorbars title 'BETA' ls 2, '$filehisto' using 1:28:29  with yerrorbars title 'INS' ls 3, '$filehisto' using 1:30:31  with yerrorbars title 'DEL' ls 4\n";

	print GP "plot '$filehisto' using 1:32:33  with yerrorbars title 'TTR' ls 1, '$filehisto' using 1:34:35  with yerrorbars title 'ALPHA + BETA' ls 2\n";

    }
    if ($opt_o) {
	
	$ylabel = "LN LIKELIHOOD";
	print GP "set ylabel '$ylabel'\n";
	print GP "plot '$filehisto' using 1:3:4  with yerrorbars title '$newkeyx' ls 1, '$filehisto' using 1:20:21  with yerrorbars title '$newkeye' ls 2, '$filehisto' using 1:37:38  with yerrorbars title '$newkeyo' ls 3\n";

	$ylabel = "AVERAGE BRANCH LENGTH";
	print GP "set ylabel '$ylabel'\n";
	print GP "plot '$filehisto' using 1:5:6  with yerrorbars title '$newkeyx' ls 1, '$filehisto' using 1:22:23  with yerrorbars title '$newkeye' ls 2, '$filehisto' using 1:39:40  with yerrorbars title '$newkeyo' ls 3\n";

	$ylabel = "RATE PARAMETERS ** $newkeyo";
	print GP "set ylabel '$ylabel'\n";
	print GP "plot '$filehisto' using 1:41:42  with yerrorbars title 'ALPHA' ls 1, '$filehisto' using 1:43:44  with yerrorbars title 'BETA' ls 2, '$filehisto' using 1:45:46  with yerrorbars title 'INS' ls 3, '$filehisto' using 1:47:48  with yerrorbars title 'DEL' ls 4\n";

	print GP "plot '$filehisto' using 1:49:50  with yerrorbars title 'TTR' ls 1, '$filehisto' using 1:51:52  with yerrorbars title 'ALPHA + BETA' ls 2\n";
    }
    
    if ($opt_e) { 
	my $singlekey .= "dnaml-erate"; 
	
	$ylabel = "LN LIKELIHOOD";
	print GP "set ylabel '$ylabel'\n";
	print GP "plot '$filehisto' using 1:3:4  with yerrorbars title '$singlekey' ls 1\n";

	$ylabel = "AVERAGE BRANCH LENGTH";
	print GP "set ylabel '$ylabel'\n";
	print GP "plot '$filehisto' using 1:5:6  with yerrorbars title '$singlekey' ls 1\n";

	$ylabel = "RATE PARAMETERS ** $singlekey";
	print GP "set ylabel '$ylabel'\n";
	print GP "plot '$filehisto' using 1:7:8  with yerrorbars title 'ALPHA' ls 1, '$filehisto' using 1:9:10  with yerrorbars title 'BETA' ls 2, '$filehisto' using 1:11:12  with yerrorbars title 'INS' ls 3, '$filehisto' using 1:13:14  with yerrorbars title 'DEL' ls 4\n";

	print GP "plot '$filehisto' using 1:15:16  with yerrorbars title 'TTR' ls 1, '$filehisto' using 1:17:18  with yerrorbars title 'ALPHA + BETA' ls 2\n";
    }
    
    close (GP);
    
    if ($seeplots) { system ("$evince $outplot&\n"); }
}

sub header_for_testrun {
    my ($case_ref, $ntest, 
	$meanpairid,   $stdpairid, 
	$meanpairmut,  $stdpairmut, 
	$meanpairindl, $stdpairindl, 
	$msalen, $indelfreq, $afreq, $cfreq, $gfreq, $tfreq) = @_;

    open(OUT, ">>".$$case_ref->{"Case::outfile"});
    printf OUT "\nCASE: %s\t(ntest=%d)\n", $$case_ref->{"Case::name"}, $$case_ref->{"Case::nmsa"};
    printf OUT "TESTRUN=%d\n", $ntest+1;
    printf OUT "mean_pairwise_id = %f +/- %f\n", $meanpairid, $stdpairid;
    printf OUT "mean_pairwise_mut = %f +/- %f\n", $meanpairmut, $stdpairmut;
    printf OUT "mean_pairwise_indl = %f +/- %f\n", $meanpairindl, $stdpairindl;
    printf OUT "msalen = %f\n", $msalen;
    printf OUT "indelfreq = %f\n", $indelfreq;
    printf OUT "residuefreq = %f %f %f %f\n", $afreq, $cfreq, $gfreq, $tfreq;
    close(OUT);
}

sub init_histogram {
    my ($N, $k, $tot_ref, $tp_ref, 
	$dist1_ave_ref, $dist1_std_ref, 
	$dist2_ave_ref, $dist2_std_ref, 
	$runtm_ave_ref, $runtm_std_ref) = @_;
    
    init_histo_array($N, $k, $tot_ref);
    init_histo_array($N, $k, $tp_ref);
    init_histo_array($N, $k, $dist1_ave_ref);
    init_histo_array($N, $k, $dist1_std_ref);
    init_histo_array($N, $k, $dist2_ave_ref);
    init_histo_array($N, $k, $dist2_std_ref);
    init_histo_array($N, $k, $runtm_ave_ref);
    init_histo_array($N, $k, $runtm_std_ref);
}
    
sub init_comp_histogram {
    my ($N, $k, $tot_ref, $tp_ref, 
	$dist1_ave_ref, $dist1_std_ref, 
	$dist2_ave_ref, $dist2_std_ref) = @_;
    
    init_histo_array($N, $k, $tot_ref);
    init_histo_array($N, $k, $tp_ref);
    init_histo_array($N, $k, $dist1_ave_ref);
    init_histo_array($N, $k, $dist1_std_ref);
    init_histo_array($N, $k, $dist2_ave_ref);
    init_histo_array($N, $k, $dist2_std_ref);
}
    
sub init_paramhisto {
    my ($N, $k, $tot_ref, 
	$like_ave_ref, $like_std_ref, $abl_ave_ref, $abl_std_ref, 
	$alp_ave_ref,  $alp_std_ref,  $bet_ave_ref, $bet_std_ref, 
	$ins_ave_ref,  $ins_std_ref,  $del_ave_ref, $del_std_ref, 
	$ttr_ave_ref,  $ttr_std_ref,  $apb_ave_ref, $apb_std_ref) = @_;

    init_histo_array($N, $k, $tot_ref);
    init_histo_array($N, $k, $like_ave_ref);
    init_histo_array($N, $k, $like_std_ref);
    init_histo_array($N, $k, $abl_ave_ref);
    init_histo_array($N, $k, $abl_std_ref);
    init_histo_array($N, $k, $alp_ave_ref);
    init_histo_array($N, $k, $alp_std_ref);
    init_histo_array($N, $k, $bet_ave_ref);
    init_histo_array($N, $k, $bet_std_ref);
    init_histo_array($N, $k, $ins_ave_ref);
    init_histo_array($N, $k, $ins_std_ref);
    init_histo_array($N, $k, $del_ave_ref);
    init_histo_array($N, $k, $del_std_ref);
    init_histo_array($N, $k, $ttr_ave_ref);
    init_histo_array($N, $k, $ttr_std_ref);
    init_histo_array($N, $k, $apb_ave_ref);
    init_histo_array($N, $k, $apb_std_ref);

}

# initialize a histogram array
sub init_histo_array {
    my ($N, $k, $his_ref) = @_;
    my $dim = $N * $k;
    for (my $i=0; $i<=$dim; $i++) { $his_ref->[$i] = 0; }    
}

sub make_rose_dna_defaults{
    my ($case) = @_;

    my $insfunc = create_TheInsFunc($case->outfile, $case->indeldist, $case->maxindel, $case->lambda, $case->c_over_t);
    my $delfunc = create_TheInsFunc($case->outfile, $case->indeldist, $case->maxindel, $case->lambda, $case->c_over_t);

    open(FILE, ">".$case->{"Case::dnadefaultsfile"});

    print FILE "\#\n\n";

    print FILE "\# rose include file\n";
    print FILE "\#\n\n";

    print FILE "InputType = 4 \/\/ DNA\n";
    print FILE "TheAlphabet = \"ACGT\"\n";
    print FILE "TheFreq = [", ${$case->freq}[0], ",", ${$case->freq}[1], ",",${$case->freq}[2], ",",${$case->freq}[3], "]\n";
    
    print FILE "TheDeleteThreshold = ", $case->deletionper, "\n";
    print FILE "TheInsertThreshold = ", $case->insertionper, "\n";
    if ($case->simcoding == 1) { print FILE "SimCoding = True\n"; }
    
    print FILE "TheInsFunc = $insfunc\n";
    print FILE "TheDelFunc = $delfunc\n";
    
    print FILE "TheDNAmodel = \"", $case->model, "\"\n";
    print FILE "MeanSubstitution = ", $case->rosemeansubs, " \/\/ this correspond to alpha+beta = 1 as in phylip\n";
    print FILE "TTratio = ", $case->rosettratio, " \/\/ this corresponds to phylip 2.0\n";
    
    close (FILE);

    if ($verbose) { system("more ".$case->{"Case::dnadefaultsfile"}."\n"); }
}

sub make_rose_input_file{
    my ($case) = @_;

    open(FILE, ">".$case->{"Case::roseinputfile"});

    print FILE "\# rose sample ", $case->{"Case::name"}, "\n\n";
    print FILE "\n";
    print FILE "\%include ", $case->{"Case::dnadefaultsfile"}, "\n";
    print FILE "\n";

    print FILE "SequenceLen = ", $case->sqlen, "\n";
    print FILE "SequenceNum = ", $case->sqnum, "\n";
    # since rose is using the unnormalized rate, 
    # we also give it the unnormalized tree
    print FILE "TheTree = ", $case->treeunh, "\n";
    print FILE "ChooseFromLeaves = True\n";

    close (FILE);

    if ($verbose) { system("more ".$case->{"Case::roseinputfile"}."\n"); }
}
  
# a symmetric 8-taxon tree
sub nhtree_eightaxa {
    my ($time) = @_;

    my $halftime = $time/2.0;
    my $doubletime = 2.0*$time;

    decimal(\$time);
    decimal(\$halftime);
    decimal(\$doubletime);

    my $treenh = "(ta0:$halftime,(ta1:$time,((ta2:$time,ta3:$time):$time,((ta4:$time,ta5:$time):$time,(ta6:$time,ta7:$time):$time):$doubletime):$time):$halftime);";

    return $treenh;
}
sub nhtree_eightaxa_binary {
    my ($time) = @_;

    my $doubletime = 2.0*$time;

    decimal(\$time);
    decimal(\$doubletime);

    my $treenh = "(((ta0:$time,ta1:$time):$time,(ta2:$time,ta3:$time):$time):$time,((ta4:$time,ta5:$time):$time,(ta6:$time,ta7:$time):$time):$time);";

    return $treenh;
}

# a 4-taxon tree with 2 different branch lengths
sub nhtree_fourtaxa {
    my ($twobranchratio, $abl, $treenh_ref, $abl_binary, $treenh_binary_ref) = @_;

    my $rand;

    # the abl given  here corresponds to the rooted tree
    # that is why we multiply by 6 not by 5.

    my $ta   = 6.0*$abl / (3.0+2.0*$twobranchratio); 
    my $ta_b = 6.0*$abl_binary / (3.0+2.0*$twobranchratio); 
    my $tb   = $twobranchratio * $ta;
    my $tb_b = $twobranchratio * $ta_b;
    my $halfta_b = $ta_b/2.0;

    decimal(\$ta);
    decimal(\$ta_b);
    decimal(\$tb);
    decimal(\$tb_b);
    decimal(\$halfta_b);

    #randomize the position in the tree
    my @time;
    my @time_b;
    $rand = rand(); if ($rand < 0 || $rand > 1) { print "bad randomization\n"; die; }
    if ($rand < 0.5) { $time[0] = $ta; $time_b[0] = $ta_b; $time[1] = $tb; $time_b[1] = $tb_b; }
    else             { $time[0] = $tb; $time_b[0] = $tb_b; $time[1] = $ta; $time_b[1] = $ta_b; }

    $rand = rand(); if ($rand < 0 || $rand > 1) { print "bad randomization\n"; die; }
    if ($rand < 0.5) { $time[2] = $ta; $time_b[2] = $ta_b; $time[3] = $tb; $time_b[3] = $tb_b; }
    else             { $time[2] = $tb; $time_b[2] = $tb_b; $time[3] = $ta; $time_b[3] = $ta_b; }

    $$treenh_ref        = "(ta0:$time[0],ta1:$time[1],(ta2:$time[2],ta3:$time[3]):$ta);";
    $$treenh_binary_ref = "((ta0:$time_b[0],ta1:$time_b[1]):$halfta_b,(ta2:$time_b[2],ta3:$time_b[3]):$halfta_b);";
}

sub nh2tree {
    my ($hash_ref, $nhtree, $tree_ref, $doesnotparse_ref) = @_;

    my $tree         = $$tree_ref;
    my $doesnotparse = 0;

    my $N;                # number of ',' in the nh notation plus one
    my $nodestart;        # 
    my $dim;              # number of internal nodes 
    my $ncd = 0;          # counter for number of nodes+leaves in the tree 
    my $nsq = 0;          # counter for number of leaves in the tree 
    my $nnd = 0;          # counter for number of internal nodes
 
    my @stack;
    my @blen;
    my $name;
    my $time;
    my $node;
    my $lnode;
    my $rnode;
    my $pnode;
    my $otaxon;

    # number of leaves of the tree 
    $N = ( $nhtree =~ tr/,/,/ ) + 1;
    $tree->{"Tree::ntaxa"} = $N;

    # if the number of parenthesis+1 = number of leaves  -->  tree is of the form (a,b)   --> start with node N   (root)
    # if the number of parenthesis+2 = number of leaves  -->  tree is of the form (a,b,c) --> start with node N+1 (next to root)
    if    ($N == ( $nhtree =~ tr/\(/\(/ )+1) { $nodestart = $N; }
    elsif ($N == ( $nhtree =~ tr/\(/\(/ )+2) { $nodestart = $N+1; }

    if ($N == 1) { $dim = $N;     }
    else         { $dim = $N - 1; }

    my $string = $nhtree;

    # Counter is:
    #  0...N-1  leaves
    #  N...2N-2 nodes (N is the root)
    #
    # in the structure I follow easels' conventions
    #
    #  0,1,...,N-1     leaves (in taxaparent)
    #  0,-1,...,-(N-1) leaves 
    #  0,...,N-2       nodes (0 is the root)
    
    while($string) {
	if (0) {
	    print "STRING=$string\n";
	    print "stack=@stack\n\n";
	}

	if ($string =~ /^\((.+)/) {
	    $string = $1;
	    # push node on stack 
	    push(@stack, $nodestart+$nnd);
	    $nnd ++;
	    $ncd ++;	
	}
	elsif ($string =~ /^([^\(\)\,\:]+)\:(.+)/) {
	    $name = $1;
	    $string = $2;
	    # push leaf on stack 
	    push(@stack, $hash_ref->{$name});
 
	    $nsq ++;
            $ncd ++;
  	}
	elsif ($string =~ /^([\d\.e\-]+)(.+)/) {
	    $time   = $1;
	    decimal(\$time);
	    $string = $2;
	    # add branch length for node and put back on the stack 
	    $node = pop(@stack);
	    $blen[$node] = $time;
	    push(@stack, $node);
 	}
	elsif ($string =~ /^\)\;$/) {
	    # end case;
	    # have to deal differently whether the tree is of the general form (a,b) or (a,b,c)
	   
	    my $len = $#stack+1;
	    if ($len == 2) { # tree with one leaf only
		if ($N != 1) { print "this tree should have one taxon only\n"; die; }
		$lnode = pop(@stack);
		$pnode = pop(@stack);
		${$tree->left}[$pnode-$N]  = ($lnode < $N)? $lnode : $lnode-$N;    # set left  node of root
		${$tree->ld}[$pnode-$N] = $blen[$lnode];                           # set left node branch length
		if ($lnode >= $N) { ${$tree->parent}[$lnode-$N] = $pnode-$N;     } # if a internal node, add the parent
		if ($lnode <  $N) { ${$tree->taxaparent}[$lnode] = $pnode-$N; } # if a leaf, fill array taxaparent
	    }

	    elsif ($len == 3) { # tree of the form (a,b)
		$rnode = pop(@stack);
		$lnode = pop(@stack);
		$pnode = pop(@stack);
		${$tree->left}[$pnode-$N]  = ($lnode < $N)? -$lnode : $lnode-$N;   # set left  node of root
		${$tree->right}[$pnode-$N] = ($rnode < $N)? -$rnode : $rnode-$N;   # set right node of root
		${$tree->ld}[$pnode-$N] = $blen[$lnode];                           # set left  node branch length
		${$tree->rd}[$pnode-$N] = $blen[$rnode];                           # set right node branch length
		if ($lnode >= $N) { ${$tree->parent}[$lnode-$N] = $pnode-$N;     } # if left  is a internal node, add the parent
		if ($rnode >= $N) { ${$tree->parent}[$rnode-$N] = $pnode-$N;     } # if rigth is a internal node, add the parent
		if ($lnode <  $N) { ${$tree->taxaparent}[$lnode] = $pnode-$N; } # if left  is a leaf, fill array taxaparent
		if ($rnode <  $N) { ${$tree->taxaparent}[$rnode] = $pnode-$N; } # if rigth is a leaf, fill array taxaparent
	    }

	    elsif ($len == 4) { # tree of the form (a:ta,b:tb,c:tc) --> ((a:ta,b:tb):tc/2,c:tc/2)
		$rnode  = pop(@stack);
		$lnode  = pop(@stack);
		$otaxon = pop(@stack); if ($otaxon != 0) { print "bad zero taxon $otaxon\n"; die; }
		$pnode  = pop(@stack);
		${$tree->left}[$pnode-$N]  = ($lnode < $N)? -$lnode : $lnode-$N;   # set left  node of root
		${$tree->right}[$pnode-$N] = ($rnode < $N)? -$rnode : $rnode-$N;   # set right node of root
		${$tree->ld}[$pnode-$N] = $blen[$lnode];                           # set left  node branch length
		${$tree->rd}[$pnode-$N] = $blen[$rnode];                           # set right node branch length
		if ($lnode >= $N) { ${$tree->parent}[$lnode-$N] = $pnode-$N;     } # if left  is a internal node, add the parent
		if ($rnode >= $N) { ${$tree->parent}[$rnode-$N] = $pnode-$N;     } # if rigth is a internal node, add the parent
		if ($lnode <  $N) { ${$tree->taxaparent}[$lnode] = $pnode-$N; } # if left  is a leaf, fill array taxaparent
		if ($rnode <  $N) { ${$tree->taxaparent}[$rnode] = $pnode-$N; } # if rigth is a leaf, fill array taxaparent

		# now add the last node, the root
		$nnd ++;
		$ncd ++;	
		$lnode = $otaxon; # the convenction is to put the 0 taxon to the left of the root
		$rnode = $pnode;
		$pnode = $pnode-1; if ($pnode != $N) { print "bad root node=$node\n"; die; }
		${$tree->left}[$pnode-$N]  = ($lnode < $N)? -$lnode : $lnode-$N;   # set left  node of root
		${$tree->right}[$pnode-$N] = ($rnode < $N)? -$rnode : $rnode-$N;   # set right node of root
		${$tree->ld}[$pnode-$N] = $blen[$lnode]/2.0;                       # set left  node branch length
		${$tree->rd}[$pnode-$N] = $blen[$lnode]/2.0;                       # set right node branch length
		if ($lnode >= $N) { ${$tree->parent}[$lnode-$N] = $pnode-$N;     } # if left  is a internal node, add the parent
		if ($rnode >= $N) { ${$tree->parent}[$rnode-$N] = $pnode-$N;     } # if rigth is a internal node, add the parent
		if ($lnode <  $N) { ${$tree->taxaparent}[$lnode] = $pnode-$N; } # if left  is a leaf, fill array taxaparent
		if ($rnode <  $N) { ${$tree->taxaparent}[$rnode] = $pnode-$N; } # if rigth is a leaf, fill array taxaparent
		
	    }
	    
	    else { print "nh2tree not parse right\n"; $doesnotparse = 1;}
	    
	    # set parent for root to itself
	    if ($pnode != $N) { print "bad root node=$node\n"; die; }
	    ${$tree->parent}[$pnode-$N] = $pnode-$N;
	    
 	    undef($string);

	}
	elsif ($string =~ /^\)(.+)/) {
            #create a node 
	    $string = $1;
	    $rnode = pop(@stack);
	    $lnode = pop(@stack);
	    $pnode = pop(@stack);
	    
	    if ($pnode < $N) { print "bad tree pnode=$pnode\n"; die; }
	    
	    # if a internal node, add the parent
	    if ($lnode >= $N) { ${$tree->parent}[$lnode-$N] = $pnode-$N; }
	    if ($rnode >= $N) { ${$tree->parent}[$rnode-$N] = $pnode-$N; }
	    
	    # if a leaf, fill array taxaparent
	    if ($lnode < $N) { ${$tree->taxaparent}[$lnode] = $pnode-$N; }
	    if ($rnode < $N) { ${$tree->taxaparent}[$rnode] = $pnode-$N; }

            # identify left and right nodes
	    if ($lnode <  $N) { ${$tree->left}[$pnode-$N]  = -$lnode;   }
	    if ($rnode <  $N) { ${$tree->right}[$pnode-$N] = -$rnode;   }
	    if ($lnode >= $N) { ${$tree->left}[$pnode-$N]  = $lnode-$N; }
	    if ($rnode >= $N) { ${$tree->right}[$pnode-$N] = $rnode-$N; }
	    
	    # branch lengths of left and right nodes.
	    ${$tree->ld}[$pnode-$N] = $blen[$lnode];
	    ${$tree->rd}[$pnode-$N] = $blen[$rnode]; 
	    
	    # put node back parent node in the stack 
	    push(@stack, $pnode);
 	}
	elsif ($string =~ /^\,(.+)/) {
	    $string = $1;
	}
	elsif ($string =~ /^\:(.+)/) {
	    $string = $1;
	}
	else {
	    print "bad tree parsing sring=$string\n"; die;
	}
	
    }

    if ($nsq != $N)     { print "tree reading failed. N $N nsq $nsq\n"; die; }
    if ($nnd != $dim)   { print "tree reading failed. dim $dim nintnodes $nnd\n"; die; }
    if ($ncd != $N+$dim){ print "tree reading failed. nnodes ", $N+$dim, " found $ncd\n"; die; }

    # reorder the nodes to be able to compare with
    # the original tree
    tree_renumber_nodes(\$tree);

    if ($verbose) {     
	printf "\nnhtree %s\n", $nhtree;
	print_tree($tree); 
    }

    $$tree_ref         = $tree;
    $$doesnotparse_ref = $doesnotparse;
}

sub parse_phylipscreenout {

    my ($phylipscreenoutfile) = @_;

    my $runtime = -1;

    if ($verbose) { system("more $phylipscreenoutfile\n"); }

    open(FILE, $phylipscreenoutfile);
    while(<FILE>) {
	if (/^real\s+(\S+)/) {
	    $runtime = $1;
	}
    }
    close(FILE);
    if ($verbose) { print "runtime $runtime\n"; }

   if ($runtime < 0) { print "did not get the right runtime $runtime secs\n"; die; }

    return $runtime;
}

sub parse_phylipoutfile {
    my ($phylipmode, $phylipoutfile, 
	$like_ref, 
	$frqa_ref, $frqc_ref, $frqg_ref, $frqt_ref, $frqo_ref,
	$alp_ref, $bet_ref,  
	$ins_ref, $del_ref,
	$ttr_ref, $apb_ref, $verbose) = @_;

    if    ($phylipmode == 0) {
	parse_phylip_dnaml($phylipoutfile, $like_ref, $frqa_ref, $frqc_ref, $frqg_ref, $frqt_ref, $frqo_ref, 
			   $alp_ref, $bet_ref, $ins_ref, $del_ref, $ttr_ref, $apb_ref);
    }
    elsif ($phylipmode == 1 || $phylipmode == 2) {
	parse_phylip_dnaml_erate($phylipoutfile, $like_ref, $frqa_ref, $frqc_ref, $frqg_ref, $frqt_ref, $frqo_ref,
				 $alp_ref, $bet_ref, $ins_ref, $del_ref, $ttr_ref, $apb_ref);
    }
    if ($verbose) {
	printf "alpha %.5f beta %.5f ins %.5f del %.5f like %5.5f -- ", $$alp_ref, $$bet_ref, $$ins_ref, $$del_ref, $$like_ref;
    }


}

sub parse_phylip_dnaml {
    my ($phylipoutfile, $like_ref, 
	$frqa_ref, $frqc_ref, $frqg_ref, $frqt_ref, $frqo_ref, 
	$alp_ref, $bet_ref, $ins_ref, $del_ref, $ttr_ref, $apb_ref) = @_;
    
    my $frqa;
    my $frqc;
    my $frqg;
    my $frqt;
    my $frqo = 0.0;

    my $alp;
    my $bet;
    my $ins = 0.0;
    my $del = 0.0;
    
    my $like;
    my $ttr;
    my $apb  = 1.0;
    
    open(FILE, $phylipoutfile);
    while(<FILE>) {
	if (/^Ln Likelihood\s+=\s+(\S+)/) {
	    $like = $1;
	}
	elsif (/^\s+A\s+(\S+)/) {
	    $frqa = $1;
	}
	elsif (/^\s+C\s+(\S+)/) {
	    $frqc = $1;
	}
	elsif (/^\s+G\s+(\S+)/) {
	    $frqg = $1;
	}
	elsif (/^\s+T\(U\)\s+(\S+)/) {
	    $frqt = $1;
	}
	elsif (/^Transition\/transversion ratio\s+=\s+(\S+)/) {
	    $ttr = $1;
	}
    }
    close(FILE);
    
    if (!$like) {
	print "dnaml did not finish properly\n"; die; 
    }
    
    # normalize frequencies
    my $sum = $frqa +  $frqc +  $frqg +  $frqt;
    if ($sum > 0) {
	$frqa /= $sum;
	$frqc /= $sum;
	$frqg /= $sum;
	$frqt /= $sum;
    }

    # calculate alp bet ins del
    my $frqr  = $frqa + $frqg;
    my $frqy  = $frqc + $frqt;
    my $frqar = 0.0;
    my $frqcy = 0.0;
    my $frqgr = 0.0;
    my $frqty = 0.0;

    if ($frqr > 0.0) {
	$frqar = $frqa / $frqr;
	$frqgr = $frqg / $frqr;
    }
   if ($frqy > 0.0) {
	$frqcy = $frqc / $frqy;
	$frqty = $frqt / $frqy;
    }

    my $aa = $ttr * $frqr * $frqy - $frqa * $frqg - $frqc * $frqt;
    my $bb = $frqa * $frqgr + $frqc * $frqty;
    if ($aa > 0.0 && $aa + $bb > 0.0) { $alp = $aa / ($aa + $bb); }
    else                              { $alp = 0.0; }
    $bet = 1.0 - $alp;

    
    $$frqa_ref = $frqa;
    $$frqc_ref = $frqc;
    $$frqg_ref = $frqg;
    $$frqt_ref = $frqt;
    $$frqo_ref = $frqo;

    $$alp_ref  = $alp;
    $$bet_ref  = $bet;
    $$ins_ref  = $ins;
    $$del_ref  = $del;

    $$ttr_ref  = $ttr;
    $$apb_ref  = $apb;
    $$like_ref = $like;
}

sub parse_phylip_dnaml_erate {
    my ($phylipoutfile, 
	$like_ref, 
	$frqa_ref, $frqc_ref, $frqg_ref, $frqt_ref, $frqo_ref,
	$alp_ref, $bet_ref, 
	$ins_ref, $del_ref, 
	$ttr_ref, $apb_ref) = @_;

    my $frqa;
    my $frqc;
    my $frqg;
    my $frqt;
    my $frqo;

    my $alp;
    my $bet;
    my $ins;
    my $del;

    my $ttr;
    my $apb;

    my $like;

    open(FILE, $phylipoutfile);
    while(<FILE>) {
	if (/^alpha\s+=\s+(\S+)/) {
	    $alp = $1;
	}
	elsif (/^beta\s+=\s+(\S+)/) {
	    $bet = $1;
	}
	elsif (/^Insertions rate\s+=\s+(\S+)/) {
	    $ins = $1;
	}
	elsif (/^Deletions rate\s+=\s+(\S+)/) {
	    $del = $1;
	}
	elsif (/^\s+A\s+(\S+)/) {
	    $frqa = $1;
	}
	elsif (/^\s+C\s+(\S+)/) {
	    $frqc = $1;
	}
	elsif (/^\s+G\s+(\S+)/) {
	    $frqg = $1;
	}
	elsif (/^\s+T\(U\)\s+(\S+)/) {
	    $frqt = $1;
	}
	elsif (/^\s+\-\s+(\S+)/) {
	    $frqo = $1;
	}
	elsif (/^Transition\/transversion ratio\s+=\s+(\S+)/) {
	    $ttr = $1;
	}
	elsif (/^Ln Likelihood\s+=\s+(\S+)/) {
	    $like = $1;
	}
    }
    close(FILE);

   if (!$alp || !$bet || !$ins || !$del) {
	print "parse_phylip_dnaml_erate(): cannot parse all the parameters from output\n"; die; 
    }
    if (!$like) {
	print "parse_phylip_dnaml_erate(): cannot parse likelihood from output\n"; die; 
    }
    
    # normalize frequencies
    my $sum = $frqa +  $frqc +  $frqg +  $frqt;
    if ($sum > 0) {
	$frqa /= $sum;
	$frqc /= $sum;
	$frqg /= $sum;
	$frqt /= $sum;
    }
    
    $apb = $alp + $bet;
    
    $$frqa_ref = $frqa;
    $$frqc_ref = $frqc;
    $$frqg_ref = $frqg;
    $$frqt_ref = $frqt;
    $$frqo_ref = $frqo;

    $$alp_ref  = $alp;
    $$bet_ref  = $bet;
    $$ins_ref  = $ins;
    $$del_ref  = $del;
    $$ttr_ref  = $ttr;
    $$apb_ref  = $apb;
    $$like_ref = $like;
}


sub phylip_analysis {

    my ($case_ref, $globals_ref, $nali, $rosemsafile, $msalen, $sqleng, $sqlena, $indelfrq, $phylipmode, $phylipoutputfile, $phylipouttreefile) = @_;

    my $phyliptimefile = $$case_ref->{"Case::tmpdir"}."/phyliptime.".$$case_ref->{"Case::name"}.".n$nali";

    create_phylip_tree($case_ref, $globals_ref, $nali, 
		       $rosemsafile, $phylipoutputfile, $phylipouttreefile, $phyliptimefile,
		       $msalen, $sqleng, $sqlena, $indelfrq, $phylipmode);
    compare_to_giventrees($case_ref, $globals_ref, $nali, 
			  $phylipoutputfile, $phylipouttreefile, $phyliptimefile,  
			  $msalen, $sqleng, $sqlena, $indelfrq, $phylipmode);
	
    if ($clean) { system("rm $phyliptimefile\n"); }
}

sub presetupcase {
    my ($case_ref, $casedir, $casetmpdir, $abl, $twobranchratio, $ntree, $nmsa, $sqlen, $sqnum, $phylipv,  
	$model, $freqa, $freqc, $freqg, $freqt, $ttratio, 
	$insertionper, $deletionper, $simcoding, $indeldist, $maxindel, $lambda, $c_over_t) = @_;

    my $case = Case->new();

    $case->dir($casedir);
    $case->tmpdir($casetmpdir);

    $case->ntree($ntree);
    $case->nmsa($nmsa);
    $case->sqlen($sqlen);
    $case->sqnum($sqnum);

    $case->model($model);
    $case->freq([$freqa, $freqc, $freqg, $freqt]);

    # we want to make the rose model as close as possible to the phylip model
    #
    # F84 has two parameters alpha and beta [FelsensteinChurchill96]
    #
    # dnaml allows to set R (the average ratio of transitions to transversion given by
    #
    #  R = [ p_a*p_g + p_c*p_t + alpha/beta * (p_a*p_g/p_R + p_c*p_t/p_Y) ]/ (p_R*p_Y)
    #
    #  introduce the quantities  aa = p_R * p_Y * R - p_a*p_g - p_c*p_t
    #
    #                            bb = p_a*p_g/p_R + p_c*p_t/p_Y
    #
    # then alpha/beta = aa/bb
    #
    # phylip introduces also the condition alpha+beta=1
    #
    # and then sets alpha = aa / (aa+bb)
    #               beta  = bb / (aa+bb)
    # 
    # rose [in  rose.c, function build_matrix() ]
    #  allows to set two parameters meansubs and ttratio, such that
    #
    #           beta  = meansubs
    #           alpha = meansubs * ttratio
    #
    # then to make things correspond, for a given R (2.0 phylip default) and a set of stationary probabilities
    #
    # set for rose: ttratio  = aa / bb        = alpha / beta
    #               meansubs = bb / (aa + bb) = beta
    #
    $case->ttratio($ttratio);

    # the rose parameters given the dnaml ttratio and the nucleotide 
    # frequencies
    my $rose_ttratio;
    my $rose_meansubs;
    rose_param(\$rose_ttratio, \$rose_meansubs, $ttratio, $freqa, $freqc, $freqg, $freqt);
    $case->rosettratio($rose_ttratio);
    $case->rosemeansubs("$rose_meansubs"); # for some reason, it has to be given as a string, not a number

    #
    # calculate the dnaml rate parameters alpha and beta
    # given the ttratio and the nucleotide frequencies
    #
    my $alpha = 0.0;
    my $beta  = 0.0;
    dnaml_param(\$alpha, \$beta, $ttratio, $freqa, $freqc, $freqg, $freqt);
    $case->alpha($alpha); 
    $case->beta($beta); 
 
    # the overal rate of substitutions per site = fracchange 
    #
    # 2.0* alpha * bb + beta * (1 - p_a*p_a- p_c*p_c- p_g*p_g- p_t*p_t)
    #
    my $fracchange = fracchange($alpha, $beta, $freqa, $freqc, $freqg, $freqt);
    $case->fracchange($fracchange); 

    # the branch length ratio for a 4-taxon tree
    $case->twobranchratio($twobranchratio);
    # the average branch length for the normalized tree
    $case->abln($abl);
    # the average branch length for the unnormalized tree
    my $ablu = $abl / $fracchange;
    $case->ablu($ablu);

    $case->insertionper($insertionper);
    $case->deletionper($deletionper);
    $case->simcoding($simcoding);
    $case->indeldist($indeldist);
    $case->maxindel($maxindel);
    $case->lambda($lambda);
    $case->c_over_t($c_over_t);

    $$case_ref = $case;
}

sub print_tree {
    my ($tree) = @_;

    my $ntaxa = $tree->{"Tree::ntaxa"};
    my $nnode = ($ntaxa > 1)? $ntaxa-1 : $ntaxa;

    printf "\nTREE STRUCTURE\n";
    printf "ntaxa:\t%d\n", $tree->{"Tree::ntaxa"};
    for (my $n = 0; $n < $nnode; $n ++) {
	printf "node\t%d\tparent\t%d\tleft\t%d\t%f\tright\t%d\t%f\n", 
	$n, ${$tree->parent}[$n], ${$tree->left}[$n], ${$tree->ld}[$n], ${$tree->right}[$n], ${$tree->rd}[$n];
    }
    printf "\n";      
    for (my $t = 0; $t < $ntaxa; $t ++) {
	printf "leaf\t%d\tparent\t%d\n", $t, ${$tree->taxaparent}[$t];
    }

    printf "\n";

}

sub print_case {
    my ($case) = @_;

    open(OUT, ">>".$case->{"Case::outfile"});
    printf OUT "\nSUMMARY\n";
    printf OUT "case:\t%s\n", $case->{"Case::name"};
    printf OUT "outfile:\t%s\n", $case->{"Case::outfile"};
    printf OUT "TREE\n";
    printf OUT "ntrees %d\n", $case->{"Case::ntree"};
    printf OUT "ntaxa %d\n", $case->{"Case::treen"}{"Tree::ntaxa"};
    printf OUT "average branch length %d\n", $case->{"Case::abln"};
    printf OUT "SUBSTITUTION MODEL\n";
    printf OUT "model:\t%s\n", $case->{"Case::model"};
    printf OUT "stat distribution:\t%f\t%f\t%f\t%f\n", ${$case->{"Case::freq"}}[0], ${$case->{"Case::freq"}}[1], ${$case->{"Case::freq"}}[2], ${$case->{"Case::freq"}}[3];
    printf OUT "ttratio:\t%s\n", $case->{"Case::ttratio"};
    printf OUT "rose ttratio:\t%s\n", $case->{"Case::rosettratio"};
    printf OUT "rose meansubs:\t%f\n", $case->{"Case::rosemeansubs"};
    printf OUT "INDEL MODEL\n";
    printf OUT "percentage insertion:\t%s\n", $case->{"Case::insertionper"};
    printf OUT "percentage deletion:\t%s\n", $case->{"Case::deletionper"};
    printf OUT "indel distribution:\t%s\n", $case->{"Case::indeldist"};
    printf OUT "max indel length:\t%d\n", $case->{"Case::maxindel"};
    if ($opt_I) {
	if    ($opt_I eq "flat")      { printf OUT "flat\n"; }
	elsif ($opt_I eq "empirical") { printf OUT "empirical. c_over_t:\t%.3f\n", $case->{"Case::c_over_t"}; }
    }
    else { printf OUT "poisson. lambda:\t%.3f\n", $case->{"Case::lambda"}; }
    printf OUT "ALIGNMENTS\n";
    printf OUT " testruns:\t%d\n", $case->{"Case::nmsa"};
    printf OUT " sequences:\t%d\n", $case->{"Case::sqnum"};
    printf OUT "Target len alignment:\t%d\n", $case->{"Case::sqlen"};
    printf OUT "Actual len alignment:\t%f +/- %f\n", $case->{"Case::meanmsalen"}, $case->{"Case::stdmsalen"} ;
    printf OUT "pairwise id:\t%f +/- %f\n", $case->{"Case::meanpairid"}, $case->{"Case::stdpairid"} ;
    printf OUT "pairwise mut:\t%f +/- %f\n", $case->{"Case::meanpairmut"}, $case->{"Case::stdpairmut"} ;
    printf OUT "pairwise indl:\t%f +/- %f\n", $case->{"Case::meanpairindl"}, $case->{"Case::stdpairindl"} ;
    printf OUT "indel freq:\t%f +/- %f\n", $case->{"Case::meanindelfreq"}, $case->{"Case::stdindelfreq"} ;
    printf OUT "resid freq: A\t%f +/- %f\n", ${$case->meanresfreq}[0],  ${$case->stdresfreq}[0];
    printf OUT "resid freq: C\t%f +/- %f\n", ${$case->meanresfreq}[1],  ${$case->stdresfreq}[1];
    printf OUT "resid freq: G\t%f +/- %f\n", ${$case->meanresfreq}[2],  ${$case->stdresfreq}[2];
    printf OUT "resid freq: T\t%f +/- %f\n", ${$case->meanresfreq}[3],  ${$case->stdresfreq}[3];

    close (OUT);

}

sub print_globals {
    my ($outfile, $globals) = @_;

    my $p = $globals->{"Globalstats::param"};

    open(OUT, ">>$outfile");
    print OUT "\# GLOBAL STATS\n";
    printf OUT "\# trees so far:\t%d\n", $globals->{"Globalstats::treesofar"};
    printf OUT "\# pairwise id:\t%f +/- %f\n", 
    $globals->{"Globalstats::meanpairid"}, $globals->{"Globalstats::stdpairid"};
    printf OUT "\# pairwise mut:\t%f +/- %f\n", 
    $globals->{"Globalstats::meanpairmut"}, $globals->{"Globalstats::stdpairmut"};
    printf OUT "\# pairwise indl:\t%f +/- %f\n", 
    $globals->{"Globalstats::meanpairindl"}, $globals->{"Globalstats::stdpairindl"};
    printf OUT "\# indel freq:\t%f +/- %f\n", 
    $globals->{"Globalstats::meanindelfreq"}, $globals->{"Globalstats::stdindelfreq"} ;
    printf OUT "\# resid freq: A\t%f +/- %f\n", ${$globals->meanresfreq}[0],  ${$globals->stdresfreq}[0];
    printf OUT "\# resid freq: C\t%f +/- %f\n", ${$globals->meanresfreq}[1],  ${$globals->stdresfreq}[1];
    printf OUT "\# resid freq: G\t%f +/- %f\n", ${$globals->meanresfreq}[2],  ${$globals->stdresfreq}[2];
    printf OUT "\# resid freq: T\t%f +/- %f\n", ${$globals->meanresfreq}[3],  ${$globals->stdresfreq}[3];
    printf OUT "\# PARAMETERS\n";
    printf OUT "\# alpha_ave alpha_std beta_ave beta_std ins_ave ins_std del_ave del_std ttr_ave ttr_std apb_ave apb_std\n";
    printf OUT "\# dnaml\n";
    printf OUT "\# %f %f %f %f %f %f %f %f %f %f %f %f\n", 
    $p->{"ParamGlobal::alpx_ave"}, $p->{"ParamGlobal::alpx_std"}, 
    $p->{"ParamGlobal::betx_ave"}, $p->{"ParamGlobal::betx_std"}, 
    $p->{"ParamGlobal::insx_ave"}, $p->{"ParamGlobal::insx_std"}, 
    $p->{"ParamGlobal::delx_ave"}, $p->{"ParamGlobal::delx_std"}, 
    $p->{"ParamGlobal::ttrx_ave"}, $p->{"ParamGlobal::ttrx_std"}, 
    $p->{"ParamGlobal::apbx_ave"}, $p->{"ParamGlobal::apbx_std"};
    printf OUT "\# dnaml-erate\n";
    printf OUT "\# %f %f %f %f %f %f %f %f %f %f %f %f\n", 
    $p->{"ParamGlobal::alpe_ave"}, $p->{"ParamGlobal::alpe_std"}, 
    $p->{"ParamGlobal::bete_ave"}, $p->{"ParamGlobal::bete_std"}, 
    $p->{"ParamGlobal::inse_ave"}, $p->{"ParamGlobal::inse_std"}, 
    $p->{"ParamGlobal::dele_ave"}, $p->{"ParamGlobal::dele_std"}, 
    $p->{"ParamGlobal::ttre_ave"}, $p->{"ParamGlobal::ttre_std"}, 
    $p->{"ParamGlobal::apbe_ave"}, $p->{"ParamGlobal::apbe_std"};
    printf OUT "\# dnaml-erate4\n";
    printf OUT "\# %f %f %f %f %f %f %f %f %f %f %f %f\n", 
    $p->{"ParamGlobal::alpo_ave"}, $p->{"ParamGlobal::alpo_std"}, 
    $p->{"ParamGlobal::beto_ave"}, $p->{"ParamGlobal::beto_std"}, 
    $p->{"ParamGlobal::inso_ave"}, $p->{"ParamGlobal::inso_std"}, 
    $p->{"ParamGlobal::delo_ave"}, $p->{"ParamGlobal::delo_std"}, 
    $p->{"ParamGlobal::ttro_ave"}, $p->{"ParamGlobal::ttro_std"}, 
    $p->{"ParamGlobal::apbo_ave"}, $p->{"ParamGlobal::apbo_std"};
    printf OUT "\# AVERAGE BRANCH LENGTH\n";
    printf OUT "\# dnaml\n";
    printf OUT "\# %f %f \n", $p->{"ParamGlobal::ablx_ave"}, $p->{"ParamGlobal::ablx_std"};
    printf OUT "\# dnaml-erate\n";
    printf OUT "\# %f %f \n", $p->{"ParamGlobal::able_ave"}, $p->{"ParamGlobal::able_std"};
    printf OUT "\# dnaml-erate4\n";
    printf OUT "\# %f %f \n", $p->{"ParamGlobal::ablo_ave"}, $p->{"ParamGlobal::ablo_std"};
    close (OUT);

}

sub print_supercase {
    my ($outfile, $case) = @_;
    
    open(OUT, ">$outfile");
    
    printf OUT "\# SUBSTITUTION MODEL\n";
    printf OUT "\# model:\t%s\n", $case->{"Case::model"};
    printf OUT "\# stat distribution:\t%f\t%f\t%f\t%f\n", ${$case->freq}[0], ${$case->freq}[1], ${$case->freq}[2], ${$case->freq}[3];
    printf OUT "\# ttratio:\t%s\n", $case->{"Case::ttratio"};
    printf OUT "\# INDEL MODEL\n";
    printf OUT "\# percentage insertion:\t%s\n", $case->{"Case::insertionper"};
    printf OUT "\# percentage deletion:\t%s\n", $case->{"Case::deletionper"};
    printf OUT "\# indel distribution:\t%s\n", $case->{"Case::indeldist"};
    printf OUT "\# max indel length:\t%d\n", $case->{"Case::maxindel"};
    printf OUT "\# lambda:\t%.4f\n", $case->{"Case::lambda"};
    printf OUT "\# TREES\n";
    printf OUT "\# \# Trees:\t%d\n", $case->{"Case::ntree"};
    printf OUT "\# \# Taxa:\t%d\n", $case->{"Case::sqnum"};
    printf OUT "\# Average branch length:\t%f\n", $case->{"Case::abln"};
    printf OUT "\# \# Lengths per tree:\t%d\tAverage lenght rage:\t[%d,%d] increments = %d\n", $nlcases, $minval, $maxval, $valinc;
    printf OUT "\# \# Alignments per length and tree:\t%d\n", $case->{"Case::nmsa"};
    printf OUT "\# Total \# tests per tree:\t%d\n", $nlcases*$case->{"Case::nmsa"};
    close (OUT);
    
}

sub realign {
    my ($msafile) = @_;

    my $clustalphyfile = "$msafile.phy";
    my $clustaldndfile = "$msafile.dnd";
    my $fastafile      = "$msafile.fa";

    my @name;
    my @fasta;
    my $seq;
    my $aseq;
    my $alen;
    my $n = 0;

    #create fasta file
    open(FASTA, ">$fastafile");
    open(MSA, "$msafile");
    while(<MSA>) {
	if (/\s+(\d+)\s+(\d+)/) {
	    $aseq = $1;
	    $alen = $2;
	    
	}
	elsif (/^(\S+)\s+(\S+)/) {
	    $name[$n] = $1;
	    $seq = $2;
	    $seq =~ s/-//g; $seq =~ s/ //g; $seq =~ s/\n//g;
	    $fasta[$n++] = $seq;
	}
	elsif (/^\s+$/) {
	    if ($n != $aseq) { print "bad alignment aseq=$aseq n=$n\n"; die; }
	    $n = 0;
	}
	elsif (/\s+(\S+)/) {
	    $seq = $1;
	    $seq =~ s/-//g; $seq =~ s/ //g; $seq =~ s/\n//g;
	    $fasta[$n++] .= $seq;
	}
    }
    close (MSA);
    for (my $s = 0; $s < $aseq; $s ++) { 
	print FASTA ">$name[$s]\n";
	print FASTA "$fasta[$s]\n";
    }
    
    if ($verbose) {
	system("more $fastafile\n");
    }

    # realign using the fasta file
    # and clustalw 1.83
    system("$realignwith -infile=$fastafile -output=phylip -outorder=input\n");
    if ($verbose) {
	system("more $msafile\n");
	system("more $clustalphyfile\n");
    }
    
    # send back the new alignment
    system("mv $clustalphyfile $msafile\n");

    if ($clean) { 
	system("rm $clustaldndfile\n"); 
	system("rm $fastafile\n"); 
    }
}


sub reordermsa_rose {
    my ($phylipmsafile) = @_;
    
    my $newfile = "$phylipmsafile.new";
    
    my @msa;
    my $aseq;
    my $alen;
    my $n = 0;
    
    # The given tree has the 0 taxon as left child of the root.
    # Rose generate alignments with the 0 taxon at the end.
    # By default phylip uses the first sequence as left child of the root.
    #
    # In order to compare the given tree and the phylip tree, both unrooted,
    # we need follow the convenction that both have taxon 0 as the left child of the root.
    #
    # We take the alignment generated by rose and
    # we place the last aligned sequence (taxon 0)  as the first, and all the others randomly;
    # generate a random  number between 0 and nseq-2
    
    my @perm;
    my @col;
    my $pos;
    
    open(NEWMSA, ">$newfile");
    open(MSA, "$phylipmsafile");
    while(<MSA>) {
	if (/\s+(\d+)\s+(\d+)/) {
	    print NEWMSA $_;
	    $aseq = $1;
	    $alen = $2;
	    
	    # calculate the permutation vector
	    for (my $s = 0; $s < $aseq; $s ++)   { $perm[$s] = $s; }
	    for (my $s = 0; $s < $aseq-1; $s ++) { $col[$s]  = $s; }
	    
	    for (my $s = $aseq-2; $s >= 0; $s --) { 
		$pos = int(rand($s));
		my $c = $perm[$col[$pos]];
		$perm[$col[$pos]]   = $perm[$col[$pos-1]];
		$perm[$col[$pos-1]] = $c;		
	    }
	    for (my $s = $aseq-1; $s >= 0; $s --) { $perm[$s] = $perm[$s-1]; }
	    $perm[0] = $aseq-1;

	    if ($verbose) {
		print "PERM ";
		for (my $s  =0; $s < $aseq; $s ++) { print "$perm[$s] "; }
		print "\n";
	    }

	}
	elsif (/^(\S+)\s+(\S+)/) {
	    $msa[$n++]   = $_;
	}
	elsif (/^\s+$/) {
	    if ($n != $aseq) { print "bad alignment aseq=$aseq n=$n\n"; die; }

	    for (my $s = 0; $s < $aseq; $s ++) { 
		print NEWMSA "$msa[$perm[$s]]";
	    }
	    $n = 0;
	    print NEWMSA $_;
	    for (my $s = 0; $s < $aseq; $s ++) { undef($msa[$s]); }
	}
	elsif (/\s+(\S+)/) {
	    $msa[$n++] = $_;
	}
    }
    close (MSA);

    # in case there is not a blank at the end of the aligment
    if ($msa[0]) {
	for (my $s = 0; $s < $aseq; $s ++) { 
	    print NEWMSA "$msa[$perm[$s]]";
	}
    }
    close (NEWMSA);

    if ($verbose) {
	system("more $phylipmsafile\n");
	system("more $newfile\n");
    }

    system("mv $newfile $phylipmsafile\n");

}

sub reordermsa_erate {
    my ($phylipmsafile) = @_;
    
    my $newfile = "$phylipmsafile.new";
    
    my @msa;
    my $aseq;
    my $alen;
    my $n = 0;
    
    # The given tree has the 0 taxon as left child of the root.
    # Rose generate alignments with the 0 taxon at the end.
    # By default phylip uses the first sequence as left child of the root.
    #
    # In order to compare the given tree and the phylip tree, both unrooted,
    # we need follow the convenction that both have taxon 0 as the left child of the root.
    #
    # We take the alignment generated by erate and
    # leave the first aligned sequence (taxon 0) intact, and all the others randomly;
    # generate a random  number between 1 and nseq-1
    
    my @perm;
    my @col;
    my $pos;
    
    open(NEWMSA, ">$newfile");
    open(MSA, "$phylipmsafile");
    while(<MSA>) {
	if (/\s+(\d+)\s+(\d+)/) {
	    print NEWMSA $_;
	    $aseq = $1;
	    $alen = $2;
	    
	    # calculate the permutation vector
	    for (my $s = 0; $s < $aseq; $s ++) { $perm[$s] = $s; }
	    for (my $s = 1; $s < $aseq; $s ++) { $col[$s]  = $s; }
	    
	    for (my $s = $aseq-1; $s > 1; $s --) { 
		$pos = 1+int(rand($s));
		my $c = $perm[$col[$pos]];
		$perm[$col[$pos]]   = $perm[$col[$s]];
		$perm[$col[$s]] = $c;
	    }

	    if ($verbose) {
		print "PERM ";
		for (my $s =0; $s < $aseq; $s ++) { print "$perm[$s] "; }
		print "\n";
	    }

	}
	elsif (/^(\S+)\s+(\S+)/) {
	    $msa[$n++]   = $_;
	}
	elsif (/^\s+$/) {
	    if ($n != $aseq) { print "bad alignment aseq=$aseq n=$n\n"; die; }

	    for (my $s = 0; $s < $aseq; $s ++) { 
		print NEWMSA "$msa[$perm[$s]]";
	    }
	    $n = 0;
	    print NEWMSA $_;
	    for (my $s = 0; $s < $aseq; $s ++) { undef($msa[$s]); }
	}
	elsif (/\s+(\S+)/) {
	    $msa[$n++] = $_;
	}
    }
    close (MSA);

    # in case there is not a blank at the end of the aligment
    if ($msa[0]) {
	for (my $s = 0; $s < $aseq; $s ++) { 
	    print NEWMSA "$msa[$perm[$s]]";
	}
    }
    close (NEWMSA);

    if ($verbose) {
	system("more $phylipmsafile\n");
	system("more $newfile\n");
    }

    system("mv $newfile $phylipmsafile\n");

}

# the phylip tree has the first sequence as right from root node
# my convention is first sequence left from root node.
# this function reverse the order of the tree in  nh format
sub reverse_tree {
    my ($nhtree_ref) = @_;

    my $nhtree = $$nhtree_ref;
    my $revnhtree = "";

    my @time;
    my $time;

    if (!$nhtree) { print "phylip did not generate any tree\n"; die; }

    while($nhtree) {
	if (0) {
	    print "nh   =$nhtree\n";
	    print "revnh=$revnhtree\n";
	}

	if ($nhtree =~ /^(.+)\)\;$/) {
	    $nhtree = $1;
	    $revnhtree = "(";
	}
	elsif ($nhtree =~ /^(.+\,)([^\(\)\:\,]+)\:([\d\.]+)$/) {
	    $nhtree = $1;
	    $revnhtree .= "$2\:$3";
	}
	elsif ($nhtree =~ /^(.*\()([^\(\)\:\,]+)\:([\d\.]+)$/) {
	    $nhtree = $1;
	    $revnhtree .= "$2\:$3";
	}
	elsif ($nhtree =~ /^(.+)\)\:([\d\.]+)$/) {
	    $nhtree = $1;
	    push(@time, $2);
	    $revnhtree .= "\(";
	}
	elsif ($nhtree =~ /^(.+)\,$/) {
	    $nhtree = $1;
	    $revnhtree .= "\,";
	}
	elsif ($nhtree =~ /^(.+)\:$/) {
	    $nhtree = $1;
	    $revnhtree .= "\:";
	}
	elsif ($nhtree =~ /^(.+)\)$/) {
	    $nhtree = $1;
	    $revnhtree .= "\(";
	}
	elsif ($nhtree =~ /^(.+)\($/) {
	    $nhtree = $1;
	    $time = pop(@time);
	    $revnhtree .= "\)\:$time";
	}
	elsif ($nhtree =~ /^\($/) {
	    $nhtree = "";
	    $revnhtree .= "\)\;";
	}
	else {
	    print "bad tree reversing string=$nhtree\n"; die;
	}
    }

    if (@time) { print "bad tree reversing sring=$nhtree\n"; die; }

    $$nhtree_ref = $revnhtree;

    if ($verbose) { print "reversed tree\n$$nhtree_ref\n"; }
}

sub rnd_UniformPositive {
    my $x;
    do { $x = rand(1); } while ($x == 0.0);
    return $x;
}

# we want to make the rose model as close as possible to the phylip model
#
# F84 has two parameters alpha and beta [FelsensteinChurchill96]
#
# dnaml allows to set R (the average ratio of transitions to transversion given by
#
#  R = [ p_a*p_g + p_c*p_t + alpha/beta * (p_a*p_g/p_R + p_c*p_t/p_Y) ]/ (p_R*p_Y)
#
#  introduce the quantities  aa = p_R * p_Y * R - p_a*p_g - p_c*p_t
#
#                            bb = p_a*p_g/p_R + p_c*p_t/p_Y
#
# then alpha/beta = aa/bb
#
# phylip introduces also the condition alpha+beta=1
#
# and then sets alpha = aa / (aa+bb)
#               beta  = bb / (aa+bb)
# 
    # rose [in  rose.c, function build_matrix() ]
#  allows to set two parameters meansubs and ttratio, such that
#
#           beta  = meansubs
#           alpha = meansubs * ttratio
#
# then to make things correspond, for a given R (2.0 phylip default) and a set of stationary probabilities
#
# set for rose: ttratio  = aa / bb        = alpha / beta
#               meansubs = bb / (aa + bb) = beta
#
sub rose_param {
    my ($rose_ttratio_ref, $rose_meansubs_ref, $ttratio, $freqa, $freqc, $freqg, $freqt) = @_;

    my $rose_ttratio;
    my $rose_meansubs;

    my $aa = ($freqa + $freqg) * ($freqc + $freqt) * $ttratio - $freqa * $freqg - $freqc * $freqt;
    my $bb = $freqa * $freqg / ($freqa + $freqg) + $freqc * $freqt / ($freqc + $freqt);

    $rose_ttratio  = $aa / $bb;
    $rose_meansubs = $bb / ($aa + $bb);

    $$rose_ttratio_ref  = $rose_ttratio;
    $$rose_meansubs_ref = $rose_meansubs;
}

sub rosemsa2phylip{
    my ($rosefile, $phylipfile) = @_;

    my $write = 0;

    if (!$rosefile) {
	print"rose failed\n";
	die;
    }
    open(OUT, ">$phylipfile");
    open(FILE, "$rosefile");
    while (<FILE>) {
	if (/^Alignment:/) {
	    $write = 1;
	}
	elsif (/^PHYLIP/) { $write = 0; }
	elsif ($write == 1) {
	    print OUT $_;
	}
    }
    close (FILE);
    close (OUT);

    # rose puts the sequences in a particular order from last to first
    #
    # I rearrange the sequences in the msa, so that the first taxon goes first
    # (this is  necessary to later compare the given and the phylip tree) 
    # and the rest of the aligned sequences are arranged randomly.
    #
    reordermsa_rose($phylipfile);

    if ($verbose) { system("more $phylipfile\n"); }
}


sub run_erate {
    my ($case_ref, $nali, $msafile) = @_;
    
    my $eratescreenout = $$case_ref->{"Case::tmpdir"}."/erate_screenout.".$$case_ref->{"Case::name"}.".n$nali";
    my $treefile = $$case_ref->{"Case::tmpdir"}."/treefile_erate.".$$case_ref->{"Case::name"}.".n$nali";
    my $tree = $$case_ref->treeunh;

    # put both tree in file 
    open(FILE, ">$treefile");
    print FILE "$tree\n";
    close (FILE);
    if ($verbose) { 
	print "the erate-input tree:\n";
	system ("more $treefile\n"); 
 	print "end erate-input tree\n\n";
   }

    my $ttr    = 2.0;
    my $fixlen = $$case_ref->sqlen;
    my $ins    = $$case_ref->insertionper;
    my $del    = $$case_ref->deletionper;

    if (1) { 
	system("echo $erate --ttr $ttr --fixlen $fixlen --ins $ins --del $del --phylip $treefile $msafile\n"); 
    }
    system("$erate --ttr $ttr --fixlen $fixlen --ins $ins --del $del --phylip -v $treefile $msafile > $eratescreenout\n");
    if ($verbose) { 
	system("more $eratescreenout\n"); 
    }
    if ($verbose) { 
	system("more $treefile\n"); 
    }

    reordermsa_erate($msafile);
    if ($verbose) { 
	system("more $msafile\n"); 
    }

    if ($clean) { 
	system("rm $treefile\n"); 
	system("rm $eratescreenout\n"); 
    }
}

sub run_rose {
    my ($case_ref, $outputfile) = @_;
    
    if ($verbose) { 
	system("echo $rose -I ".$$case_ref->{"Case::tmpdir"}."  ".$$case_ref->{"Case::roseinputfile"}."\n"); 
    }
    system("$rose -I ".$$case_ref->{"Case::tmpdir"}."  ".$$case_ref->{"Case::roseinputfile"}.">$outputfile\n");

    if ($verbose) { system("more $outputfile\n"); }
}

sub setup {
    my ($globals_ref, $case_ref, $casedir, $casetmpdir, $abl, $twobranchratio, $ntree, $nmsa, $sqlen, $sqnum, $phylipv, 
	$model, $freqa, $freqc, $freqg, $freqt, $ttratio, 
	$insertionper, $deletionper, $simcoding, $indeldist, $maxindel, $lambda, $c_over_t) = @_;

    setupglobals ($globals_ref);
    presetupcase($case_ref, $casedir, $casetmpdir, $abl, $twobranchratio, $ntree, $nmsa, $sqlen, $sqnum, $phylipv, 
		 $model, $freqa, $freqc, $freqg, $freqt, $ttratio, 
		 $insertionper, $deletionper, $simcoding, $indeldist, $maxindel, $lambda, $c_over_t);
    setuphistograms($case_ref);

}

sub setup_tree {
    my ($case_ref, $simtree) = @_;

    my $treeunh;
    my $treennh;
  
    if ($simtree == 1) { 
        # generate a tree with a given abl
	tree_simulate($case_ref);
    }
    else { 
	if ($$case_ref->sqnum == 8) {
	    # the fix 8-taxon symmetric tree
	    $treeunh = nhtree_eightaxa_binary($$case_ref->ablu); # give rose the binary tree
	    $treennh = nhtree_eightaxa($$case_ref->abln);
	}
	elsif ($$case_ref->sqnum == 4) {
	    # the fix 4-taxon tree depending on two parameters Huelsenbeck 1994
	    # rose uses the unnormalize tree, which we make binary
	    nhtree_fourtaxa($$case_ref->twobranchratio, $$case_ref->abln, \$treennh, $$case_ref->ablu, \$treeunh);
	}
	else { print "tree has to have 4 or 8 taxa\n"; die; }

	if ($verbose) { print "\nUnnormalized treenh:\n$treeunh\n"; print "\nNormalized treenh:\n$treennh\n"; }
	$$case_ref->treeunh($treeunh);
	$$case_ref->treennh($treennh);
	sq_hash_table($case_ref);
	create_tree_structures($case_ref);
    }

}

sub setupcase {
    my ($case_ref, $casename, $casedir, $casetmpdir, $sqlen) = @_;

    my $case = Case->new();
    $case = $$case_ref;

    $case->name($casename);
    $case->dir($casedir);
    $case->tmpdir($casetmpdir);

    $case->dnadefaultsfile($case->{"Case::tmpdir"}."/dna-defaults.".$case->{"Case::name"});
    $case->roseinputfile($case->{"Case::tmpdir"}."/roseinput.".$case->{"Case::name"});
    $case->outfile($case->{"Case::tmpdir"}."/comparetrees.".$case->{"Case::name"});

    $case->sqlen($sqlen);

    $case->meanpairid(0.0);
    $case->stdpairid(0.0);
    $case->meanpairmut(0.0);
    $case->stdpairmut(0.0);
    $case->meanpairindl(0.0);
    $case->stdpairindl(0.0);
    $case->meanresfreq([0.0,0.0,0.0,0.0]);
    $case->stdresfreq([0.0,0.0,0.0,0.0]);
    $case->meanindelfreq(0.0);
    $case->stdindelfreq(0.0);
    $case->meanmsalen(0.0);
    $case->stdmsalen(0.0);

    system ("rm ".$case->{"Case::outfile"}."\n"); # make sure the outfile is empty to start

    make_rose_dna_defaults($case);
    make_rose_input_file($case);

    system("more ".$case->{"Case::dnadefaultsfile"}." > ".$case->{"Case::outfile"}."\n");
    system("more ".$case->{"Case::roseinputfile"}." >> ".$case->{"Case::outfile"}."\n");

    $$case_ref = $case;
}
   
sub setupglobals_param {
    my ($param_ref) = @_;

    $$param_ref->totx(0);
    $$param_ref->tote(0);
    $$param_ref->toto(0);

    $$param_ref->ablx_ave(0.0);
    $$param_ref->able_ave(0.0);
    $$param_ref->ablo_ave(0.0);
    $$param_ref->ablx_std(0.0);
    $$param_ref->able_std(0.0);
    $$param_ref->ablo_std(0.0);

    $$param_ref->alpx_ave(0.0);
    $$param_ref->alpe_ave(0.0);
    $$param_ref->alpo_ave(0.0);
    $$param_ref->alpx_std(0.0);
    $$param_ref->alpe_std(0.0);
    $$param_ref->alpo_std(0.0);

    $$param_ref->betx_ave(0.0);
    $$param_ref->bete_ave(0.0);
    $$param_ref->beto_ave(0.0);
    $$param_ref->betx_std(0.0);
    $$param_ref->bete_std(0.0);
    $$param_ref->beto_std(0.0);

    $$param_ref->insx_ave(0.0);
    $$param_ref->inse_ave(0.0);
    $$param_ref->inso_ave(0.0);
    $$param_ref->insx_std(0.0);
    $$param_ref->inse_std(0.0);
    $$param_ref->inso_std(0.0);

    $$param_ref->delx_ave(0.0);
    $$param_ref->dele_ave(0.0);
    $$param_ref->delo_ave(0.0);
    $$param_ref->delx_std(0.0);
    $$param_ref->dele_std(0.0);
    $$param_ref->delo_std(0.0);

    $$param_ref->ttrx_ave(0.0);
    $$param_ref->ttre_ave(0.0);
    $$param_ref->ttro_ave(0.0);
    $$param_ref->ttrx_std(0.0);
    $$param_ref->ttre_std(0.0);
    $$param_ref->ttro_std(0.0);

    $$param_ref->apbx_ave(0.0);
    $$param_ref->apbe_ave(0.0);
    $$param_ref->apbo_ave(0.0);
    $$param_ref->apbx_std(0.0);
    $$param_ref->apbe_std(0.0);
    $$param_ref->apbo_std(0.0);

}

sub setupglobals {
    my ($globals_ref) = @_;

    my $globals = Globalstats->new();
    my $param   = ParamGlobal->new();

    setupglobals_param(\$param);

    $globals->param($param);

    $globals->treesofar(0);
    $globals->meanpairid(0.0);
    $globals->stdpairid(0.0);
    $globals->meanpairmut(0.0);
    $globals->stdpairmut(0.0);
    $globals->meanpairindl(0.0);
    $globals->stdpairindl(0.0);
    $globals->meanresfreq([0.0,0.0,0.0,0.0]);
    $globals->stdresfreq([0.0,0.0,0.0,0.0]);
    $globals->meanindelfreq(0.0);
    $globals->stdindelfreq(0.0);

    $$globals_ref = $globals;

}

sub setuphistograms {
    my ($case_ref) = @_;

    $case = $$case_ref;

    # histogram files
    # collect date respect to:
    #        msalen
    #        seqlen geometric mean
    #        seqlen arithmetic mean
    #
    my $hfo     = Histogram->new(); 
    my $hmsalen = Histogram->new(); 
    my $hsqleng = Histogram->new(); 
    my $hsqlena = Histogram->new(); 

    my $hpfo     = ParamHisto->new(); 
    my $hpmsalen = ParamHisto->new(); 
    my $hpsqleng = ParamHisto->new(); 
    my $hpsqlena = ParamHisto->new(); 

    $hfo->histofile($case->{"Case::dir"}."/histogram_fo");
    $hmsalen->histofile($case->{"Case::dir"}."/histogram_msalen");
    $hsqleng->histofile($case->{"Case::dir"}."/histogram_sqleng");
    $hsqlena->histofile($case->{"Case::dir"}."/histogram_sqlena");

    $hfo->histofilewc($case->{"Case::dir"}."/wc_histogram_fo");
    $hmsalen->histofilewc($case->{"Case::dir"}."/wc_histogram_msalen");
    $hsqleng->histofilewc($case->{"Case::dir"}."/wc_histogram_sqleng");
    $hsqlena->histofilewc($case->{"Case::dir"}."/wc_histogram_sqlena");

    $hpfo->histofile($case->{"Case::dir"}."/paramhisto_fo");
    $hpmsalen->histofile($case->{"Case::dir"}."/paramhisto_msalen");
    $hpsqleng->histofile($case->{"Case::dir"}."/paramhisto_sqleng");
    $hpsqlena->histofile($case->{"Case::dir"}."/paramhisto_sqlena");

    $hpfo->histofilewc($case->{"Case::dir"}."/wc_paramhisto_fo");
    $hpmsalen->histofilewc($case->{"Case::dir"}."/wc_paramhisto_msalen");
    $hpsqleng->histofilewc($case->{"Case::dir"}."/wc_paramhisto_sqleng");
    $hpsqlena->histofilewc($case->{"Case::dir"}."/wc_paramhisto_sqlena");

    # histogram parameters
    # max len = $valmax, binned in increments of $valinc
    my $N = 5*$maxval;
    my $k = 1.0/$valinc;
    $hmsalen->N($N);  $hmsalen->k($k);
    $hsqleng->N($N);  $hsqleng->k($k);
    $hsqlena->N($N);  $hsqlena->k($k);

    $hpmsalen->N($N); $hpmsalen->k($k);
    $hpsqleng->N($N); $hpsqleng->k($k);
    $hpsqlena->N($N); $hpsqlena->k($k);

    my $Nfo = 1.0;
    my $kfo = 300;
    $hfo->N($Nfo);  $hfo->k($kfo);
    $hpfo->N($Nfo); $hpfo->k($kfo);

    # init histogram arrays
    init_histogram($hfo->N, $hfo->k, 
		   $hfo->totx, $hfo->tpx, 
		   $hfo->dist1x_ave, $hfo->dist1x_std, 
		   $hfo->dist2x_ave, $hfo->dist2x_std, 
		   $hfo->runtmx_ave, $hfo->runtmx_std);
    init_histogram($hmsalen->N, $hmsalen->k, 
		   $hmsalen->totx, $hmsalen->tpx, 
		   $hmsalen->dist1x_ave, $hmsalen->dist1x_std, 
		   $hmsalen->dist2x_ave, $hmsalen->dist2x_std, 
		   $hmsalen->runtmx_ave, $hmsalen->runtmx_std);
    init_histogram($hsqleng->N, $hsqleng->k, 
		   $hsqleng->totx, $hsqleng->tpx, 
		   $hsqleng->dist1x_ave, $hsqleng->dist1x_std, 
		   $hsqleng->dist2x_ave, $hsqleng->dist2x_std, 
		   $hsqleng->runtmx_ave, $hsqleng->runtmx_std);
    init_histogram($hsqlena->N, $hsqlena->k, 
		   $hsqlena->totx, $hsqlena->tpx, 
		   $hsqlena->dist1x_ave, $hsqlena->dist1x_std, 
		   $hsqlena->dist2x_ave, $hsqlena->dist2x_std, 
		   $hsqlena->runtmx_ave, $hsqlena->runtmx_std);
    
    init_histogram($hfo->N, $hfo->k, 
		   $hfo->tote, $hfo->tpe, 
		   $hfo->dist1e_ave, $hfo->dist1e_std, 
		   $hfo->dist2e_ave, $hfo->dist2e_std, 
		   $hfo->runtme_ave, $hfo->runtme_std);
    init_histogram($hmsalen->N, $hmsalen->k, 
		   $hmsalen->tote, $hmsalen->tpe, 
		   $hmsalen->dist1e_ave, $hmsalen->dist1e_std, 
		   $hmsalen->dist2e_ave, $hmsalen->dist2e_std, 
		   $hmsalen->runtme_ave, $hmsalen->runtme_std);
    init_histogram($hsqleng->N, $hsqleng->k, 
		   $hsqleng->tote, $hsqleng->tpe, 
		   $hsqleng->dist1e_ave, $hsqleng->dist1e_std, 
		   $hsqleng->dist2e_ave, $hsqleng->dist2e_std,
		   $hsqleng->runtme_ave, $hsqleng->runtme_std);
    init_histogram($hsqlena->N, $hsqlena->k, 
		   $hsqlena->tote, $hsqlena->tpe, 
		   $hsqlena->dist1e_ave, $hsqlena->dist1e_std, 
		   $hsqlena->dist2e_ave, $hsqlena->dist2e_std, 
		   $hsqlena->runtme_ave, $hsqlena->runtme_std);
    
    init_histogram($hfo->N, $hfo->k, 
		   $hfo->toto, $hfo->tpo, 
		   $hfo->dist1o_ave, $hfo->dist1o_std, 
		   $hfo->dist2o_ave, $hfo->dist2o_std, 
		   $hfo->runtmo_ave, $hfo->runtmo_std);
    init_histogram($hmsalen->N, $hmsalen->k, 
		   $hmsalen->toto, $hmsalen->tpo, 
		   $hmsalen->dist1o_ave, $hmsalen->dist1o_std, 
		   $hmsalen->dist2o_ave, $hmsalen->dist2o_std, 
		   $hmsalen->runtmo_ave, $hmsalen->runtmo_std);
    init_histogram($hsqleng->N, $hsqleng->k, 
		   $hsqleng->toto, $hsqleng->tpo, 
		   $hsqleng->dist1o_ave, $hsqleng->dist1o_std, 
		   $hsqleng->dist2o_ave, $hsqleng->dist2o_std, 
		   $hsqleng->runtmo_ave, $hsqleng->runtmo_std);
    init_histogram($hsqlena->N, $hsqlena->k, 
		   $hsqlena->toto, $hsqlena->tpo, 
		   $hsqlena->dist1o_ave, $hsqlena->dist1o_std, 
		   $hsqlena->dist2o_ave, $hsqlena->dist2o_std, 
		   $hsqlena->runtmo_ave, $hsqlena->runtmo_std);
    
    init_comp_histogram($hfo->N, $hfo->k, 
		   $hfo->totxe, $hfo->tpxe, 
		   $hfo->dist1xe_ave, $hfo->dist1xe_std, 
		   $hfo->dist2xe_ave, $hfo->dist2xe_std);
    init_comp_histogram($hmsalen->N, $hmsalen->k, 
		   $hmsalen->totxe, $hmsalen->tpxe, 
		   $hmsalen->dist1xe_ave, $hmsalen->dist1xe_std, 
		   $hmsalen->dist2xe_ave, $hmsalen->dist2xe_std);
    init_comp_histogram($hsqleng->N, $hsqleng->k, 
		   $hsqleng->totxe, $hsqleng->tpxe, 
		   $hsqleng->dist1xe_ave, $hsqleng->dist1xe_std, 
		   $hsqleng->dist2xe_ave, $hsqleng->dist2xe_std);
    init_comp_histogram($hsqlena->N, $hsqlena->k, 
		   $hsqlena->totxe, $hsqlena->tpxe, 
		   $hsqlena->dist1xe_ave, $hsqlena->dist1xe_std, 
		   $hsqlena->dist2xe_ave, $hsqlena->dist2xe_std);
    
    init_comp_histogram($hfo->N, $hfo->k, 
		   $hfo->totxo, $hfo->tpxo, 
		   $hfo->dist1xo_ave, $hfo->dist1xo_std, 
		   $hfo->dist2xo_ave, $hfo->dist2xo_std);
    init_comp_histogram($hmsalen->N, $hmsalen->k, 
		   $hmsalen->totxo, $hmsalen->tpxo, 
		   $hmsalen->dist1xo_ave, $hmsalen->dist1xo_std, 
		   $hmsalen->dist2xo_ave, $hmsalen->dist2xo_std);
    init_comp_histogram($hsqleng->N, $hsqleng->k, 
		   $hsqleng->totxo, $hsqleng->tpxo, 
		   $hsqleng->dist1xo_ave, $hsqleng->dist1xo_std, 
		   $hsqleng->dist2xo_ave, $hsqleng->dist2xo_std);
    init_comp_histogram($hsqlena->N, $hsqlena->k, 
		   $hsqlena->totxo, $hsqlena->tpxo, 
		   $hsqlena->dist1xo_ave, $hsqlena->dist1xo_std, 
		   $hsqlena->dist2xo_ave, $hsqlena->dist2xo_std);
    
    init_paramhisto($hpfo->N, $hpfo->k, $hpfo->totx, 
		    $hpfo->likex_ave, $hpfo->likex_std, $hpfo->ablx_ave, $hpfo->ablx_std, 
		    $hpfo->alpx_ave,  $hpfo->alpx_std,  $hpfo->betx_ave, $hpfo->betx_std, 
		    $hpfo->insx_ave,  $hpfo->insx_std,  $hpfo->delx_ave, $hpfo->delx_std, 
		    $hpfo->ttrx_ave,  $hpfo->ttrx_std,  $hpfo->apbx_ave, $hpfo->apbx_std);
    init_paramhisto($hpmsalen->N, $hpmsalen->k, $hpmsalen->totx, 
		    $hpmsalen->likex_ave, $hpmsalen->likex_std, $hpmsalen->ablx_ave, $hpmsalen->ablx_std, 
		    $hpmsalen->alpx_ave,  $hpmsalen->alpx_std,  $hpmsalen->betx_ave, $hpmsalen->betx_std, 
		    $hpmsalen->insx_ave,  $hpmsalen->insx_std,  $hpmsalen->delx_ave, $hpmsalen->delx_std, 
		    $hpmsalen->ttrx_ave,  $hpmsalen->ttrx_std,  $hpmsalen->apbx_ave, $hpmsalen->apbx_std);
    init_paramhisto($hpsqleng->N, $hpsqleng->k, $hpsqleng->totx, 
		    $hpsqleng->likex_ave, $hpsqleng->likex_std, $hpsqleng->ablx_ave, $hpsqleng->ablx_std, 
		    $hpsqleng->alpx_ave,  $hpsqleng->alpx_std,  $hpsqleng->betx_ave, $hpsqleng->betx_std, 
		    $hpsqleng->insx_ave,  $hpsqleng->insx_std,  $hpsqleng->delx_ave, $hpsqleng->delx_std, 
		    $hpsqleng->ttrx_ave,  $hpsqleng->ttrx_std,  $hpsqleng->apbx_ave, $hpsqleng->apbx_std);
    init_paramhisto($hpsqlena->N, $hpsqlena->k, $hpsqlena->totx, 
		    $hpsqlena->likex_ave, $hpsqlena->likex_std, $hpsqlena->ablx_ave, $hpsqlena->ablx_std, 
		    $hpsqlena->alpx_ave,  $hpsqlena->alpx_std,  $hpsqlena->betx_ave, $hpsqlena->betx_std, 
		    $hpsqlena->insx_ave,  $hpsqlena->insx_std,  $hpsqlena->delx_ave, $hpsqlena->delx_std, 
		    $hpsqlena->ttrx_ave,  $hpsqlena->ttrx_std,  $hpsqlena->apbx_ave, $hpsqlena->apbx_std);

    init_paramhisto($hpfo->N, $hpfo->k, $hpfo->tote, 
		    $hpfo->likee_ave, $hpfo->likee_std, $hpfo->able_ave, $hpfo->able_std, 
		    $hpfo->alpe_ave,  $hpfo->alpe_std,  $hpfo->bete_ave, $hpfo->bete_std, 
		    $hpfo->inse_ave,  $hpfo->inse_std,  $hpfo->dele_ave, $hpfo->dele_std, 
		    $hpfo->ttre_ave,  $hpfo->ttre_std,  $hpfo->apbe_ave, $hpfo->apbe_std);
    init_paramhisto($hpmsalen->N, $hpmsalen->k, $hpmsalen->tote, 
		    $hpmsalen->likee_ave, $hpmsalen->likee_std, $hpmsalen->able_ave, $hpmsalen->able_std, 
		    $hpmsalen->alpe_ave,  $hpmsalen->alpe_std,  $hpmsalen->bete_ave, $hpmsalen->bete_std, 
		    $hpmsalen->inse_ave,  $hpmsalen->inse_std,  $hpmsalen->dele_ave, $hpmsalen->dele_std, 
		    $hpmsalen->ttre_ave,  $hpmsalen->ttre_std,  $hpmsalen->apbe_ave, $hpmsalen->apbe_std);
    init_paramhisto($hpsqleng->N, $hpsqleng->k, $hpsqleng->tote, 
		    $hpsqleng->likee_ave, $hpsqleng->likee_std, $hpsqleng->able_ave, $hpsqleng->able_std, 
		    $hpsqleng->alpe_ave,  $hpsqleng->alpe_std,  $hpsqleng->bete_ave, $hpsqleng->bete_std, 
		    $hpsqleng->inse_ave,  $hpsqleng->inse_std,  $hpsqleng->dele_ave, $hpsqleng->dele_std, 
		    $hpsqleng->ttre_ave,  $hpsqleng->ttre_std,  $hpsqleng->apbe_ave, $hpsqleng->apbe_std);
    init_paramhisto($hpsqlena->N, $hpsqlena->k, $hpsqlena->tote, 
		    $hpsqlena->likee_ave, $hpsqlena->likee_std, $hpsqlena->able_ave, $hpsqlena->able_std, 
		    $hpsqlena->alpe_ave,  $hpsqlena->alpe_std,  $hpsqlena->bete_ave, $hpsqlena->bete_std, 
		    $hpsqlena->inse_ave,  $hpsqlena->inse_std,  $hpsqlena->dele_ave, $hpsqlena->dele_std, 
		    $hpsqlena->ttre_ave,  $hpsqlena->ttre_std,  $hpsqlena->apbe_ave, $hpsqlena->apbe_std);

    init_paramhisto($hpfo->N, $hpfo->k, $hpfo->toto, 
		    $hpfo->likeo_ave, $hpfo->likeo_std, $hpfo->ablo_ave, $hpfo->ablo_std, 
		    $hpfo->alpo_ave,  $hpfo->alpo_std,  $hpfo->beto_ave, $hpfo->beto_std, 
		    $hpfo->inso_ave,  $hpfo->inso_std,  $hpfo->delo_ave, $hpfo->delo_std, 
		    $hpfo->ttro_ave,  $hpfo->ttro_std,  $hpfo->apbo_ave, $hpfo->apbo_std);
    init_paramhisto($hpmsalen->N, $hpmsalen->k, $hpmsalen->toto, 
		    $hpmsalen->likeo_ave, $hpmsalen->likeo_std, $hpmsalen->ablo_ave, $hpmsalen->ablo_std, 
		    $hpmsalen->alpo_ave,  $hpmsalen->alpo_std,  $hpmsalen->beto_ave, $hpmsalen->beto_std, 
		    $hpmsalen->inso_ave,  $hpmsalen->inso_std,  $hpmsalen->delo_ave, $hpmsalen->delo_std, 
		    $hpmsalen->ttro_ave,  $hpmsalen->ttro_std,  $hpmsalen->apbo_ave, $hpmsalen->apbo_std);
    init_paramhisto($hpsqleng->N, $hpsqleng->k, $hpsqleng->toto, 
		    $hpsqleng->likeo_ave, $hpsqleng->likeo_std, $hpsqleng->ablo_ave, $hpsqleng->ablo_std, 
		    $hpsqleng->alpo_ave,  $hpsqleng->alpo_std,  $hpsqleng->beto_ave, $hpsqleng->beto_std, 
		    $hpsqleng->inso_ave,  $hpsqleng->inso_std,  $hpsqleng->delo_ave, $hpsqleng->delo_std, 
		    $hpsqleng->ttro_ave,  $hpsqleng->ttro_std,  $hpsqleng->apbo_ave, $hpsqleng->apbo_std);
    init_paramhisto($hpsqlena->N, $hpsqlena->k, $hpsqlena->toto, 
		    $hpsqlena->likeo_ave, $hpsqlena->likeo_std, $hpsqlena->ablo_ave, $hpsqlena->ablo_std, 
		    $hpsqlena->alpo_ave,  $hpsqlena->alpo_std,  $hpsqlena->beto_ave, $hpsqlena->beto_std, 
		    $hpsqlena->inso_ave,  $hpsqlena->inso_std,  $hpsqlena->delo_ave, $hpsqlena->delo_std, 
		    $hpsqlena->ttro_ave,  $hpsqlena->ttro_std,  $hpsqlena->apbo_ave, $hpsqlena->apbo_std);

    $$case_ref->{"Case::hfo"}      = $hfo;
    $$case_ref->{"Case::hmsalen"}  = $hmsalen;
    $$case_ref->{"Case::hsqleng"}  = $hsqleng;
    $$case_ref->{"Case::hsqlena"}  = $hsqlena;

    $$case_ref->{"Case::hpfo"}     = $hpfo;
    $$case_ref->{"Case::hpmsalen"} = $hpmsalen;
    $$case_ref->{"Case::hpsqleng"} = $hpsqleng;
    $$case_ref->{"Case::hpsqlena"} = $hpsqlena;
}


sub sq_array_table {
    my ($case_ref) = @_;

    my $case = $$case_ref;

    my $nhtree = $case->treenh;
    my $name;
    my $idx = 0;

    my @nhtree = split(/\,/,$nhtree);
    for (my $n = 0; $n <= $#nhtree; $n ++) {
	if ($nhtree[$n] =~ /([^\(\)\,\:]+)\:/) { 
	    $name = $1;
	    ${$case->sqarray}[$idx++] = $name;
	}
	else { print " you should've found a taxon!\n"; die; }
    }

    $$case_ref = $case;
    
}

sub sq_hash_table {
    my ($case_ref) = @_;

    my $case = $$case_ref;

    my $nhtree = $case->treennh;
    my $name;
    my $idx = 0;

    my @nhtree = split(/\,/,$nhtree);
    for (my $n = 0; $n <= $#nhtree; $n ++) {
	if ($nhtree[$n] =~ /([^\(\)\,\:]+)\:/) { 
	    $name = $1;
	    ${$case->sqhash}{$name} = $idx++;
	}
	else { print " you should've found a taxon!\n"; die; }
    }

    if ($verbose) {
	print "\nnames hash\n";
	foreach my $key (keys(%{$$case_ref->sqhash})) {
	    
	    print "$key ${$case->sqhash}{$key}\n";
	}
    }

    $$case_ref = $case;
    
}

sub tree_abl {
    my ($tree) = @_;
    
    my $ntaxa = $tree->{"Tree::ntaxa"};
    my $nnode = ($ntaxa > 1)? $ntaxa-1 : $ntaxa;
    my $nbranch = 2*$nnode; # it's a binary tree

    my $abl = tree_tbl($tree)/$nbranch;

    if ($verbose) {
	print_tree($tree);
	printf "\nTBL %f\n", tree_tbl($tree);
    }
          
    return $abl;
}

sub tree_copy {
    my ($treecp_ref, $tree) = @_;
    
    my $treecp = Tree->new();
    
    my $ntaxa = $tree->{"Tree::ntaxa"};
    my $nnode = ($ntaxa > 1)? $ntaxa-1 : $ntaxa;
    my $nbranch = 2*$nnode; # it's a binary tree
    
    #copy 
    $treecp->ntaxa($ntaxa);
    
    for (my $n = 0; $n < $nnode; $n ++) {
	${$treecp->left}[$n]   = ${$tree->left}[$n];
	${$treecp->right}[$n]  = ${$tree->right}[$n];
	${$treecp->parent}[$n] = ${$tree->parent}[$n];
	
	${$treecp->ld}[$n] = ${$tree->ld}[$n];
	${$treecp->rd}[$n] = ${$tree->rd}[$n];
    }
    
    for (my $x = 0; $x < $ntaxa; $x ++) {
	${$treecp->taxaparent}[$x] = ${$tree->taxaparent}[$x];
    }
    
    $$treecp_ref = $treecp;
}


sub tree_rescale {
    my ($tree_ref, $treenh_ref, $scale) = @_;

    my $tree = $$tree_ref;
    my $treenh;

    tree_target_abl(\$tree, $scale, $verbose);
    
    # now rewrite the NH version of the tree
    tree_WriteNewick($tree, \$treenh);

    $$tree_ref   = $tree;
    $$treenh_ref = $treenh;
}


sub tree_rescale_dnaml {
    my ($tree_ref, $treenh_ref, $phylipoutfile, $phylipmode, $target_alp, $target_bet, $target_abln) = @_;

    my $tree = $$tree_ref;
    my $treenh;
    my $scale = 1.0;

    my $fracchange;
    my $fracchange_mutations;
    my $target_fracchange_mutations;
    my $like = -1.;
    my $alp  = -1.;
    my $bet  = -1.;
    my $ins  = -1.;
    my $del  = -1.;
    my $ttr  = -1.;
    my $apb  = -1.;
    my $frqa = -1.;
    my $frqc = -1.;
    my $frqg = -1.;
    my $frqt = -1.;
    my $frqo = -1.;

    parse_phylipoutfile($phylipmode, $phylipoutfile, \$like, 
			\$frqa, \$frqc, \$frqg, \$frqt, \$frqo, 
			\$alp, \$bet, \$ins, \$del, \$ttr, \$apb, 0);
    
    if ($alp < 0 || $bet < 0 || $ins < 0 || $del < 0) {
	print "tree_rescale_dnaml_erate(): did not get the right rate parameters: alpha $alp beta $bet ins $ins del $del\n"; 
	die;
    }
    if ($frqa < 0 || $frqc < 0 || $frqg < 0 || $frqt < 0|| $frqo < 0) {
	print "tree_rescale_dnaml_erate(): did not get the right frequencies\n"; 
	die;
    }

    # calculate the overal rate of changes
    $fracchange                  = fracchange_erate($alp, $bet, $ins, $del, $frqa, $frqc, $frqg, $frqt, $frqo);
    $fracchange_mutations        = fracchange($alp, $bet, $frqa, $frqc, $frqg, $frqt);
    $target_fracchange_mutations = fracchange($target_alp, $target_bet, $frqa, $frqc, $frqg, $frqt);

    # correct for the indel contribution
    if ($fracchange > 0) { $scale *= $fracchange_mutations/$fracchange; }
    if (0) {
	if ($verbose) { 
	    print "\n\nSCALE ttr $ttr $scale frachange $fracchange fracchange_mutattions $fracchange_mutations fracchange_target $target_fracchange_mutations\n"; 
	}
	tree_scale_branch(\$tree, $scale);
    }
    else {
	$scale = $target_abln;
	tree_target_abl(\$tree, $scale, $verbose);
    }
    
    # now rewrite the NH version of the tree
    tree_WriteNewick($tree, \$treenh);

    $$tree_ref   = $tree;
    $$treenh_ref = $treenh;
}

# given a rooted tree, I reroot it so that if follows
# the convention that the left child of the root node is taxon "0".
# this is necessary two compare two trees.
sub tree_reroot {
    my ($tree_ref) = @_;

    my $tree  = $$tree_ref;
    my $tree2 = Tree->new(); 

    my $ntaxa = $tree->{"Tree::ntaxa"};
    my $parent;
    my $newparent;
    my $idx;
    my @us;
    my @ds;
    my $u;
    my $d;

    if ($verbose) { print_tree($tree); }

    if (${$tree->left}[0]  == 0) { return; }
    if (${$tree->right}[0] == 0) { 
	my $x = ${$tree->left}[0];
	${$tree->left}[0]  = 0;
	${$tree->right}[0] = $x;
	return; 
    }

    $tree2->{"Tree::ntaxa"} = $ntaxa;

    ${$tree2->left}[0]   = 0;
    ${$tree2->parent}[0] = 0;

    # find the branch where the 0 taxon is
    #
    # create the new root with taxon zero as left child
    #
    for ($idx = 1; $idx < $ntaxa-1; $idx++) {
	if (${$tree->left}[$idx] == 0) {
	    ${$tree2->right}[0] = $idx;
	    ${$tree2->rd}   [0] = 0.5 * ${$tree->ld}[$idx];
	    ${$tree2->ld}   [0] = 0.5 * ${$tree->ld}[$idx];

	    ${$tree2->right}[$idx] = ${$tree->right}[$idx];
	    ${$tree2->rd}[$idx]    = ${$tree->rd}[$idx];
	    last;
	}
	if (${$tree->right}[$idx] == 0) {
	    ${$tree2->right}[0] = $idx;
	    ${$tree2->rd}   [0] = 0.5 * ${$tree->rd}[$idx];
	    ${$tree2->ld}   [0] = 0.5 * ${$tree->rd}[$idx];

	    ${$tree2->right}[$idx] = ${$tree->left}[$idx];
	    ${$tree2->rd}[$idx]    = ${$tree->ld}[$idx];

	    last;
	}

    }
    ${$tree2->parent}[$idx] = 0;

    # if the rigth node of the new root is a internal node, add to 
    # the descent stack. 
    if (${$tree2->right}[$idx] > 0) { push(@ds, ${$tree2->right}[$idx]); }

    # now go up
    #
    $parent = ${$tree->parent}[$idx];
    if ($parent > 0) {
	${$tree2->parent}[$parent] = $idx;
	${$tree2->left}[$idx]      = $parent;
	if (${$tree->left}[$parent] == $idx) {
	    ${$tree2->ld}[$idx] = ${$tree->ld}[$parent];
	}
	elsif (${$tree->right}[$parent] == $idx) {
	    ${$tree2->ld}[$idx] = ${$tree->rd}[$parent];
	}
	else { print "what are you?\n"; die; }
	push(@us, $parent);
    }
    else {
	if (${$tree->left}[$parent] == $idx) { 
	    ${$tree2->left}[$idx] = ${$tree->right}[$parent]; 
	}
	elsif (${$tree->right}[$parent] == $idx) { 
	    ${$tree2->left}[$idx] = ${$tree->left}[$parent];  
	}
	else { print "what are you?\n"; die; }
	if (${$tree2->left}[$idx] > 0) { push(@ds, ${$tree2->left}[$idx]); }
	${$tree2->ld}[$idx] = ${$tree->ld}[$parent] + ${$tree->rd}[$parent];  
	${$tree2->parent}[${$tree2->left}[$idx]] = $idx; 
    }
    

    # parse the u stack
    #
    while ($u = pop(@us))
    {
	$parent    = ${$tree->parent}[$u];
	$newparent = ${$tree2->parent}[$u];

	my $isl = -1;
	if    (${$tree->right}[$u] == $newparent) { $isl = 0; }
	elsif (${$tree->left}[$u]  == $newparent) { $isl = 1; }
	if ($isl < 0) {
	    print "what are you?\n"; die;
	}

	if ($isl == 1) {
	    ${$tree2->right}[$u] = ${$tree->right}[$u];
	    ${$tree2->rd}[$u]    = ${$tree->rd}[$u];
	}
	else {
	    ${$tree2->right}[$u] = ${$tree->left}[$u];
	    ${$tree2->rd}[$u]    = ${$tree->ld}[$u];
	}
	if (${$tree2->right}[$u] > 0) { 
	    ${$tree2->parent}[${$tree2->right}[$u]] = $u;
	    push(@ds, ${$tree2->right}[$u]); 
	}
	if ($parent > 0) {
	    ${$tree2->parent}[$parent] = $u;
	    ${$tree2->left}[$u]        = $parent;
	    if (${$tree->left}[$parent]  == $u) {
		${$tree2->ld}[$u] = ${$tree->ld}[$parent];
	    }
	    elsif (${$tree->right}[$parent] == $u) {
		${$tree2->ld}[$u] = ${$tree->rd}[$parent];
	    }
	    else { print "what are you?\n"; die; }
	    push(@us, $parent);	    
	}
	else {
	    if (${$tree->left}[$parent]  == $u) { 
		${$tree2->left}[$u] = ${$tree->right}[$parent]; 
	    }
	    elsif (${$tree->right}[$parent] == $u) { 
		${$tree2->left}[$u] =  ${$tree->left}[$parent];  
	    }
	    else { print "what are you?\n"; die; }
	    if (${$tree2->left}[$u] > 0) { push(@ds, ${$tree2->left}[$u]); }
	    ${$tree2->ld}[$u] =  ${$tree->ld}[$parent] + ${$tree->rd}[$parent]; 
	    ${$tree2->parent}[${$tree2->left}[$u]] = $u; 
	}
    }    

    # parse the descendent stack
    #
    while ($d = pop(@ds))
    {
	if (${$tree->parent}[$d] > 0) {
	    ${$tree2->parent}[$d] = ${$tree->parent}[$d];
	} 
	${$tree2->right}[$d]  = ${$tree->right}[$d]; 
	${$tree2->left}[$d]   = ${$tree->left}[$d]; 
	${$tree2->rd}[$d]     = ${$tree->rd}[$d]; 
	${$tree2->ld}[$d]     = ${$tree->ld}[$d]; 

	if (${$tree2->right}[$d] > 0) { push(@ds, ${$tree2->right}[$d]); }
	if (${$tree2->left}[$d]  > 0) { push(@ds, ${$tree2->left}[$d]);  }
    }

    tree_SetTaxaParents(\$tree2);
    
    if ($verbose) { print_tree($tree2); }

    # paranoia
    if (${$tree2->parent}[$idx] != 0) {
	print "bad rearranged parent of $idx is not root\n"; die;
    }

    # the new tree has to have the same total branch lenght 
    if (abs(tree_tbl($tree) - tree_tbl($tree2)) > 0.000001) {
	print "bad rearranged tree ", tree_tbl($tree), " ", tree_tbl($tree2), "\n"; die;
    }
    # only one of the internal branches can have the root as parent
    my $c = 0;
    for ($idx = 1; $idx < $ntaxa-1; $idx++) {
	if (${$tree2->parent}[$idx] == 0) { $c ++; }
	if ($idx == ${$tree2->parent}[$idx]) {
	    print "a node cannot be its own parent\n"; die;
	}
    }
    if ($c != 1) {
	print "bad rearranged tree c=$c\n"; die;
    }
    $$tree_ref = $tree2;
}

# /* Function:  esl_tree_RenumberNodes()
#  * Synopsis:  Assure nodes are numbered in preorder.
#  * Incept:    SRE, Fri Oct 27 09:33:26 2006 [Janelia]
#  *
#  * Purpose:   Given a tree <T> whose internal nodes might be numbered in
#  *            any order, with the sole requirement that node 0 is the
#  *            root; renumber the internal nodes (if necessary) to be in Easel's
#  *            convention of preorder traversal. No other aspect of <T> is
#  *            altered (including its allocation size).
#  *
#  * Returns:   <eslOK> on success.
#  *
#  * Throws:    <eslEMEM> on allocation failure.
#  *
#  * Xref:      STL11/77
#  */
sub tree_renumber_nodes {
    my ($tree_ref) = @_;
    
    my $tree  = $$tree_ref;
    my $tree2 = Tree->new(); 
    my @map;
    my @vs;
    my $v;
    my $new;
    my $needs_rearranging = 0;
    my $ntaxa = $tree->ntaxa;

    # Pass 1. Preorder traverse of T by its children links;
    #         construct map[old] -> new.
    #
    
    push(@vs, $ntaxa);
    $new = 0;
    while ($v = pop(@vs))
    {
	$v -= $ntaxa;
	if ($v != $new) { $needs_rearranging = 1; }
	$map[$v] = $new++;
	
	if (${$tree->right}[$v] > 0) { push(@vs, $ntaxa+${$tree->right}[$v]); } 
	if (${$tree->left}[$v]  > 0) { push(@vs, $ntaxa+${$tree->left}[$v]);  } 
    }
    if ($needs_rearranging == 0) { return; }
    
    # Pass 2. Construct the guts of correctly numbered new T2.
    #         (traversal order doesn't matter here)
    #
    $tree2->{"Tree::ntaxa"} = $ntaxa;
    
    for ($v = 0; $v < $ntaxa-1; $v++)
    {
	${$tree2->parent}[$map[$v]] = $map[${$tree->parent}[$v]];
	if (${$tree->left}[$v]  > 0) { ${$tree2->left}[$map[$v]]  = $map[${$tree->left}[$v]]; } # internal nodes renumbered... 
	else                         { ${$tree2->left}[$map[$v]]  = ${$tree->left}[$v];       } # ...taxon indices unchanged 
	if (${$tree->right}[$v] > 0) { ${$tree2->right}[$map[$v]] = $map[${$tree->right}[$v]];}
	else                         { ${$tree2->right}[$map[$v]] = ${$tree->right}[$v];      }
	${$tree2->ld}[$map[$v]] = ${$tree->ld}[$v];
	${$tree2->rd}[$map[$v]] = ${$tree->rd}[$v];
  
      if (${$tree->left}[$v]  <= 0) { ${$tree2->taxaparent}[-${$tree->left}[$v]]  = $map[$v]; }
      if (${$tree->right}[$v] <= 0) { ${$tree2->taxaparent}[-${$tree->right}[$v]] = $map[$v]; }

    }

    $$tree_ref = $tree2;
}

# tree_simulate
#
#/* Function:  esl_tree_Simulate()
# * Synopsis:  Generate a random rooted ultrametric tree.
# * Incept:    SRE, Mon Oct  2 11:36:22 2006 [Janelia]
# *
# * Purpose:   Generate a random rooted ultrametric tree of <N> taxa,
# *            using the algorithm of Kuhner and Felsenstein (1996).
# *            
# *            The branch lengths are generated by choosing <N-1>
# *            exponentially distributed split times, with decreasing
# *            expectations of $\frac{1}{2},\frac{1}{3}..\frac{1}{N}$
# *            as the simulation proceeds from the root. Thus the
# *            total expected branch length on the tree is
# *            $\sum_{k=2}^{N} \frac{1}{k}$.
# *
# * Args:      r     - random number source
# *            N     - number of taxa (>= 2)
# *            ret_T - RETURN: sampled tree
# *
# * Returns:   <eslOK> on success, and the new tree is allocated
# *            here and returned via <ret_tree>; caller is 
# *            responsible for free'ing it.
# *
# * Throws:    <eslEMEM> on allocation failure, in which case
# *            the <ret_T> is returned <NULL>.
# *
# * Xref:      STL11/65.
# */
sub tree_simulate {

    my ($case_ref) = @_;
 
    my $ntaxa;
    my $abln;
    my $ablu;
    my $treen;
    my $treeu;
    my $treennh;
    my $treeunh;

    $ntaxa = $$case_ref->{"Case::sqnum"};
    $abln  = $$case_ref->{"Case::abln"}; # normalized average branch length
    $ablu  = $$case_ref->{"Case::ablu"}; # unnormalized average branch length

    # simulate the tree structure
    tree_simulate_really($ntaxa, $abln, \$treen);

    # copy the normalized tree, and scale the branch lengths 
    # to convert it in an unnormalized tree
    tree_copy(\$treeu, $treen);
    tree_target_abl(\$treeu, $ablu);

    # paranoia
    if (abs(tree_abl($treen) - $abln) > 0.00001) { print "treen is not properly scaled\n"; }
    if (abs(tree_abl($treeu) - $ablu) > 0.00001) { print "treeu is not properly scaled\n"; }

    # add to case
    $$case_ref->{"Case::treen"} = $treen;
    $$case_ref->{"Case::treeu"} = $treeu;

    # create the treenh string
    tree_WriteNewick($treen, \$treennh, \%{$$case_ref->sqhash});
    tree_WriteNewick($treeu, \$treeunh, \%{$$case_ref->sqhash});
    if (1) { 
	print "treennh:\n$treennh\n"; 
	print_tree($treen); 
	print "treeunh:\n$treeunh\n"; 
	print_tree($treeu);
    }
    if ($verbose) {
	print "\nnames hash\n";
	foreach my $key (keys(%{$$case_ref->sqhash})) {
	    
	    print "$key ${$case->sqhash}{$key}\n";
	}
    }

    # add to case
    $$case_ref->{"Case::treennh"} = $treennh;
    $$case_ref->{"Case::treeunh"} = $treeunh;

}

sub tree_scale_branch {
    my ($tree_ref, $scale) = @_;
    
    my $tree = $$tree_ref;
    my $ntaxa = $tree->{"Tree::ntaxa"};
    my $nnode = ($ntaxa > 1)? $ntaxa-1 : $ntaxa;
    my $nbranch = 2*$nnode; # it's a binary tree
    
    # do the scaling
    for (my $n = 0; $n < $nnode; $n ++) {
	${$tree->ld}[$n] *= $scale;
	${$tree->rd}[$n] *= $scale;
    }
    # make sure it is writen in decimal format
    for (my $n = 0; $n < $nnode; $n ++) {
	decimal(\${$tree->ld}[$n]);
	decimal(\${$tree->rd}[$n]);
    }
    
    $$tree_ref = $tree;
}

sub tree_target_abl {
    my ($tree_ref, $abl, $verbose) = @_;
    
    my $tol = 0.00001;
    my $tbl;
    my $scale = 1.0;
    my $tree = $$tree_ref;
    
    my $ntaxa = $tree->{"Tree::ntaxa"};
    my $nnode = ($ntaxa > 1)? $ntaxa-1 : $ntaxa;
    my $nbranch = 2*$nnode; # it's a binary tree
    
    $tbl = tree_tbl($tree);
    
    # scaling factor
    if ($tbl > 0) { $scale *= $abl * $nbranch / $tbl; }
    
    # do the scaling
    tree_scale_branch(\$tree, $scale);

    if ($verbose) { 
	printf "original abl %f new abl %f scale %f\n", $tbl/$nbranch, $abl, $scale; 
    }

    #calculate the new tbl
    $tbl = tree_tbl($tree);
    if ($tbl > 0 &&
	abs($tbl - $abl * $nbranch) > $tol) 
    { 
	printf "\ntree_target_abl(): bad rescaling tbl=%f abl=%f abl*nbranch=%f\n", $tbl, $abl, $abl*$nbranch; die; 
    }

    $$tree_ref = $tree;
}

sub tree_simulate_really {
    my ($ntaxa, $abl, $tree_ref) = @_;
    
    if ($ntaxa < 2) { print "tree_simulate_realy(): need more than one taxon\n"; die; }
    
    # Kuhner/Felsenstein uses a list of active branches,
    # which we implement by tracking the index of the parent
    # node (in <branchpapa>) and a 0/1 flag (in <branchside>)
    # for the branch to the left vs. right child.
    my $tree = Tree->new(); 
    my @branchpapa;
    my @branchside;
    my $nactive;
    my $node;
    my $bidx;
    my $d;
    my $x;
    
    $tree->{"Tree::ntaxa"} = $ntaxa;
    
    # Initialize: add two branches from the root
    # onto the active list, and set internal node
    # counter to start at 1.
    $branchpapa[0] = 0;   $branchside[0] = 0;
    $branchpapa[1] = 0;   $branchside[1] = 1;
    $nactive = 2;
    $node    = 1;			
    
    ${$tree->parent}[0] = 0;

    # Algorithm proceeds by iterating:
    #     1. choose random time <d> from exponential(1/nactive)
    #     2. choose random active branch, <bidx>
    #     3. add new <node> to active branch at length d
    #     4. add d to all other active branches      
    #     5. delete the old parent branch from the active list,
    #        add the two new child branches to the active list
    while ($nactive < $ntaxa)
    {
	$x = rnd_UniformPositive();
	$d    = $nactive * (-log($x));
	$bidx = int(rand($nactive));
	${$tree->parent}[$node] = $branchpapa[$bidx];
	
	if ($branchside[$bidx] == 0) {
	    ${$tree->left}[$branchpapa[$bidx]]  = $node;
	    ${$tree->ld  }[$branchpapa[$bidx]] += $d;
	} else {
	    ${$tree->right}[$branchpapa[$bidx]]  = $node;
	    ${$tree->rd   }[$branchpapa[$bidx]] += $d;
	}
	
	swap(\$branchpapa[$bidx], \$branchpapa[$nactive-1]);
	swap(\$branchside[$bidx], \$branchside[$nactive-1]);
	for ($bidx = 0; $bidx < $nactive-1; $bidx++) {
	    if ($branchside[$bidx] == 0) { ${$tree->ld}[$branchpapa[$bidx]] += $d; }
	    else                         { ${$tree->rd}[$branchpapa[$bidx]] += $d; }
	}
	
	# delete the branch at nactive-1 that we just added to;
	# replace it with two new branches
	#
	$branchpapa[$nactive-1]  = $node;  $branchside[$nactive-1] = 0;
	$branchpapa[$nactive]    = $node;  $branchside[$nactive]   = 1;
	$node++;
	$nactive++;
    }
    
    # Terminate by adding the N taxa to the N active branches.
    #
    $x = rnd_UniformPositive();
    $d = $ntaxa * (-log($x));
    for ($bidx = 0; $bidx < $ntaxa; $bidx++)
    {
	if ($branchside[$bidx] == 0) {
	    ${$tree->left}[$branchpapa[$bidx]]  = -$bidx; #taxa indices stored as neg numbers 
	    ${$tree->ld  }[$branchpapa[$bidx]] += $d;
	} else {
	    ${$tree->right}[$branchpapa[$bidx]]  = -$bidx;
	    ${$tree->rd   }[$branchpapa[$bidx]] += $d;
	}
    }
    tree_SetTaxaParents(\$tree);

    # reroot the tree so that the left child of the
    # root is taxon "0"
    tree_reroot(\$tree);
    tree_renumber_nodes(\$tree);

    # Impose of T->rd[0] = 0
    #
    ${$tree->ld}[0] += ${$tree->rd}[0];
    ${$tree->rd}[0]  = 0.0;

    # rescale the branch lengths to a desired abl (average branch lenth)
    tree_target_abl(\$tree, $abl);
    
    $$tree_ref = $tree;    
}

# Function:  esl_tree_SetTaxaParents()
# Incept:    SRE, Fri Sep 22 13:39:49 2006 [Janelia]
#
# Purpose:   Constructs the <T->taxaparent[]> array in the tree
#            structure <T>, by an O(N) traversal of the tree.
#            Upon return, <T->taxaparent[i]> is the index
#            of the internal node that taxon <i> is a child of.
#
# Args:      T   - the tree structure to map
#
# Returns:   <eslOK> on success.
#
# Throws:    <eslEMEM> on internal allocation error. In this case, the tree is 
#            returned unchanged.
#
# Xref:      STL11/63
#
sub tree_SetTaxaParents {
    
    my ($tree_ref) = @_;
    
    my $tree = $$tree_ref;
    
    for (my $i = 0; $i < $tree->ntaxa-1; $i++)	# traversal order doesn't matter 
    {
	if (${$tree->left}[$i]  <= 0) { ${$tree->taxaparent}[-(${$tree->left}[$i])]  = $i; }
	if (${$tree->right}[$i] <= 0) { ${$tree->taxaparent}[-(${$tree->right}[$i])] = $i; }
    }

    $$tree_ref = $tree;
}

sub tree_tbl {
    my ($tree) = @_;
    
    my $tbl = 0.0;
    
    my $ntaxa = $tree->{"Tree::ntaxa"};
    my $nnode = ($ntaxa > 1)? $ntaxa-1 : $ntaxa;
    my $nbranch = 2*$nnode; # it's a binary tree
    
    #calculate the tbl
    for (my $n = 0; $n < $nnode; $n ++) {
	$tbl += ${$tree->ld}[$n];
	$tbl += ${$tree->rd}[$n];
    }
    
    return $tbl;
}


# Function:  esl_tree_WriteNewick()
# Incept:    SRE, Fri Oct  6 14:35:51 2006 [Janelia]
#
# Purpose:   Writes tree <T> to stream <fp> in Newick format.
#  
#            Certain options are set in <T> to control output style.
#            If <T->show_unrooted> is <TRUE>, <T> is printed as an
#            unrooted tree starting with a trifurcation, a la PHYLIP
#            format (default=<FALSE>). If <T->show_node_labels> is
#            <TRUE>, then labels are shown for internal nodes, if any
#            are available (default=<TRUE>). If
#            <T->show_branchlengths> is <TRUE>, then branch lengths
#            are shown, as opposed to just printing a labeled
#            topology (default=<TRUE>). If
#            <T->show_root_branchlength> is also <TRUE>, then a 0.0
#            branchlength is shown to the root node, a la Hein's
#            TreeAlign Newick format (default=<FALSE>). If
#            <T->show_quoted_labels> is <TRUE>, then all labels are
#            shown in Newick's quoted format, as opposed to only
#            using quoted labels where necessary (default=<FALSE>).
#
# Returns:   <eslOK> on success.
#
# Throws:    <eslEMEM> on allocation error.
#            <eslEINCONCEIVABLE> on internal error.
#
# Xref:      STL11/74
#
sub tree_WriteNewick {
    
    my ($tree, $treenh_ref, $sqhash_ref) = @_;
    
    my $treenh = "";
    my @vs;
    my @cs;
    my $v;
    my $c;

    # Initialization.
    # Push a trifurcation (swallowing the right internal node) if unrooted;
    # else push the first bifurcation.
    # 
    # When we push a trifurcation, the branch lengths will come out fine
    # on output, if the tree followed the correct convention of having
    # a T->rd[0] = 0.0.
    #
    $treenh .= "(";
    
    if (${$tree->right}[0] > 0)
    {
	$v = ${$tree->right}[0];
	push(@cs, "x");
	push(@vs, ${$tree->right}[$v]);
	push(@cs, ",");
	push(@cs, "x");
	push(@vs, ${$tree->left}[$v]);
    }
    else 
    {
	push(@cs, "x");
	push(@vs, ${$tree->right}[0]);
    }
    push(@cs, ",");
    push(@cs, "x");
    push(@vs, ${$tree->left}[0]);
    
    
    # Main iteration. Pop off stacks 'til they're empty.
    #
    while ($c = pop(@cs))
    {
	if ($c eq ",") { $treenh .= ","; next; } # comma doesn't have a v stacked with it 
	
	$v = pop(@vs);
	
	if ($c eq "x") {	# a subtree, which could be a node or a taxon: 
	    if ($v > 0)		# internal node 1..N-2
	    {
		$treenh .= "("; 
		push(@cs, ")");
		push(@vs, $v);
		push(@cs, "x");
		push(@vs, ${$tree->right}[$v]);
		push(@cs, ",");
		push(@cs, "x");
		push(@vs, ${$tree->left}[$v]);
	    }
	    else			# taxon -(N-1)..0 
	    { 	    # -v below to convert taxon code to 0..N-1 
		newick_write_taxonlabel(\$treenh, $tree, -$v, $sqhash_ref);
		newick_write_branchlength(\$treenh, $tree,  $v);
	    }
	}
	elsif ($c eq "\)") { # closing an internal node. v > 0 is a node code.
	    $treenh .= ")"; 
	    newick_write_branchlength(\$treenh, $tree, $v);
	}
	else {
	    print "bad state code $c\n"; die;
	}            
    }
    
    # Termination
    #
    $treenh .= ");"; 

    $$treenh_ref = $treenh;
}

# newick_write_branchlength()
#    Writes the branch length #to# <v>.
#    If <v> is negative, it's a leaf; if <v> is positive, it's an internal node.
#    You can't pass the root node 0 to this. 0 always means taxon 0.
#    There is no branch to the root node.
#
sub newick_write_branchlength {
    my ($treenh_ref, $tree, $v) = @_;
    
    my $branchlength;
    
    if ($v <= 0)			# leaf 
    {
	if    (${$tree->left} [${$tree->taxaparent}[-$v]] == $v) { 
	    $branchlength = decimal(\${$tree->ld}[${$tree->taxaparent}[-$v]]); }
	elsif (${$tree->right}[${$tree->taxaparent}[-$v]] == $v) { 
	    $branchlength = decimal(\${$tree->rd}[${$tree->taxaparent}[-$v]]); }
	else                                                     { 
	    print "Can't find branch length\n"; die; }
    }
    else				# internal node 
    {
	if    (${$tree->left }[${$tree->parent}[$v]] == $v) { $branchlength = decimal(\${$tree->ld}[${$tree->parent}[$v]]); }
	elsif (${$tree->right}[${$tree->parent}[$v]] == $v) { $branchlength = decimal(\${$tree->rd}[${$tree->parent}[$v]]); }
	else                                                { print "Can't find branch length\n"; die; }
    }
    
    $$treenh_ref .= ":$branchlength";
}

# newick_write_taxonlabel():
#    Print the label for taxon <v> to stream <fp>.
#    Tries to print label as an unquoted label, then
#    as a quoted label, (then fails).
#    If label isn't available, does nothing.
#    If label contains invalid characters, throws <eslECORRUPT>.
#
sub newick_write_taxonlabel {
    my ($treenh_ref, $tree, $v, $sqhash_ref) = @_;

    $$treenh_ref .= "ta$v";
    $sqhash_ref->{"ta$v"} = $v;    
}

sub write_one_histogram {
    my ($globals_ref, $case, $h, $iswc, $title, $xlabel, $key) = @_;
   
    my $N = $h->N;
    my $k = $h->k;
    my $dim = $N * $k;
    my $hfile;
    
    if ($iswc == 1) { $hfile = $h->{"Histogram::histofilewc"}; }
    else            { $hfile = $h->{"Histogram::histofile"}; }

    # start file by print header
    print_supercase($hfile, $case);    

    # print the global stats (so far if a working copy)
    print_globals($hfile, $$globals_ref);
    
    open(HIS, ">>".$hfile);
    for (my $i=0; $i<=$dim; $i++) { 
	
	my $len = $i/$k;
	
	my $tpx  = ${$h->tpx}[$i];
	my $tpe  = ${$h->tpe}[$i];
	my $tpo  = ${$h->tpo}[$i];
	my $tpxe = ${$h->tpxe}[$i];
	my $tpxo = ${$h->tpxo}[$i];

	my $totalx  = ${$h->totx}[$i];
	my $totale  = ${$h->tote}[$i];
	my $totalo  = ${$h->toto}[$i];
	my $totalxe = ${$h->totxe}[$i];
	my $totalxo = ${$h->totxo}[$i];

	my $fracx  = ($totalx  > 0)? $tpx/$totalx : 0;
	my $frace  = ($totale  > 0)? $tpe/$totale : 0;
	my $fraco  = ($totalo  > 0)? $tpo/$totalo : 0;
	my $fracxe = ($totalxe > 0)? $tpxe/$totalxe : 0;
	my $fracxo = ($totalxo > 0)? $tpxo/$totalxo : 0;

	my $dist1x_ave = ${$h->dist1x_ave}[$i];
	my $dist1x_std = ${$h->dist1x_std}[$i];
	calculate_averages(\$dist1x_ave, \$dist1x_std, $totalx);

	my $dist1e_ave = ${$h->dist1e_ave}[$i];
	my $dist1e_std = ${$h->dist1e_std}[$i];
	calculate_averages(\$dist1e_ave, \$dist1e_std, $totale);

	my $dist1o_ave = ${$h->dist1o_ave}[$i];
	my $dist1o_std = ${$h->dist1o_std}[$i];
	calculate_averages(\$dist1o_ave, \$dist1o_std, $totalo);

	my $dist1xe_ave = ${$h->dist1xe_ave}[$i];
	my $dist1xe_std = ${$h->dist1xe_std}[$i];
	calculate_averages(\$dist1xe_ave, \$dist1xe_std, $totalxe);

	my $dist1xo_ave = ${$h->dist1xo_ave}[$i];
	my $dist1xo_std = ${$h->dist1xo_std}[$i];
	calculate_averages(\$dist1xo_ave, \$dist1xo_std, $totalxo);

	my $dist2x_ave = ${$h->dist2x_ave}[$i];
	my $dist2x_std = ${$h->dist2x_std}[$i];
	calculate_averages(\$dist2x_ave, \$dist2x_std, $totalx);

	my $dist2e_ave = ${$h->dist2e_ave}[$i];
	my $dist2e_std = ${$h->dist2e_std}[$i];
	calculate_averages(\$dist2e_ave, \$dist2e_std, $totale);

	my $dist2o_ave = ${$h->dist2o_ave}[$i];
	my $dist2o_std = ${$h->dist2o_std}[$i];
	calculate_averages(\$dist2o_ave, \$dist2o_std, $totalo);

	my $dist2xe_ave = ${$h->dist2xe_ave}[$i];
	my $dist2xe_std = ${$h->dist2xe_std}[$i];
	calculate_averages(\$dist2xe_ave, \$dist2xe_std, $totalxe);

	my $dist2xo_ave = ${$h->dist2xo_ave}[$i];
	my $dist2xo_std = ${$h->dist2xo_std}[$i];
	calculate_averages(\$dist2xo_ave, \$dist2xo_std, $totalxo);

	my $runtmx_ave = ${$h->runtmx_ave}[$i];
	my $runtmx_std = ${$h->runtmx_std}[$i];
	calculate_averages(\$runtmx_ave, \$runtmx_std, $totalx);

	my $runtme_ave = ${$h->runtme_ave}[$i];
	my $runtme_std = ${$h->runtme_std}[$i];
	calculate_averages(\$runtme_ave, \$runtme_std, $totale);

	my $runtmo_ave = ${$h->runtmo_ave}[$i];
	my $runtmo_std = ${$h->runtmo_std}[$i];
	calculate_averages(\$runtmo_ave, \$runtmo_std, $totalo);

	
	if ($totalx > 0 || $totale > 0 || $totalo > 0) {
	    my $cmdx  = "$tpx\t$totalx\t$fracx\t$dist1x_ave\t$dist1x_std\t$dist2x_ave\t$dist2x_std\t$runtmx_ave\t$runtmx_std";
	    my $cmde  = "$tpe\t$totale\t$frace\t$dist1e_ave\t$dist1e_std\t$dist2e_ave\t$dist2e_std\t$runtme_ave\t$runtme_std";
	    my $cmdo  = "$tpo\t$totalo\t$fraco\t$dist1o_ave\t$dist1o_std\t$dist2o_ave\t$dist2o_std\t$runtmo_ave\t$runtmo_std";
	    my $cmdxe = "$tpxe\t$totalxe\t$fracxe\t$dist1xe_ave\t$dist1xe_std\t$dist2xe_ave\t$dist2xe_std";
	    my $cmdxo = "$tpxo\t$totalxo\t$fracxo\t$dist1xo_ave\t$dist1xo_std\t$dist2xo_ave\t$dist2xo_std";
	    print HIS "$len\t$cmdx\t$cmde\t$cmdo\t$cmdxe\t$cmdxo\n";
	}

    }
   close (HIS);

    if ($seeplots) { gnuplot_histo($hfile, $title, $xlabel, $key); }
}

sub write_one_paramhisto {
    my ($globals_ref, $case, $h, $iswc, $title, $xlabel, $key) = @_;
   
    my $N = $h->N;
    my $k = $h->k;
    my $dim = $N * $k;
    my $hfile;
    
    if ($iswc == 1) { $hfile = $h->{"ParamHisto::histofilewc"}; }
    else            { $hfile = $h->{"ParamHisto::histofile"}; }

    # start file by print header
    print_supercase($hfile, $case);    

    # print the global stats (so far if a working copy)
    print_globals($hfile, $$globals_ref);
    
    open(HIS, ">>".$hfile);
    for (my $i=0; $i<=$dim; $i++) { 
	
	my $len = $i/$k;
	
	my $totalx = ${$h->totx}[$i];
	my $totale = ${$h->tote}[$i];
	my $totalo = ${$h->toto}[$i];

	my $likex_ave = ${$h->likex_ave}[$i];
	my $likex_std = ${$h->likex_std}[$i];
 	calculate_averages(\$likex_ave, \$likex_std, $totalx);

	my $likee_ave = ${$h->likee_ave}[$i];
	my $likee_std = ${$h->likee_std}[$i];
	calculate_averages(\$likee_ave, \$likee_std, $totale);

	my $likeo_ave = ${$h->likeo_ave}[$i];
	my $likeo_std = ${$h->likeo_std}[$i];
	calculate_averages(\$likeo_ave, \$likeo_std, $totalo);

	my $ablx_ave = ${$h->ablx_ave}[$i];
	my $ablx_std = ${$h->ablx_std}[$i];
 	calculate_averages(\$ablx_ave, \$ablx_std, $totalx);

	my $able_ave = ${$h->able_ave}[$i];
	my $able_std = ${$h->able_std}[$i];
	calculate_averages(\$able_ave, \$able_std, $totale);

	my $ablo_ave = ${$h->ablo_ave}[$i];
	my $ablo_std = ${$h->ablo_std}[$i];
	calculate_averages(\$ablo_ave, \$ablo_std, $totalo);

	my $alpx_ave = ${$h->alpx_ave}[$i];
	my $alpx_std = ${$h->alpx_std}[$i];
 	calculate_averages(\$alpx_ave, \$alpx_std, $totalx);

	my $alpe_ave = ${$h->alpe_ave}[$i];
	my $alpe_std = ${$h->alpe_std}[$i];
	calculate_averages(\$alpe_ave, \$alpe_std, $totale);

	my $alpo_ave = ${$h->alpo_ave}[$i];
	my $alpo_std = ${$h->alpo_std}[$i];
	calculate_averages(\$alpo_ave, \$alpo_std, $totalo);

	my $betx_ave = ${$h->betx_ave}[$i];
	my $betx_std = ${$h->betx_std}[$i];
 	calculate_averages(\$betx_ave, \$betx_std, $totalx);
	
	my $bete_ave = ${$h->bete_ave}[$i];
	my $bete_std = ${$h->bete_std}[$i];
	calculate_averages(\$bete_ave, \$bete_std, $totale);
	
	my $beto_ave = ${$h->beto_ave}[$i];
	my $beto_std = ${$h->beto_std}[$i];
	calculate_averages(\$beto_ave, \$beto_std, $totalo);
	
	my $insx_ave = ${$h->insx_ave}[$i];
	my $insx_std = ${$h->insx_std}[$i];
 	calculate_averages(\$insx_ave, \$insx_std, $totalx);

	my $inse_ave = ${$h->inse_ave}[$i];
	my $inse_std = ${$h->inse_std}[$i];
	calculate_averages(\$inse_ave, \$inse_std, $totale);

	my $inso_ave = ${$h->inso_ave}[$i];
	my $inso_std = ${$h->inso_std}[$i];
	calculate_averages(\$inso_ave, \$inso_std, $totalo);

	my $delx_ave = ${$h->delx_ave}[$i];
	my $delx_std = ${$h->delx_std}[$i];
	calculate_averages(\$delx_ave, \$delx_std, $totalx);

	my $dele_ave = ${$h->dele_ave}[$i];
	my $dele_std = ${$h->dele_std}[$i];
	calculate_averages(\$dele_ave, \$dele_std, $totale);

	my $delo_ave = ${$h->delo_ave}[$i];
	my $delo_std = ${$h->delo_std}[$i];
	calculate_averages(\$delo_ave, \$delo_std, $totalo);

	my $ttrx_ave = ${$h->ttrx_ave}[$i];
	my $ttrx_std = ${$h->ttrx_std}[$i];
 	calculate_averages(\$ttrx_ave, \$ttrx_std, $totalx);

	my $ttre_ave = ${$h->ttre_ave}[$i];
	my $ttre_std = ${$h->ttre_std}[$i];
	calculate_averages(\$ttre_ave, \$ttre_std, $totale);

	my $ttro_ave = ${$h->ttro_ave}[$i];
	my $ttro_std = ${$h->ttro_std}[$i];
	calculate_averages(\$ttro_ave, \$ttro_std, $totalo);

	my $apbx_ave = ${$h->apbx_ave}[$i];
	my $apbx_std = ${$h->apbx_std}[$i];
	calculate_averages(\$apbx_ave, \$apbx_std, $totalx);

	my $apbe_ave = ${$h->apbe_ave}[$i];
	my $apbe_std = ${$h->apbe_std}[$i];
	calculate_averages(\$apbe_ave, \$apbe_std, $totale);

	my $apbo_ave = ${$h->apbo_ave}[$i];
	my $apbo_std = ${$h->apbo_std}[$i];
	calculate_averages(\$apbo_ave, \$apbo_std, $totalo);

	if ($totalx > 0 || $totale > 0 || $totalo > 0) {
	    my $cmdx = "$totalx\t$likex_ave\t$likex_std\t$ablx_ave\t$ablx_std\t$alpx_ave\t$alpx_std\t$betx_ave\t$betx_std\t$insx_ave\t$insx_std\t$delx_ave\t$delx_std\t$ttrx_ave\t$ttrx_std\t$apbx_ave\t$apbx_std";
	    my $cmde = "$totale\t$likee_ave\t$likee_std\t$able_ave\t$able_std\t$alpe_ave\t$alpe_std\t$bete_ave\t$bete_std\t$inse_ave\t$inse_std\t$dele_ave\t$dele_std\t$ttre_ave\t$ttre_std\t$apbe_ave\t$apbe_std";
	    my $cmdo = "$totalo\t$likeo_ave\t$likeo_std\t$ablo_ave\t$ablo_std\t$alpo_ave\t$alpo_std\t$beto_ave\t$beto_std\t$inso_ave\t$inso_std\t$delo_ave\t$delo_std\t$ttro_ave\t$ttro_std\t$apbo_ave\t$apbo_std";
	    print HIS "$len\t$cmdx\t$cmde\t$cmdo\n";
	}

    }
    close (HIS);
    
    if ($seeplots) { gnuplot_paramhisto($hfile, $title, $xlabel, $key); }
}

sub write_histograms {
    my ($case, $globals_ref, $iswc) = @_;

    my $meanindelfrq = int($$globals_ref->{"Globalstats::meanindelfreq"}*10000)/100;
    my $stdindelfrq  = int($$globals_ref->{"Globalstats::stdindelfreq"}*10000)/100;
    my $meanpairid   = int($$globals_ref->{"Globalstats::meanpairid"}*100)/100;
    my $stdpairid    = int($$globals_ref->{"Globalstats::stdpairid"}*100)/100;
    my $meanpairmut  = int($$globals_ref->{"Globalstats::meanpairmut"}*100)/100;
    my $stdpairmut   = int($$globals_ref->{"Globalstats::stdpairmut"}*100)/100;
    my $meanpairindl = int($$globals_ref->{"Globalstats::meanpairindl"}*100)/100;
    my $stdpairindl  = int($$globals_ref->{"Globalstats::stdpairindl"}*100)/100;

    my $nsamples_per_tree = $case->{"Case::nmsa"} * (($maxval-$minval)/$valinc + 1);

    my $title  = "Ntaxa:".$case->{"Case::sqnum"};
    $title    .= " ** <branch len> ".$case->{"Case::abln"};
    $title    .= " ** Ntrees ".$case->{"Case::ntree"};
    $title    .= " ** Nsamples/tree ".$nsamples_per_tree;

    my $key  = "indelfraq: $meanindelfrq +/- $stdindelfrq ** ";
    $key    .= "pairwise ID: $meanpairid +/- $stdpairid ** ";
    $key    .= "pairwise MUT: $meanpairmut +/- $stdpairmut";
    $key    .= "pairwise INDL: $meanpairindl +/- $stdpairindl";

    my $xlabel;

    $xlabel = "FRACTION TOTAL INDELS";
    write_one_histogram($globals_ref,  $case, $case->hfo,  $iswc, $title, $xlabel, $key);
    write_one_paramhisto($globals_ref, $case, $case->hpfo, $iswc, $title, $xlabel, $key);

    $xlabel = "ALIGNMENT LENTGH";
    write_one_histogram($globals_ref,  $case, $case->hmsalen,  $iswc, $title, $xlabel, $key);
    write_one_paramhisto($globals_ref, $case, $case->hpmsalen, $iswc, $title, $xlabel, $key);

    $xlabel = "GEOMETRIC MEAN SEQUENCES LENTGH";
    write_one_histogram($globals_ref,  $case, $case->hsqleng,  $iswc, $title, $xlabel, $key);
    write_one_paramhisto($globals_ref, $case, $case->hpsqleng, $iswc, $title, $xlabel, $key);

    $xlabel = "ARITHMETIC MEAN SEQUENCES LENTGH";
    write_one_histogram($globals_ref,  $case, $case->hsqlena,  $iswc, $title, $xlabel, $key);
    write_one_paramhisto($globals_ref, $case, $case->hpsqlena, $iswc, $title, $xlabel, $key);

}

    
sub swap {
    
    my ($a_ref, $b_ref) = @_;
    
    my $a = $$a_ref;
    my $b = $$b_ref;
    
    my $x = $a;
    $a = $b;
    $b = $x;
    
    $$a_ref = $a;
    $$b_ref = $b;
}
