#!/usr/bin/perl -w
#histo_boostrap.pl

use strict;
use Class::Struct;

use vars qw ($opt_b $opt_s);  # required if strict used
use Getopt::Std;
use constant GNUPLOT => '/usr/bin/gnuplot';
getopts ('b:s');


# Print a helpful message if the user provides no input file.
if (!@ARGV) {
        print "usage:  histo_boostrap.pl [options] N histofile1 ... histofileN\n\n";
        print "options:\n";
	print "-b <num> :  number of boostraps [ default given in the filename]\n";
	print "-s       :  see the plots\n";
	exit;
}
my $N = shift;
my @histofile;
for (my $n = 0; $n < $N; $n ++) {
    $histofile[$n] = shift;
}

my $seeplots = 0;
if ($opt_s) { $seeplots = 1; }

my @nboot;
my @NT;
my @ave_bootstrap;
my @std_bootstrap;
my @key;
my @title;
for (my $n = 0; $n < $N; $n ++) {
    parse_histofile ($histofile[$n], \$nboot[$n], \$NT[$n], \$ave_bootstrap[$n], \$std_bootstrap[$n], \$title[$n], \$key[$n], );
}

gnuplot_histo($N, \@histofile, $title[0], \@key);

######################################################################
sub calculate_averages {
    my ($meanval_ref, $stdval_ref, $number) = @_;

    my $mean = $$meanval_ref;
    my $std  = $$stdval_ref;

    if ($number > 1) {
	$mean /= $number;
	
	$std -= $mean*$mean*$number;
	$std /= ($number-1);
	if ($std < 0. && $std> -0.00001) { $std = 0.0; }
	$std  = sqrt($std);
    }
    elsif ($number == 1) {
	$mean /= $number;
	$std   = 0.0;
    }
    else {
	$mean = 0.0;
	$std  = 0.0;
    }

    $$meanval_ref = $mean;
    $$stdval_ref  = $std;

}

sub gnuplot_histo {

    my ($N, $filehisto_ref, $title, $key_ref) = @_;

    my $outplot = "histo.ps";
    my $ylabel;
    my $xlabel = "AVERAGE BOOSTRAP FRACTION";

    open(GP,'|'.GNUPLOT) || die "Gnuplot: $!";

    print GP "set terminal postscript color solid 14\n";
    print GP "set style line 1 lt 1 lw 4\n";
    print GP "set style line 2 lt 2 lw 4\n";
    print GP "set style line 3  lt 3 lw 4\n";
    print GP "set style line 4 lt 4 lw 4\n";+
    print GP "set style line 5 lt 5 lw 4\n";
    print GP "set style line 6 lt 6 lw 4\n";
    print GP "set style line 7 lt 7 lw 4\n";
    print GP "set style line 8 lt 8 lw 4\n";
    print GP "set style line 9 lt 9 lw 4\n";
    print GP "set style line 11 lt 1 lw 2 pt 1 ps 1.2\n";
    print GP "set style line 22 lt 2 lw 2 pt 2 ps 1.2\n";
    print GP "set style line 33 lt 3 lw 2 pt 3 ps 1.2\n";
    print GP "set style line 44 lt 4 lw 2 pt 4 ps 1.2\n";
    print GP "set style line 55 lt 5 lw 2 pt 5 ps 1.2\n";
    print GP "set style line 66 lt 6 lw 2 pt 6 ps 1.2\n";
    print GP "set style line 77 lt 7 lw 2 pt 7 ps 1.2\n";
    print GP "set style line 88 lt 8 lw 2 pt 8 ps 1.2\n";
    print GP "set style line 99 lt 9 lw 2 pt 9 ps 1.2\n";

    print GP "set output '$outplot'\n";
    #print GP "set nokey\n";
    print GP "set xlabel '$xlabel'\n";

    #print GP "set title \"$title\\n\\n$key\"\n";
    print GP "set title '$title'\n";

    $ylabel = "\# TRIALS";
    print GP "set ylabel '$ylabel'\n";

    my $cmd = "";
    for (my $n = 0; $n < $N-1; $n ++) {
	my $m = $n+1;
	$cmd .= "'$filehisto_ref->[$n]' using 1:2  with boxes title '$key_ref->[$n]' ls $m, ";
    }
    $cmd .= "'$filehisto_ref->[$N-1]' using 1:2  with boxes title '$key_ref->[$N-1]' ls $N";

    print "$cmd\n";
    print GP "plot $cmd\n";

    close (GP);

    if ($seeplots) { system ("ggv -landscape -magstep -2 $outplot&\n"); }

}

sub parse_histofile {
    my ($histofile, $nboot_ref, $NT_ref, $ave_bootstrap_ref, $std_bootstrap_ref, $title_ref, $key_ref) = @_;

    my $nboot = 0;
    my $NT = 0;
    my $ave_bootstrap = 0.0;
    my $std_bootstrap = 0.0;
    my $key = "";
    my $title;
    my $NB;
    my $NS;
    my $TB;
    my $TS;

    if    ($histofile =~ /\.(dnaml)\./)          { $key = $1; }
    elsif ($histofile =~ /\.(dnaml-erate\.C)\./) { $key = $1; }
    elsif ($histofile =~ /\.(dnaml-erate)\./)    { $key = $1; }
    
    if ($histofile =~ /^(16S\_[^\_]+)\_/) { $title .= $1; }

    if ($histofile =~ /\.NB(\S+)\.TB(\S+)\.NS(\S+)\.TS(\S+)\.boot([^\.]+)\./) { 
	$NB    = $1; 
	$TB    = $2; 
	$NS    = $3; 
	$TS    = $4; 
	$nboot = $5;
    }

     open(FILE, "$histofile");
     while(<FILE>) {
	 if (/^(\S+)\s+(\S+)$/) { 
	     my $bootstrap = $1;
	     my $ncases    = $2;
	     $ave_bootstrap += $bootstrap * $ncases;
	     $std_bootstrap += $bootstrap * $bootstrap * $ncases;

	     $NT += $ncases; 
	 }
     }
    close(FILE);

    calculate_averages(\$ave_bootstrap, \$std_bootstrap, $NT);

    $ave_bootstrap = int($ave_bootstrap*100)/100;
    $std_bootstrap = int($std_bootstrap*100)/100;

    $title .= " ** N=$NT / Nb=$NB TB=$TB NS=$NS TS=$TS / $nboot boostrap each";
    $key   .= " ** ave bootstrap $ave_bootstrap +\- $std_bootstrap";
  
    $$nboot_ref = $nboot;
    $$NT_ref = $NT;
    $$ave_bootstrap_ref = $ave_bootstrap;
    $$std_bootstrap_ref = $std_bootstrap;
    $$key_ref = $key;
    $$title_ref = $title;
}
