#!/usr/bin/perl -w
#plot_erate.pl

use strict; 
use Class::Struct;

use vars qw ($opt_i $opt_d $opt_l $opt_p $opt_v);  # required if strict used
use Getopt::Std;
use constant GNUPLOT => '/usr/bin/gnuplot';
getopts ('i:d:l:p:v');


# Print a helpful message if the user provides no input file.
if (!@ARGV) {
    print "usage:  plot_erate.pl [options] file\n\n";
    print "-i <num>     :  insertion rate [ l=0.5]\n";
    print "-d <num>     :  deletion rate [ m=0.5]\n";
    print "-l <num>     :  expected length ancestral sequences\n";
    print "-p <num>     :  bernoulli probability parameter [ p=0.99]\n";
    print "-v           :  verbose\n";
    exit;
}
my $file = shift;

my $lambda = 0.5;
my $mu     = 0.5;
if ($opt_i) { $lambda = $opt_i; }
if ($opt_d) { $mu     = $opt_d; }

# one or the other
my $pb     = 0.99;
if    ($opt_p) { $pb  = $opt_p; }
elsif ($opt_l) { $pb  = $opt_l/($opt_l+1); }
if ($pb == 0 || $pb == 1) { printf("bad paramters\n"); die; }

my $verbose = 0;
if ($opt_v) { $verbose = $opt_v; }

my $plotfile = "$file.ps";
plot_erate($lambda, $mu, $plotfile);


sub plot_erate {
    
    my ($ins, $del, $plotfile) = @_;
    
    my $xlabel = "TIME";
    my $ylabel = "";
    my $title  = "insertion rate = $ins,  deletion rate = $del, bernoulli probabiliy parameter = $pb";
    my $key    = "";

    my $tsmall = 0.5;
    my $tmed = 10;
    my $tmax = 1000;

    open(GP,'|'.GNUPLOT) || die "Gnuplot: $!";

    print GP "set terminal postscript color solid 14\n";
    print GP "set style line 1 lt 1 lw 4\n";
    print GP "set style line 2 lt 2 lw 4\n";
    print GP "set style line 3  lt 3 lw 4\n";
    print GP "set style line 4 lt 4 lw 4\n";+
    print GP "set style line 5 lt 5 lw 4\n";
    print GP "set style line 6 lt 6 lw 4\n";
    print GP "set style line 7 lt 7 lw 4\n";
    print GP "set style line 8 lt 8 lw 4\n";
    print GP "set style line 9 lt 9 lw 4\n";
    print GP "set style line 11 lt 1 lw 2 pt 1 ps 1.2\n";
    print GP "set style line 22 lt 2 lw 2 pt 2 ps 1.2\n";
    print GP "set style line 33 lt 3 lw 2 pt 3 ps 1.2\n";
    print GP "set style line 44 lt 4 lw 2 pt 4 ps 1.2\n";
    print GP "set style line 55 lt 5 lw 2 pt 5 ps 1.2\n";
    print GP "set style line 66 lt 6 lw 2 pt 6 ps 1.2\n";
    print GP "set style line 77 lt 7 lw 2 pt 7 ps 1.2\n";
    print GP "set style line 88 lt 8 lw 2 pt 8 ps 1.2\n";
    print GP "set style line 99 lt 9 lw 2 pt 9 ps 1.2\n";

    print GP "set output '$plotfile'\n";
    #print GP "set nokey\n";
    print GP "set xlabel '$xlabel'\n";

    #print GP "set title \"$title\\n\\n$key\"\n";
  
    print GP "expterm(x)=exp(-($ins+$del)*x)\n";
    print GP "xi(x) = $ins/($ins+$del)* (1-expterm(x))\n";
    print GP "gammat(x) = $del/($ins+$del)* (1-expterm(x))\n";

    # Bernoulli frequency parameters

    print GP "pb_lp(x) = ( xi(x) + $pb*(1-xi(x))*(1-gammat(x)) ) / ( 1-$pb*gammat(x)*(1-xi(x)) )\n";
    print GP "pb_L(x) = xi(x) + $pb*(1-xi(x))\n";

    print GP "pb_lp_small(x) = $pb + (1-$pb)*($ins-$del*$pb)*x\n";
    print GP "pb_lp_infty(x) = $ins * ( $ins+$del+$del*$pb)/(($ins+$del)*($ins+$del)-$del*$del*$pb )\n";

    print GP "pb_L_small(x) = $pb+$ins*(1-$pb)*x\n";
    print GP "pb_L_infty(x) = ( $ins+$del*$pb)/( $ins+$del)\n";
    
    $key = "bernoulli parameter for length distributions";
    print GP "set title \"$title\\n\\n$key\"\n";
    print GP "set ylabel 'probability'\n";
    print GP "plot [0:$tmax] [0:1] $pb ls 2, pb_lp(x) ls 3, pb_L(x) ls 4\n";
    print GP "plot [0:$tmax] $pb ls 2, pb_lp(x) ls 3, pb_L(x) ls 4\n";
    
    $key = "bernoulli parameter for descendant length distribution";
    print GP "set title \"$title\\n\\n$key\"\n";
    print GP "set ylabel 'probability'\n";
    print GP "plot [0:$tsmall] pb_lp(x) ls 2, pb_lp_small(x) ls 3\n";
    print GP "plot [$tmed:$tmax] pb_lp(x) ls 2, pb_lp_infty(x) ls 4\n";
    
    $key = "bernoulli parameter for alignment length distribution";
    print GP "set title \"$title\\n\\n$key\"\n";
    print GP "set ylabel 'probability'\n";
    print GP "plot [0:$tsmall] pb_L(x) ls 2, pb_L_small(x) ls 3\n";
    print GP "plot [$tmed:$tmax] pb_L(x) ls 2, pb_L_infty(x) ls 4\n";
    
    # average lengths

    print GP "l(x) = $pb/(1-$pb)\n";
    print GP "L(x) = ( $pb*(1-xi(x)) + xi(x) ) / ((1-$pb)*(1-xi(x)))\n";
    print GP "lp(x) = ( xi(x) + $pb*(1-xi(x))*(1-gammat(x))  ) / ((1-$pb)*(1-xi(x)))\n";
    
    print GP "l_small(x) = l(x)\n";
    print GP "lp_small(x) = l(x) + ($ins - $pb*$del ) / (1-$pb) * x\n";
    print GP "L_small(x) = l(x) + $ins /(1-$pb) * x\n";
    
    print GP "l_infty(x) = l(x)\n";
    print GP "lp_infty(x) = $ins/$del * ($ins+$del+$del*$pb) / (($ins+$del)*(1-$pb))\n";
    print GP "L_infty(x) = ($ins+$del*$pb) / ($del*(1-$pb))\n";
 
    $key = "average lengths";
    print GP "set title \"$title\\n\\n$key\"\n";
    print GP "set ylabel '# residues'\n";
    print GP "plot [0:$tmax] l(x) ls 2, lp(x) ls 3, L(x) ls 4\n";

    $key = "average descendant lengths";
    print GP "set title \"$title\\n\\n$key\"\n";
    print GP "set ylabel '# residues'\n";
    print GP "plot [0:$tsmall] lp(x) ls 2, lp_small(x) ls 3\n";
    print GP "plot [$tmed:$tmax] lp(x) ls 2, lp_infty(x) ls 4\n";

    $key = "average alignment lengths";
    print GP "set title \"$title\\n\\n$key\"\n";
    print GP "set ylabel '# residues'\n";
    print GP "plot [0:$tsmall] L(x) ls 2, L_small(x) ls 3\n";
    print GP "plot [$tmed:$tmax] L(x) ls 2, L_infty(x) ls 4\n";

    # expected frequencies

    print GP "den(x) = $pb*(1-xi(x)) + xi(x)\n";
    print GP "fi(x) = xi(x)/den(x)\n";
    print GP "fd(x) = gammat(x)*$pb*(1-xi(x))/den(x)\n";
    print GP "fs(x) = (1-gammat(x))*$pb*(1-xi(x))/den(x)\n";
 
    print GP "fi_small(x) = $ins/$pb*x\n";
    print GP "fd_small(x) = $del*x\n";
    print GP "fs_small(x) = 1 - fi_small(x) - fd_small(x)\n";
 
    print GP "den2(x) = $ins+$del*$pb\n";
    print GP "fi_infty(x) = $ins/den2(x)\n";
    print GP "fd_infty(x) = $del/($ins+$del)*$del*$pb/den2(x)\n";
    print GP "fs_infty(x) = $ins/($ins+$del)*$del*$pb/den2(x)\n";
 
    $key = "expected frequencies in alignment";
    print GP "set title \"$title\\n\\n$key\"\n";
    print GP "set ylabel 'frequency'\n";
    print GP "plot [0:$tmax] fi(x) ls 2, fd(x) ls 3, fs(x) ls 4\n";
    
    $key = "expected insertions frequency in alignment";
    print GP "set title \"$title\\n\\n$key\"\n";
    print GP "set ylabel 'frequency'\n";
    print GP "plot [0:$tsmall] fi(x) ls 2, fi_small(x) ls 3\n";
    print GP "plot [$tmed:$tmax] fi(x) ls 2, fi_infty(x) ls 4\n";
    
    $key = "expected deletions frequency in alignment";
    print GP "set title \"$title\\n\\n$key\"\n";
    print GP "set ylabel 'frequency'\n";
    print GP "plot [0:$tsmall] fd(x) ls 2, fd_small(x) ls 3\n";
    print GP "plot [$tmed:$tmax] fd(x) ls 2, fd_infty(x) ls 4\n";

    $key = "expected substitutions frequency in alignment";
    print GP "set title \"$title\\n\\n$key\"\n";
    print GP "set ylabel 'frequency'\n";
    print GP "plot [0:$tsmall] fs(x) ls 2, fs_small(x) ls 3\n";
    print GP "plot [$tmed:$tmax] fs(x) ls 2, fs_infty(x) ls 4\n";

    close (GP);

    system ("evince $plotfile&\n"); 
}
