// ATVgraphic.java
// Copyright (C) 1999-2001 Washington University School of Medicine
// and Howard Hughes Medical Institute
// All rights reserved


// AWT version.


package forester.atv_awt;


import forester.tree.*;

import java.awt.*;
import java.awt.event.*;
import java.text.NumberFormat;
import java.net.*;


/**

@author Christian M. Zmasek

@version AWT 1.020 -- last modified: 10/31/00

*/
class ATVgraphic extends Canvas {

    Tree     tree;
    ATVpanel atvpanel;
    
    boolean editable,
            use_real_br_lenghts,
            seq_name_internal_nodes,
            species_internal_nodes,
            ec_internal_nodes,
            seq_name_ext_nodes,
            species_ext_nodes,
            ec_ext_nodes,
            write_lnL_values,
            write_br_length_values,
            write_bootstrap_values,
            write_dup_spec,
            color_branches_acc_to_lnL,
            draw_bl_of_seq_on_parent,
            write_bl_of_seq_on_parent;

    final static int MAX_SUBTREES = 50;
    Tree[] trees = new Tree[ MAX_SUBTREES ]; // to store trees in "subTree"
    int j = 0;                               // index for trees[]
    
    final static int BOX_SIZE = 6,
                     HALF_BOX_SIZE = (int ) BOX_SIZE / 2,
                     MOVE = 30;
                     // Tree graphic is moved by this distance in both x and y.
    
    int longest_ext_node_info,
        x_current,
        y_current,
        x1, y1, x2, y2, factor,
        xx, xxd, xxdd, yyd,
        green,red, 
        x = 0; // for offsetting text in x direction
        
    double x_correction_factor,
           x_distance,
           y_distance;

    int action_when_node_clicked = 0;
    final static int SHOW_INFO   = 0,
                     COLLAPSE    = 1,
                     REROOT      = 3,
                     SUBTREE     = 4,
                     SWAP        = 5;

    final int MAX_NODEJFRAMES = 50;
    int i = 0;
    ATVnodeFrame[] atvnodeframes = new ATVnodeFrame[ MAX_NODEJFRAMES ];

    boolean done;

    int color_scheme;

    Color ext_node_seq_name_color,
          int_node_seq_name_color,
          species_color,
          bootstrap_color,
          ec_color,
          dub_spec_color,
          lnL_color,
          branch_length_color,
          branch_color,
          box_color,
          branch_p_color,
          background_color,
          duplication_box_color;

    Font small_font,
         large_font,
         small_italic_font,
         large_italic_font;

    FontMetrics fm_small,
                fm_large,
                fm_small_italic,
                fm_large_italic;

    int small_maxDescent,
        small_maxAscent;

    NumberFormat lnL_nf = NumberFormat.getNumberInstance();


    /**

    Constructor.

    */
    ATVgraphic( Tree t, ATVpanel tjp ) {

        tree = t;

        atvpanel = tjp;
        
        color_scheme = 1;
        setColors1();
        mediumFonts();

        setBackground( background_color );

        addMouseListener( new ATVmouseListener( this ) );
        
        lnL_nf.setMaximumFractionDigits( 2 );
        lnL_nf.setMinimumFractionDigits( 2 );
        
        setPropertiesForPainting( tree );
        
        setVisible( true );
    }

   
    /**

    Default constructor.

    */
    ATVgraphic() {}



    void setParametersForPainting( int x, int y ) {
        if ( tree != null && !tree.isEmpty() ) {
            
            double xdist = ( double ) ( x - getLongestExtNodeInfo() - MOVE
             - getLengthOfRootSpecies() )
             / ( tree.getMostBranchesPerExtNode() + 1 );
           
            double ydist = ( double ) ( y - MOVE ) / ( tree.getNumberOfExtNodes() * 2 );
           
            
            if ( xdist < 0.0 ) {
                xdist = 0.0;   
            }    
            if ( ydist < 0.0 ) {
                ydist = 0.0;   
            }
            if ( ydist > 2.0 ) {
                ydist = ( int ) ydist;    
            }
            
            setXdistance( xdist );
            setYdistance( ydist );
            
            if ( tree.getLongestDistance() > 0.0 ) {
                double corr = ( x - getLongestExtNodeInfo() - MOVE
                 - getXdistance() - getLengthOfRootSpecies() )
                 / tree.getLongestDistance();
                if ( corr < 0.0 ) {
                    corr = 0.0;
                } 
                setXcorrectionFactor( corr );
            }
            else {     
                setXcorrectionFactor( 0.0 );
            }
        }
    }
    
  
    void resetPreferredSize() {
        if ( tree == null || tree.isEmpty() ) {
            return;
        }
        int x = 0;
        int y  = ( int ) ( MOVE + ( getYdistance() * ( tree.getNumberOfExtNodes() * 2 ) ) );
        if ( useRealBranchLenghts() ) {
            x = ( int ) ( getXcorrectionFactor() * tree.getLongestDistance()
             + getLongestExtNodeInfo() + MOVE
             + getXdistance() + getLengthOfRootSpecies() );
        }
        else {
            x = ( int ) ( getXdistance() * ( tree.getMostBranchesPerExtNode() + 1 )
             + getLongestExtNodeInfo() + MOVE + getLengthOfRootSpecies() );
        }    
        setSize( new Dimension( x, y ) );
            
    }



    Tree getTree() {
        return tree;
    }

    void setTree( Tree t ) {
        tree = t;
    }    


    ATVpanel getATVpanel() {
        return atvpanel;
    }

    int getLongestExtNodeInfo() {
        return longest_ext_node_info;
    }
    void setLongestExtNodeInfo( int i ) {
        longest_ext_node_info = i;
    }

    int getLengthOfRootSpecies() {
        if ( tree == null ) {
            return 0;
        }
        if ( tree.isEmpty() ) {
            return 0;
        }
        return (int) fm_small_italic.stringWidth( tree.getRoot().getSpecies() );
    }

    void calculateLongestExtNodeInfo() {
        if ( tree == null ) {
            return;
        }
        if ( tree.isEmpty() ) {
            return;
        }
        int longest = 0;
        int sum = 0;
        Node node;
        node = tree.getExtNode0();
        while ( node != null ) {
            sum = fm_large_italic.stringWidth( node.getSpecies() + " " )
             + fm_large.stringWidth( node.getSeqName() + " " )
             + fm_large.stringWidth( node.getECnumber() );
            node = node.getNextExtNode();
            if ( sum > longest ) {
                longest = sum;
            }
            sum = 0;
        }
        setLongestExtNodeInfo( longest );
    }


    
    Node findNode( int x, int y ) {
        if ( tree == null ) {
            return null;
        }
        if ( tree.isEmpty() ) {
            return null;
        }
        Node node2, node1 = tree.getExtNode0();
        while ( node1 != null ) {
            node2 = node1;
            while ( node2 != null ) {
                if ( !node2.isPseudoNode()
                && ( tree.isRooted() || !node2.isRoot() )
                && node2.getXcoord() - HALF_BOX_SIZE <= x
                && node2.getXcoord() + HALF_BOX_SIZE >= x
                && node2.getYcoord() - HALF_BOX_SIZE <= y
                && node2.getYcoord() + HALF_BOX_SIZE >= y ) {
                    return node2;
                }
                node2 = node2.getParent();
            }
            node1 = node1.getNextExtNode();
        }
        return null; // not found
    }

    void setActionWhenNodeClicked( int i ) {
        action_when_node_clicked = i;
    }
    int getActionWhenNodeClicked() {
        return action_when_node_clicked;
    }


    void collapse( Node node ) {
        if ( !node.isExternal() ) {
            node.setCollapse( !node.collapse() );
            tree.adjustNodeCount();
            tree.recalculateAndReset();
            resetPreferredSize();
            atvpanel.adjustJScrollPane();
            repaint();
        }
    }



    void reRoot( Node node ) {
        try {
            tree.reRoot( node );
            tree.adjustNodeCount();
            tree.recalculateAndReset();
        }
        catch ( Exception e ) {
            System.err.println( "ATVgraphic: reRoot( node ): " + e );
        }
        resetPreferredSize();
        atvpanel.adjustJScrollPane();
        repaint();
        atvpanel.validate();
        
    }


    void subTree( Node node ) {
        if ( !node.isExternal() && !node.isRoot() && j <= MAX_SUBTREES - 1 ) {
            try {
                trees[ j++ ] = tree;
                tree = tree.subTree( node.getID() );
            }
            catch ( Exception e ) {
                System.err.println( "ATVgraphic: subTree( Node ): " + e );
            }
        }
        else if ( node.isRoot() && j >= 1 ) {
            try {
                trees[ j ] = null;
                tree = trees[ --j ];
            }
            catch ( Exception e ) {
                System.err.println( "ATVgraphic: subTree( Node ): " + e );
            }
        }

        
        atvpanel.getATVcontrol().setCheckBoxes();
        atvpanel.getATVcontrol().showWhole();
    }


    void swap( Node node ) {
        if ( !node.isExternal() ) {
            tree.swapChildren( node );
        }
        repaint();
    }


    void removeRoot() {
        if ( tree == null ) {
            return;
        }
        if ( tree.isEmpty() ) {
            return;
        }
        tree.unRoot();
        repaint();
    }


    void removeRootTri() {
        if ( tree == null ) {
            return;
        }
        if ( tree.isEmpty() ) {
            return;
        }
        tree.unRootAndTrifurcate();
        repaint();
    }



    /**

    Mouse clicked.

    */
    public void MouseClicked( int x, int y ) {
        Node node = findNode( x, y );
        if ( node != null ) {

            if ( action_when_node_clicked == COLLAPSE ) {
                collapse( node );
            }

            else if ( action_when_node_clicked == REROOT ) {
                reRoot( node );
            }
            else if ( action_when_node_clicked == SUBTREE ) {
                subTree( node );
            }
            else if ( action_when_node_clicked == SWAP ) {
                swap( node );
            }
            // show info is default:
            else {
                i = 0;
            while ( atvnodeframes[ i ] != null ) { i++; }
                atvnodeframes[ i ] = new ATVnodeFrame( node, this, i );
            }
        }
    }

    void removeNodeJFrame( int i ) {
        atvnodeframes[ i ] = null;
    }

    void removeAllNodeJFrames() {
        for ( i = 0; i <= MAX_NODEJFRAMES - 1; i++ ) {
            if ( atvnodeframes[ i ] != null ) {
                atvnodeframes[ i ].dispose();
                atvnodeframes[ i ] = null;
            }
        }
        i = 0;
    }

    double getXdistance() {
        return x_distance;
    }
    
    double getYdistance() {
        return y_distance;
    }
    void setXdistance( double i ) {
        x_distance = i;
    }
    
    void setYdistance( double i ) {
        y_distance = i;
    }

    void setXcorrectionFactor( double i ) {
        x_correction_factor = i;
    }
    double getXcorrectionFactor() {
        return x_correction_factor;
    }

    boolean isEditable() {
        return editable;
    }
    
    void setEditable( boolean b ) {
        editable = b;
    }
    
    void setUseRealBranchLenghts( boolean b ) {
        use_real_br_lenghts = b;
    }
    
    boolean useRealBranchLenghts() {
        return use_real_br_lenghts;
    }
    
    void setSeqNameInternalNodes( boolean b ) {
        seq_name_internal_nodes = b;
    }
   
    boolean seqNameInternalNodes() {
        return seq_name_internal_nodes;
    }
    
    void setSpeciesInternalNodes( boolean b ) {
        species_internal_nodes = b;
    }
    
    boolean speciesInternalNodes() {
        return species_internal_nodes;
    }
    
    void setECInternalNodes( boolean b ) {
        ec_internal_nodes = b;
    }
    
    boolean ECInternalNodes() {
        return ec_internal_nodes;
    }
    
    void setSeqNameExtNodes( boolean b ) {
        seq_name_ext_nodes = b;
    }
    
    boolean seqNameExtNodes() {
        return seq_name_ext_nodes;
    }
    
    void setSpeciesExtNodes( boolean b ) {
        species_ext_nodes = b;
    }
    
    boolean speciesExtNodes() {
        return species_ext_nodes;
    }
    
    void setECExtNodes( boolean b ) {
        ec_ext_nodes = b;
    }
    
    boolean ECExtNodes() {
        return ec_ext_nodes;
    }
    
    void setWriteLnLValues( boolean b ) {
        write_lnL_values = b;
    }
    
    boolean writeLnLValues() {
        return write_lnL_values;
    }
    
    void setWriteBranchLengthValues( boolean b ) {
        write_br_length_values = b;
    }
    
    boolean writeBranchLengthValues() {
        return write_br_length_values;
    }
    
    void setWriteBootstrapValues( boolean b ) {
        write_bootstrap_values = b;
    }
    
    boolean writeBootstrapValues() {
        return write_bootstrap_values;
    }
    
    void setWriteDupSpec( boolean b ) {
        write_dup_spec = b;
    }
    
    boolean writeDupSpec() {
        return write_dup_spec;
    }
    
    void setColorBranchesAccToLnL( boolean b ) {
        color_branches_acc_to_lnL = b;
    }
    
    boolean colorBranchesAccToLnL() {
        return color_branches_acc_to_lnL;
    }
    
    boolean drawBlOfSeqOnParent() {
        return draw_bl_of_seq_on_parent;
    }
    
    void setDrawBlOfSeqOnParent( boolean b ) {
        draw_bl_of_seq_on_parent = b;
    }
    
    boolean writeBlOfSeqOnParent() {
        return write_bl_of_seq_on_parent;
    }
    
    void setWriteBlOfSeqOnParent( boolean b ) {
        write_bl_of_seq_on_parent = b;
    }
    
    
    /**
    
    Paints the Tree. 
    
    */
    public void paint( Graphics g ) {

        if ( tree == null ) {
            return;
        }
        if ( tree.isEmpty() ) {
            return;
        }

        done = false;

        Node node = tree.getRoot();

        tree.setIndicatorsToZero();

        if ( !tree.isRooted() ) {
            x_current = MOVE;
        }
        else if ( tree.getRoot().getDistanceToParent() > 0.0
        && useRealBranchLenghts() ) {
            x_current = ( int ) ( tree.getRoot().getDistanceToParent()
            * x_correction_factor + MOVE );
        }
        else {
            x_current = ( int ) ( getXdistance() + MOVE );
        }

        y_current = ( int ) ( getYdistance() * tree.getNumberOfExtNodes()
        + MOVE / 2 );

        setBackground( background_color );

        do {
            if ( !node.isPseudoNode() ) {
                if ( writeBranchLengthValues() && node.getIndicator() == 0
                && node.getDistanceToParent() >= 0.0 ) {
                    g.setFont( small_font );
                    g.setColor( branch_length_color );

                    if ( !node.isRoot() ) {
                        g.drawString( Double.toString( node.getDistanceToParent() ),
                        node.getParent().getXcoord() + 3,
                        y_current - small_maxDescent );
                    }
                    else {
                        // Deepest node (=root):
                        g.drawString( Double.toString( node.getDistanceToParent() ),
                        3, y_current - small_maxDescent );
                    }
                }
                if ( writeLnLValues() && node.getIndicator() == 0
                && node.isLnLonParentBranchAssigned() ) {
                    g.setFont( small_font );
                    g.setColor( lnL_color );

                    if ( !node.isRoot() ) {
                        g.drawString( lnL_nf.format( tree.getHighestLnL() -
                        node.getLnLonParentBranch() ) +
                        " (" + lnL_nf.format( node.getLnLonParentBranch() ) + ")",
                        node.getParent().getXcoord() + 3,
                        y_current + small_maxAscent );
                    }
                    else if ( tree.getNumberOfExtNodes() >= 2 ) {
                        // Deepest node (=root):
                        g.drawString( lnL_nf.format( tree.getHighestLnL()
                        - node.getLnLonParentBranch() ) +
                        " (" + lnL_nf.format( node.getLnLonParentBranch() ) + ")",
                        3, y_current + small_maxAscent );
                    }
                    else {
                        // Tree is just one node:
                        g.drawString( " (" + lnL_nf.format( node.getLnLonParentBranch() ) + ")",
                        3, y_current + small_maxAscent );
                    }
                }
                
                // Write Bootstrap values:
                if ( writeBootstrapValues()&& node.getIndicator() == 0
                && node.getBootstrap() > 0 && tree.getNumberOfExtNodes() >= 2
                && !node.collapse() ) {
                    g.setFont( small_font );
                    g.setColor( bootstrap_color );
                    g.drawString( Integer.toString( node.getBootstrap() ), x_current -
                    fm_small.stringWidth( Integer.toString( node.getBootstrap() ) ) - 2
                    - HALF_BOX_SIZE, y_current + small_maxAscent - 1 );
                }
            }

            // Draw a line as root:
            if ( node.isRoot() && tree.isRooted()
            && node.getIndicator() == 0 ) {

                if ( !colorBranchesAccToLnL()
                || !node.isLnLonParentBranchAssigned() ) {
                    g.setColor( branch_color );
                }
                else {
                    green = ( int ) ( ( node.getLnLonParentBranch()
                    - tree.getLowestLnL()  ) * 254.0 / Math.abs(
                    tree.getHighestLnL()
                    - tree.getLowestLnL() ) );
                    red = 255 - green;
                    g.setColor( new Color( red, green, 0 ) );
                }

                if ( useRealBranchLenghts() && tree.getRoot().getDistanceToParent() > 0.0 ) {
                    g.drawLine( x_current, y_current, x_current - ( int ) ( x_correction_factor *
                    tree.getRoot().getDistanceToParent() ), y_current );
                }
                else {
                    g.drawLine( x_current, y_current, ( int ) ( x_current - x_distance ), y_current );
                    if ( colorBranchesAccToLnL()
                    && !node.significantlyWorse() ) {
                        g.drawLine( x_current, y_current-1, ( int ) ( x_current - x_distance ), y_current-1 );
                        g.drawLine( x_current, y_current+1, ( int ) ( x_current - x_distance ), y_current+1 );
                    }

                }
                if ( !node.collapse() ) {
                    if ( !writeDupSpec() && node.isDuplicationOrSpecAssigned()
                    && node.isDuplication() ) {
                        g.setColor( duplication_box_color );
                    }
                    else {
                        g.setColor( box_color );
                    }
                    g.fillRect( x_current - HALF_BOX_SIZE, y_current - HALF_BOX_SIZE, BOX_SIZE, BOX_SIZE );
                }
                else {
                    drawCollapsedNode( x_current, y_current, g, node );
                }
            }


            //                _
            // Paint child1: |
            if ( node.getIndicator() == 0 && !node.isExternal() ) {

                factor = node.getSumExtNodes() - node.getChild1().getSumExtNodes();

                node.setXcoord( x1 = x_current );
                node.setYcoord( y1 = y_current );
                y2 = ( int ) ( y_current - y_distance * factor );
                if ( !node.isPseudoNode() ) {
                    // Species internal node:
                    if ( speciesInternalNodes()
                    && !node.collapse() && !node.getSpecies().equals( "" )  ) {
                        g.setFont( small_italic_font );
                        g.setColor( species_color );
                        g.drawString( node.getSpecies(), x_current + 3 + HALF_BOX_SIZE,
                        y_current + ( int ) fm_small_italic.getAscent() / 2 );
                    }

                    // Sequence name int node:
                    if ( seqNameInternalNodes() && !node.getSeqName().equals( "" )
                    && !node.collapse() ) {
                        g.setColor( int_node_seq_name_color );
                        g.setFont( large_font );
                        g.drawString( node.getSeqName(), x_current -
                        fm_large.stringWidth( node.getSeqName() ) - 3 - HALF_BOX_SIZE,
                        y_current - fm_large.getMaxDescent() );
                    }

                    // EC int node:
                    if ( ECInternalNodes() && !node.getECnumber().equals( "" )
                    && !node.collapse() ) {
                        g.setColor( ec_color );
                        g.setFont( large_font );
                        if ( seqNameInternalNodes() && !node.getSeqName().equals( "" ) ) {
                            x = fm_large.stringWidth( node.getSeqName() + " " );
                        }
                        else {
                            x = 0;
                        }
                        
                        g.drawString( node.getECnumber(), x_current - x
                        - fm_large.stringWidth( node.getECnumber() ) - 4 - HALF_BOX_SIZE,
                        y_current - fm_large.getMaxDescent() );
                        
                    }

                    // Indicate whether D or S if assigned:
                    if ( writeDupSpec() && node.isDuplicationOrSpecAssigned()
                    && !node.collapse() ) {
                        g.setColor( dub_spec_color );
                        g.setFont( large_font );
                        
                        if ( speciesInternalNodes() ) {
                            x = fm_large.getMaxAscent();
                        }
                        else {
                            x = 0;    
                        }
                        
                        if ( node.isDuplication() ) {
                            g.drawString( "D", x_current + 3 + HALF_BOX_SIZE,
                            y_current + ( int ) fm_large.getAscent() / 2 + x );
                        }
                        else {
                            g.drawString( "S", x_current + 3 + HALF_BOX_SIZE,
                            y_current + ( int ) fm_large.getAscent() / 2 + x );
                        }
                    }
                }

                if ( !node.collapse() ) {
                    if ( useRealBranchLenghts() ) {
                        x2 = x_current + ( int ) ( x_correction_factor *
                        ( ( node.getChild1().getDistanceToParent() >= 0.0 ) ?
                        node.getChild1().getDistanceToParent() : 0.0 ) );
                    }
                    else if ( node.getChild1().isPseudoNode() ) {
                        x2 = x_current;
                    }
                    else if ( node.getChild1().isExternal() ) {
                        x2 = ( int ) ( getXdistance() + MOVE + x_distance *
                        tree.getMostBranchesPerExtNode() );
                    }

                    else {
                        x2 = ( int ) ( x_current + x_distance * factor );
                    }

                    g.setColor( branch_color );
                    if ( !node.isPseudoNode()
                    && !( node.isRoot() && !tree.isRooted() ) ) {
                        g.drawLine( x1, y1 - HALF_BOX_SIZE, x1, y2 );
                    }
                    else {
                        g.drawLine( x1, y1, x1, y2 );
                    }

                    if ( !colorBranchesAccToLnL()
                    || !node.getChild1().isLnLonParentBranchAssigned()
                    || tree.getHighestLnL() == tree.getLowestLnL() ) {
                        g.setColor( branch_color );
                        g.drawLine( x1, y2, x2, y2 );
                    }
                    else {
                        green = ( int ) ( ( node.getChild1().getLnLonParentBranch()
                        - tree.getLowestLnL() ) * 254.0 / Math.abs( tree.getHighestLnL()
                        - tree.getLowestLnL() ) );
                        red = 255 - green;
                        g.setColor( new Color( red, green, 0 ) );
                        g.drawLine( x1, y2, x2, y2 );

                        if ( !node.getChild1().significantlyWorse() ) {
                            g.drawLine( x1, y2-1, x2, y2-1 );
                            g.drawLine( x1, y2+1, x2, y2+1 );
                        }
                    }

                    if ( drawBlOfSeqOnParent()
                    && node.getChild1().getBlOfSeqOnParentBranch() >= 0.0 ) {

                        drawBranchOfSeqOnParent( node.getChild1(), g );

                    }

                    if ( writeBlOfSeqOnParent()
                    && node.getChild1().getBlOfSeqOnParentBranch() >= 0.0 ) {

                        writeBranchLengthOfSeqOnParent( node.getChild1(), g );

                    }

                    if ( !node.getChild1().isPseudoNode() ) {
                        if ( !node.getChild1().collapse() ) {
                            if ( !writeDupSpec()
                            && node.getChild1().isDuplicationOrSpecAssigned()
                            && node.getChild1().isDuplication() ) {
                                g.setColor( duplication_box_color );
                            }
                            else {
                                g.setColor( box_color );
                            }
                            g.fillRect( x2 - HALF_BOX_SIZE, y2 - HALF_BOX_SIZE, BOX_SIZE, BOX_SIZE );
                        }
                        else {
                            drawCollapsedNode( x2, y2, g, node.getChild1() );
                        }
                    }
                    x_current = x2;
                    y_current = y2;

                    node.setIndicator( 1 );
                    node = node.getChild1();

                }
                else {
                    node.setIndicator( 1 );
                }
            }

            // Paint child2: |_
            if ( node.getIndicator() == 1 && !node.isExternal() ) {
                if ( !node.collapse() ) {
                    factor = node.getSumExtNodes() - node.getChild2().getSumExtNodes();

                    x1 = x_current;
                    y1 = y_current;
                    y2 = ( int ) ( y_current + y_distance * factor );

                    if ( useRealBranchLenghts() ) {
                        x2 = x_current + ( int ) ( x_correction_factor *
                        ( ( node.getChild2().getDistanceToParent() >= 0.0 ) ?
                        node.getChild2().getDistanceToParent() : 0.0 ) );
                    }
                    else if ( node.getChild2().isPseudoNode() ) {
                        x2 = x_current;
                    }
                    else if ( node.getChild2().isExternal() ) {
                        x2 = ( int ) ( getXdistance() + MOVE + x_distance
                        * tree.getMostBranchesPerExtNode() );
                    }
                    else {
                        x2 = ( int ) ( x_current + x_distance * factor );
                    }

                    g.setColor( branch_color );
                    if ( !node.isPseudoNode()
                    && !( node.isRoot() && !tree.isRooted() ) ) {
                        g.drawLine( x1, y1 + HALF_BOX_SIZE, x1, y2 );
                    }
                    else {
                        g.drawLine( x1, y1, x1, y2 );
                    }


                    if ( !colorBranchesAccToLnL()
                    || !node.getChild2().isLnLonParentBranchAssigned()
                    || tree.getHighestLnL() == tree.getLowestLnL() ) {
                        g.setColor( branch_color );
                        g.drawLine( x1, y2, x2, y2 );
                    }
                    else {
                        green = ( int ) ( ( node.getChild2().getLnLonParentBranch()
                        - tree.getLowestLnL()  ) * 254.0 / Math.abs( tree.getHighestLnL()
                        - tree.getLowestLnL() ) );
                        red = 255 - green;
                        g.setColor( new Color( red, green, 0 ) );
                        g.drawLine( x1, y2, x2, y2 );
                        if ( !node.getChild2().significantlyWorse() ) {
                            g.drawLine( x1, y2-1, x2, y2-1 );
                            g.drawLine( x1, y2+1, x2, y2+1 );
                        }
                    }

                    if ( drawBlOfSeqOnParent()
                    && node.getChild2().getBlOfSeqOnParentBranch() >= 0.0 ) {

                        drawBranchOfSeqOnParent( node.getChild2(), g );

                    }

                    if ( writeBlOfSeqOnParent()
                    && node.getChild2().getBlOfSeqOnParentBranch() >= 0.0 ) {

                        writeBranchLengthOfSeqOnParent( node.getChild2(), g );

                    }

                    if ( !node.getChild2().isPseudoNode() ) {
                        if ( !node.getChild2().collapse() ) {
                            if ( !writeDupSpec()
                            && node.getChild2().isDuplicationOrSpecAssigned()
                            && node.getChild2().isDuplication() ) {
                                g.setColor( duplication_box_color );
                            }
                            else {
                                g.setColor( box_color );
                            }
                            g.fillRect( x2 - HALF_BOX_SIZE, y2 - HALF_BOX_SIZE, BOX_SIZE, BOX_SIZE );
                        }
                        else {
                            drawCollapsedNode( x2, y2, g, node.getChild2() );
                        }

                    }
                    x_current = x2;
                    y_current = y2;
                    node.setIndicator( 2 );
                    node = node.getChild2();
                }
                else {
                    node.setIndicator( 2 );
                }
            }

            // In case of collapsed root or collapsed root.child2:
            if ( node.isRoot() ) {
                done = true;
            }

            // Moving towards the root:
            else if ( node.getIndicator() == 2 && !node.isExternal() ) {
                node = node.getParent();
                x_current = node.getXcoord();
                y_current = node.getYcoord();
            }



            // Labelling ext node:
            if ( node.isExternal() ) {

                if ( speciesExtNodes() && !node.getSpecies().equals( "" ) ) {
                    g.setFont( large_italic_font );
                    g.setColor( species_color );
                    g.drawString( node.getSpecies() + " ", x_current + 3 + HALF_BOX_SIZE,
                    y_current + ( int ) fm_large.getAscent() / 2 );
                }
                if ( seqNameExtNodes() && !node.getSeqName().equals( "" ) ) {
                    g.setFont( large_font );
                    g.setColor( ext_node_seq_name_color );
                    if ( speciesExtNodes() && !node.getSpecies().equals( "" ) ) {
                        x = fm_large_italic.stringWidth( node.getSpecies() + " " );
                    }
                    else {
                        x = 0;
                    }
                    g.drawString( node.getSeqName(), x_current + x + 3 + HALF_BOX_SIZE,
                    y_current + ( int ) fm_large.getAscent() / 2 );
                }
                if ( ECExtNodes() && !node.getECnumber().equals( "" ) ) {
                    g.setFont( large_font );
                    g.setColor( ec_color );
                    if ( speciesExtNodes() && !node.getSpecies().equals( "" ) ) {
                        x = fm_large_italic.stringWidth( node.getSpecies() + " " );
                    }
                    else {
                        x = 0;
                    }
                    if ( seqNameExtNodes() && !node.getSeqName().equals( "" ) ) {
                        x += fm_large.stringWidth( node.getSeqName() + " " );
                    }

                    g.drawString( node.getECnumber(), x_current + x + 3  + HALF_BOX_SIZE,
                    y_current + ( int ) fm_large.getAscent() / 2 );
                }


                if ( writeBranchLengthValues()
                && !node.isRoot() && node.getDistanceToParent() >= 0.0 ) {
                    g.setFont( small_font );
                    g.setColor( branch_length_color );
                    g.drawString( Double.toString( node.getDistanceToParent() ),
                    node.getParent().getXcoord() + 3,
                    y_current - small_maxDescent );
                }
                if ( writeLnLValues()
                && !node.isRoot() && node.isLnLonParentBranchAssigned() ) {
                    g.setFont( small_font );
                    g.setColor( lnL_color );

                    g.drawString( lnL_nf.format( tree.getHighestLnL()
                    - node.getLnLonParentBranch() ) +
                    " (" + lnL_nf.format( node.getLnLonParentBranch() ) + ")",
                    node.getParent().getXcoord() + 3,
                    y_current + small_maxAscent );
                }

                node.setXcoord( x_current );
                node.setYcoord( y_current );

                if ( node.getNextExtNode() == null ) {
                    done = true;
                }

                // Need to check whether parent is null in case the tree is only
                // one node:
                if ( !node.isRoot() ) {
                    node = node.getParent();
                }

                x_current = node.getXcoord();
                y_current = node.getYcoord();

            }

        } while ( !done );

        tree.setIndicatorsToZero();

    } // end of paint()


    void drawBranchOfSeqOnParent( Node node, Graphics g ) {
        if ( !colorBranchesAccToLnL() ) {
            g.setColor( branch_p_color );
        }

        xx = ( int ) ( ( x1 + x2 ) / 2 );
        xxd = ( int ) ( ( x1 + x2 ) / 2 + y_distance * 0.7 );
        yyd = ( int ) ( y2 - y_distance * 0.7 );
        xxdd = xxd + ( int ) ( node.getBlOfSeqOnParentBranch() * x_correction_factor );

        g.drawLine( xx, y2, xxd, yyd );
        g.drawLine( xxd, yyd, xxdd, yyd );
        g.drawLine( xxdd + 2 , yyd - 2 , xxdd - 2, yyd + 2 );

    }



    void writeBranchLengthOfSeqOnParent( Node node, Graphics g ) {
        g.setFont( small_font );
        g.setColor( branch_p_color );
        g.drawString( Double.toString( node.getBlOfSeqOnParentBranch() ),
        ( int ) ( ( x1 + x2 ) / 2 ), y2 + small_maxAscent );
    }



    void drawCollapsedNode( int x, int y, Graphics g, Node node ) {
        int offset = 0;
        g.setColor( branch_color );
        g.drawRect( x - HALF_BOX_SIZE, y - HALF_BOX_SIZE, BOX_SIZE, BOX_SIZE );
        g.setColor( background_color );
        g.fillRect( x - HALF_BOX_SIZE + 1, y - HALF_BOX_SIZE + 1, BOX_SIZE - 2, BOX_SIZE - 2 );

        if ( speciesExtNodes() && !node.getSpecies().equals( "" ) ) {
            g.setFont( large_italic_font );
            g.setColor( species_color);
            g.drawString( node.getSpecies() + ": ", x + 3 + HALF_BOX_SIZE,
            y + ( int ) fm_large_italic.getAscent() / 2 );
            offset = fm_large_italic.stringWidth( node.getSpecies() + ": " );
        }

        if ( seqNameExtNodes() && !node.getSeqName().equals( "" ) ) {
            g.setFont( large_font );
            g.setColor( int_node_seq_name_color );
            g.drawString( node.getSeqName() + ": ", x + 3 + HALF_BOX_SIZE + offset,
            y + ( int ) fm_large.getAscent() / 2 );
            offset += fm_large.stringWidth( node.getSeqName() + ": " );
        }

        g.setColor( branch_color );
        g.setFont( large_font );
        g.drawString( "collapsed", x + offset + 3 + HALF_BOX_SIZE,
        y + ( int ) fm_large.getAscent() / 2 );
    }



    void setPropertiesForPainting( Tree t ) {
        
        editable                   = false;
        use_real_br_lenghts        = false;
        seq_name_internal_nodes    = true;
        species_internal_nodes     = false;
        ec_internal_nodes          = false;
        seq_name_ext_nodes         = true;
        species_ext_nodes          = false;
        ec_ext_nodes               = false;
        write_lnL_values           = false;
        write_br_length_values     = false;
        write_bootstrap_values     = false;
        write_dup_spec             = false;
        color_branches_acc_to_lnL  = false;
        draw_bl_of_seq_on_parent   = false;
        write_bl_of_seq_on_parent  = false;
            
            
        if ( t != null && !t.isEmpty() ) {
            
            calculateLongestExtNodeInfo();
            
            if ( t.getHighestLnL() != t.getLowestLnL() ) {
                setColorBranchesAccToLnL( true );
            }
            
            // Checks whether bootstrap, branch lenghts
            // have been assigned.
            if ( t.getExtNode0() != null ) {
                if ( t.getExtNode0().getDistanceToParent()
                != Node.DISTANCE_DEFAULT ) {
                    setUseRealBranchLenghts( true );
                }
                
                if ( t.getExtNode0().getParent() != null ) {
                    if ( t.getExtNode0().getParent().getBootstrap()
                    != Node.BOOTSTRAP_DEFAULT ) {
                        setWriteBootstrapValues( true );
                    }
                }    
            }
        } 
    }
    
    
    
    /**

    Switches colors between different schemes.

    */
    void switchColors() {

        switch ( color_scheme ) {
            case 1:
                color_scheme = 2;
                setColors2();
                break;
            case 2:
                color_scheme = 3;
                setColors3();
                break;
            case 3:
                color_scheme = 4;
                setColors4();
                break;
            case 4:
               color_scheme = 1;
               setColors1();
               break;
            
        }
        
        repaint();
    }


    /**

    Sets the colors to the "original" scheme.

    */
    void setColors1() {
        ext_node_seq_name_color = new Color( 0, 0, 0 );
        int_node_seq_name_color = new Color( 255, 0, 0 );
        species_color           = new Color( 40, 40, 40 );
        bootstrap_color         = new Color( 0, 125, 0 );
        ec_color                = new Color( 255, 100, 0 );
        dub_spec_color          = new Color( 255, 0, 0 );
        lnL_color               = new Color( 40, 40, 40 );
        branch_length_color     = new Color( 70, 70, 0 );
        branch_color            = new Color( 0, 20, 200 );
        box_color               = new Color( 0, 20, 200 );
        branch_p_color          = new Color( 140, 140, 140 );
        background_color        = new Color( 250, 250, 250 );
        duplication_box_color   = new Color( 255, 0, 0 );

    }

    /**

    Sets the colors to "grey".

    */
    void setColors2() {
        ext_node_seq_name_color = new Color( 0, 0, 0 );
        int_node_seq_name_color = new Color( 0, 0, 0 );
        species_color           = new Color( 0, 0, 150 );
        bootstrap_color         = new Color( 0, 0, 0 );
        ec_color                = new Color( 150, 0, 0 );
        dub_spec_color          = new Color( 150, 0, 150 );
        lnL_color               = new Color( 255, 255, 255 );
        branch_length_color     = new Color( 150, 0, 0 );
        branch_color            = new Color( 255, 255, 0 );
        box_color               = new Color( 255, 255, 0 );
        branch_p_color          = new Color( 255, 100, 0 );
        background_color        = new Color( 180, 180, 180 );
        duplication_box_color   = new Color( 255, 0, 0 );

    }

    /**

    Sets the colors to "b/w".

    */
    void setColors3() {
        ext_node_seq_name_color = new Color( 0, 0, 0 );
        int_node_seq_name_color = new Color( 0, 0, 0 );
        species_color           = new Color( 50, 50, 50 );
        bootstrap_color         = new Color( 100, 100, 100 );
        ec_color                = new Color( 0, 0, 0 );
        dub_spec_color          = new Color( 0, 0, 0 );
        lnL_color               = new Color( 100, 100, 100 );
        branch_length_color     = new Color( 50, 50, 50 );
        branch_color            = new Color( 0, 0, 0 );
        box_color               = new Color( 0, 0, 0 );
        branch_p_color          = new Color( 100, 100, 100 );
        background_color        = new Color( 255, 255, 255 );
        duplication_box_color   = new Color( 255, 0, 0 );

    }
 
 
 
    /**

    Sets the colors to "blue".

    */
    void setColors4() {
        ext_node_seq_name_color = new Color( 255, 255, 255 );
        int_node_seq_name_color = new Color( 255, 255, 255 );
        species_color           = new Color( 200, 200, 200 );
        bootstrap_color         = new Color( 255, 255, 255 );
        ec_color                = new Color( 200, 200, 200 );
        dub_spec_color          = new Color( 255, 255, 255 );
        lnL_color               = new Color( 200, 200, 200 );
        branch_length_color     = new Color( 200, 200, 200 );
        branch_color            = new Color( 0, 255, 255 );
        box_color               = new Color( 0, 255, 255 );
        branch_p_color          = new Color( 200, 200, 200 );
        background_color        = new Color( 0, 0, 70 );
        duplication_box_color   = new Color( 255, 0, 0 );

    }



    /**

    Medium size.

    */
    void mediumFonts() {
        small_font        = new Font( "SansSerif", Font.PLAIN,  10 );
        large_font        = new Font( "SansSerif", Font.PLAIN,  11 );
        small_italic_font = new Font( "SansSerif", Font.ITALIC, 10 );
        large_italic_font = new Font( "SansSerif", Font.ITALIC, 11 );

        fm_small        = getFontMetrics( small_font );
        fm_large        = getFontMetrics( large_font );
        fm_small_italic = getFontMetrics( small_italic_font );
        fm_large_italic = getFontMetrics( large_italic_font );

        small_maxDescent = fm_small.getMaxDescent();
        small_maxAscent  = fm_small.getMaxAscent() + 2;
    }


    /**

    Large size.

    */
    void largeFonts() {
        small_font        = new Font( "SansSerif", Font.PLAIN,  13 );
        large_font        = new Font( "SansSerif", Font.PLAIN,  13 );
        small_italic_font = new Font( "SansSerif", Font.ITALIC, 13 );
        large_italic_font = new Font( "SansSerif", Font.ITALIC, 13 );

        fm_small        = getFontMetrics( small_font );
        fm_large        = getFontMetrics( large_font );
        fm_small_italic = getFontMetrics( small_italic_font );
        fm_large_italic = getFontMetrics( large_italic_font );

        small_maxDescent = fm_small.getMaxDescent();
        small_maxAscent  = fm_small.getMaxAscent() + 2;
    }


    /**

    Small size.

    */
    void smallFonts() {
        small_font        = new Font( "SansSerif", Font.PLAIN,  10 );
        large_font        = new Font( "SansSerif", Font.PLAIN,  10 );
        small_italic_font = new Font( "SansSerif", Font.ITALIC, 10 );
        large_italic_font = new Font( "SansSerif", Font.ITALIC, 10 );

        fm_small        = getFontMetrics( small_font );
        fm_large        = getFontMetrics( large_font );
        fm_small_italic = getFontMetrics( small_italic_font );
        fm_large_italic = getFontMetrics( large_italic_font );

        small_maxDescent = fm_small.getMaxDescent();
        small_maxAscent  = fm_small.getMaxAscent() + 1;
    }


    /**

    Tiny size.

    */
    void tinyFonts() {
        small_font        = new Font( "SansSerif", Font.PLAIN,  8 );
        large_font        = new Font( "SansSerif", Font.PLAIN,  8 );
        small_italic_font = new Font( "SansSerif", Font.ITALIC, 8 );
        large_italic_font = new Font( "SansSerif", Font.ITALIC, 8 );

        fm_small        = getFontMetrics( small_font );
        fm_large        = getFontMetrics( large_font );
        fm_small_italic = getFontMetrics( small_italic_font );
        fm_large_italic = getFontMetrics( large_italic_font );

        small_maxDescent = fm_small.getMaxDescent();
        small_maxAscent  = fm_small.getMaxAscent() + 1;
    }



} // End of ATVgraphic.
