// ATVprinter.java
// Copyright (C) 1999-2001 Washington University School of Medicine
// and Howard Hughes Medical Institute
// All rights reserved

// AWT version

package forester.atv_awt;


import forester.tree.*;

import java.io.*;
import java.awt.*;
import java.awt.event.*;
import java.text.NumberFormat;


/**

@author Christian Zmasek

@version 1.300 -- last modified: 10/31/00

*/
class ATVprinter extends Frame {

    private Tree tree;
    private String title;
    private ATVgraphic ag;

    private final static double FACTOR_TO_FILL_SHEET = 1.7;
    private final static int    BOX_SIZE = 3,
                                half_box_size = ( int ) BOX_SIZE / 2,
                                MOVE = 30, // tree graphic is moved by this distance in both x and y.
                                TITLE = MOVE * 2;
    private int longest_ext_node_info = 0;
    
    private int x_current,
                y_current,
                x1, y1, x2, y2, factor,
                xx, xxd, xxdd, yyd;
    
    private int x = 0; // for offsetting text in x direction

    private double x_correction_factor,
                   x_distance,
                   y_distance;

    private boolean done;


    private final static Color ext_node_seq_name_color = new Color( 0, 0, 0 ),
                               int_node_seq_name_color = new Color( 0, 0, 0 ),
                               species_color           = new Color( 0, 0, 0 ),
                               bootstrap_color         = new Color( 0, 0, 0 ),
                               ec_color                = new Color( 0, 0, 0 ),
                               dub_spec_color          = new Color( 0, 0, 0 ),
                               lnL_color               = new Color( 0, 0, 0 ),
                               branch_length_color     = new Color( 0, 0, 0 ),
                               branch_color            = new Color( 0, 0, 0 ),
                               box_color               = new Color( 0, 0, 0 ),
                               branch_p_color          = new Color( 140, 140, 140 ),
                               bl_p_color              = new Color( 0, 0, 0 ),
                               collapsed_node_color    = new Color( 0, 0, 0 ),
                               background_color        = new Color( 255, 255, 255 );

    private final static Font small_font        = new Font( "Helvetica", Font.PLAIN,  5 );//int names
    private final static Font large_font        = new Font( "Helvetica", Font.PLAIN,  6 );//ext names
    private final static Font small_italic_font = new Font( "Helvetica", Font.ITALIC, 5 );//int species
    private final static Font large_italic_font = new Font( "Helvetica", Font.ITALIC, 6 );

    private final FontMetrics fm_small        = getFontMetrics( small_font );
    private final FontMetrics fm_large        = getFontMetrics( large_font );
    private final FontMetrics fm_small_italic = getFontMetrics( small_italic_font );
    private final FontMetrics fm_large_italic = getFontMetrics( large_italic_font );

    private final int small_maxDescent = fm_small.getMaxDescent();
    private final int small_maxAscent  = fm_small.getMaxAscent() + 2;

    private NumberFormat lnL_nf = NumberFormat.getNumberInstance();



    ATVprinter( ATVgraphic a, String s ) {

        ag = a;

        tree = ag.getTree();

        title = s;

        x_distance          = ag.getXdistance() * FACTOR_TO_FILL_SHEET;
        y_distance          = ag.getYdistance();
        x_correction_factor = ag.getXcorrectionFactor() * FACTOR_TO_FILL_SHEET;

        setBackground( background_color );

        setVisible( true );

        printTree();

        ag = null;
        tree = null;

        dispose();

    }

    

    private void printTree() {

        if ( tree == null ) {
            return;
        }

        Graphics g;
        Toolkit tk = Toolkit.getDefaultToolkit();

        if ( tk != null ) {

            PrintJob pj = tk.getPrintJob( this, "ATVprinter", null );

            if ( pj != null ) {

                g = pj.getGraphics();

                if ( g != null ) {

                    done = false;


                    lnL_nf.setMaximumFractionDigits( 2 );
                    lnL_nf.setMinimumFractionDigits( 2 );

                    Node node = tree.getRoot();

                    tree.setIndicatorsToZero();

                    if ( tree.getRoot().getDistanceToParent() > 0.0 && ag.useRealBranchLenghts() ) {
                        x_current = ( int ) ( tree.getRoot().getDistanceToParent() * x_correction_factor + MOVE );
                    }
                    else {
                        x_current = ( int ) x_distance + MOVE;
                    }
                    g.setFont( large_font );
                    g.drawString( title, MOVE, TITLE - MOVE );

                    y_current = ( int ) ( y_distance * tree.getNumberOfExtNodes()
                    + TITLE + MOVE / 2 );

                    setBackground( background_color );

                    do {
                        
                        if ( node.getDistanceToParent() != Node.DISTANCE_NULL ) {
                            if ( ag.writeBranchLengthValues() && node.getIndicator() == 0
                            && node.getDistanceToParent() >= 0.0 ) {
                                g.setFont( small_font );
                                g.setColor( branch_length_color );

                                if ( node.getParent() != null /*|| !isRooted()*/ ) {
                                    g.drawString( Double.toString( node.getDistanceToParent() ),
                                    node.getParent().getXcoord() + 3,
                                    y_current - small_maxDescent );
                                }
                                else {
                                    // Deepest node (=root):
                                    g.drawString( Double.toString( node.getDistanceToParent() ),
                                    3, y_current - small_maxDescent );
                                }
                            }
                            if ( ag.writeLnLValues() && node.getIndicator() == 0
                            && node.isLnLonParentBranchAssigned() ) {
                                g.setFont( small_font );
                                g.setColor( lnL_color );

                                if ( node.getParent() != null  ) {
                                    g.drawString( lnL_nf.format( tree.getHighestLnL()
                                    - node.getLnLonParentBranch() ) +
                                    " (" + lnL_nf.format( node.getLnLonParentBranch() ) + ")",
                                    node.getParent().getXcoord() + 3,
                                    y_current + small_maxAscent );
                                }
                                else if ( tree.getNumberOfExtNodes() >= 2 ) {
                                    // Deepest node (=root):
                                    g.drawString( lnL_nf.format( tree.getHighestLnL()
                                    - node.getLnLonParentBranch() ) +
                                    " (" + lnL_nf.format( node.getLnLonParentBranch() ) + ")",
                                    3, y_current + small_maxAscent );
                                }
                                else {
                                    // Tree is just one node:
                                    g.drawString( " (" + lnL_nf.format( node.getLnLonParentBranch() ) + ")",
                                    3, y_current + small_maxAscent );
                                }
                            }

                            // Write Bootstrap values:
                            if ( ag.writeBootstrapValues()
                            && node.getBootstrap() > 0 && tree.getNumberOfExtNodes() >= 2
                            && !node.collapse() ) {
                                g.setFont( small_font );
                                g.setColor( bootstrap_color );
                                g.drawString( Integer.toString( node.getBootstrap() ), x_current -
                                fm_small.stringWidth( Integer.toString( node.getBootstrap() ) ) - 3
                                - half_box_size, y_current + small_maxAscent );
                            }
                        }

                        // Draw a line as root:
                        if ( node.getParent() == null && tree.isRooted()
                        && node.getIndicator() == 0 ) {
                            g.setColor( branch_color );
                            if ( ag.useRealBranchLenghts() && tree.getRoot().getDistanceToParent() > 0.0 ) {
                                g.drawLine( x_current, y_current, x_current - ( int ) ( x_correction_factor *
                                tree.getRoot().getDistanceToParent() ), y_current );
                            }
                            else {
                                g.drawLine( x_current, y_current, x_current - ( int ) x_distance, y_current );
                                if ( ag.colorBranchesAccToLnL()
                                && !node.significantlyWorse() ) {
                                    g.drawLine( x_current, y_current-1, x_current - ( int ) x_distance, y_current-1 );
                                    g.drawLine( x_current, y_current+1, x_current - ( int ) x_distance, y_current+1 );
                                }

                            }
                            if ( !node.collapse() ) {
                                if ( !ag.writeDupSpec() && node.isDuplicationOrSpecAssigned()
                                && node.isDuplication() ) {
                                     drawDuplicationNode( x_current, y_current, g );
                                }
                                else {
                                    g.setColor( box_color );
                                    g.fillRect( x_current - half_box_size, y_current - half_box_size, BOX_SIZE, BOX_SIZE );
                                }
                            }
                            else {
                                drawCollapsedNode( x_current, y_current, g, node );
                            }
                        }



                        //                _
                        // Paint child1: |
                        if ( node.getIndicator() == 0 && !node.isExternal() ) {

                            factor = node.getSumExtNodes() - node.getChild1().getSumExtNodes();

                            node.setXcoord( x1 = x_current );
                            node.setYcoord( y1 = y_current );
                            y2 = ( int ) ( y_current - y_distance * factor );
                            if ( node.getDistanceToParent() != Node.DISTANCE_NULL ) {
                                // Species internal node:
                                if ( ag.speciesInternalNodes()
                                && !node.collapse() && !node.getSpecies().equals( "" )  ) {
                                    g.setFont( small_italic_font );
                                    g.setColor( species_color );
                                    g.drawString( node.getSpecies(), x_current + 3 + half_box_size,
                                    y_current + ( int ) fm_small_italic.getAscent() / 2 );
                                }

                                // Sequence name int node:
                                if ( ag.seqNameInternalNodes() && !node.getSeqName().equals( "" )
                                && !node.collapse() ) {
                                    g.setColor( int_node_seq_name_color );
                                    g.setFont( large_font );
                                    g.drawString( node.getSeqName(), x_current -
                                    fm_large.stringWidth( node.getSeqName() ) - 3 - half_box_size,
                                    y_current - fm_large.getMaxDescent() );
                                }

                                // EC int node:
                                if ( ag.ECInternalNodes() && !node.getECnumber().equals( "" )
                                && !node.collapse() ) {
                                    g.setColor( ec_color );
                                    g.setFont( large_font );
                                    if ( ag.seqNameInternalNodes() && !node.getSeqName().equals( "" ) ) {
                                        x = fm_large.stringWidth( node.getSeqName() + " " );
                                    }
                                    else {
                                        x = 0;
                                    }
                        
                                    g.drawString( node.getECnumber(), x_current - x
                                    - fm_large.stringWidth( node.getECnumber() ) - 4 - half_box_size,
                                    y_current - fm_large.getMaxDescent() );
                        
                                }

                                // Indicate whether D or S if assigned:
                                if ( ag.writeDupSpec() && node.isDuplicationOrSpecAssigned()
                                && !node.collapse() ) {
                                    g.setColor( dub_spec_color );
                                    g.setFont( large_font );
                                    
                                    if ( ag.speciesInternalNodes() ) {
                                        x = fm_large.getMaxAscent();
                                    }
                                    else {
                                        x = 0;    
                                    }
                                    
                                    if ( node.isDuplication() ) {
                                        g.drawString( "D", x_current + 3 + half_box_size,
                                        y_current + ( int ) fm_large.getAscent() / 2 + x );
                                    }
                                    else {
                                        g.drawString( "S", x_current + 3 + half_box_size,
                                        y_current + ( int ) fm_large.getAscent() / 2 + x );
                                    }
                                }
                            }

                            if ( !node.collapse() ) {
                                if ( ag.useRealBranchLenghts() ) {
                                    x2 = x_current + ( int ) ( x_correction_factor *
                                    ( ( node.getChild1().getDistanceToParent() >= 0.0 ) ?
                                    node.getChild1().getDistanceToParent() : 0.0 ) );
                                }
                                else if ( node.getChild1().getDistanceToParent() == Node.DISTANCE_NULL ) {
                                    x2 = x_current;
                                }
                                else if ( node.getChild1().getChild1() == null ) {
                                    x2 = ( int ) ( x_distance + MOVE + x_distance * tree.getMostBranchesPerExtNode() );
                                }

                                else {
                                    x2 = ( int ) ( x_current + x_distance * factor );
                                }

                                g.setColor( branch_color );
                                g.drawLine( x1, y1, x1, y2 );
                                g.drawLine( x1, y2, x2, y2 );

                                if ( ag.colorBranchesAccToLnL()
                                && !node.getChild1().significantlyWorse() ) {
                                    g.drawLine( x1, y2-1, x2, y2-1 );
                                    g.drawLine( x1, y2+1, x2, y2+1 );
                                }

                                if ( ag.drawBlOfSeqOnParent()
                                && node.getChild1().getBlOfSeqOnParentBranch() >= 0.0 ) {

                                    drawBranchOfSeqOnParent( node.getChild1(), g );

                                }

                                if ( ag.writeBlOfSeqOnParent()
                                && node.getChild1().getBlOfSeqOnParentBranch() >= 0.0 ) {

                                    writeBranchLengthOfSeqOnParent( node.getChild1(), g );

                                }

                                if ( node.getChild1().getDistanceToParent() != Node.DISTANCE_NULL ) {
                                    if ( !node.getChild1().collapse() ) {
                                        if ( !ag.writeDupSpec()
                                        && node.getChild1().isDuplicationOrSpecAssigned()
                                        && node.getChild1().isDuplication() ) {
                                            drawDuplicationNode( x2, y2, g );
                                        }
                                        else {
                                            g.setColor( box_color );
                                            g.fillRect( x2 - half_box_size, y2 - half_box_size, BOX_SIZE, BOX_SIZE );
                                        }
                                    }
                                    else {
                                        drawCollapsedNode( x2, y2, g, node.getChild1() );
                                    }
                                }
                                x_current = x2;
                                y_current = y2;

                                node.setIndicator( 1 );
                                node = node.getChild1();

                            }
                            else {
                                node.setIndicator( 1 );
                            }
                        }

                        // Paint child2: |_
                        if ( node.getIndicator() == 1 && !node.isExternal() ) {
                            if ( !node.collapse() ) {
                                factor = node.getSumExtNodes() - node.getChild2().getSumExtNodes();

                                x1 = x_current;
                                y1 = y_current;
                                y2 = ( int ) ( y_current + y_distance * factor );

                                if ( ag.useRealBranchLenghts() ) {
                                    x2 = x_current + ( int ) ( x_correction_factor *
                                    ( ( node.getChild2().getDistanceToParent() >= 0.0 ) ?
                                    node.getChild2().getDistanceToParent() : 0.0 ) );
                                }
                                else if ( node.getChild2().getDistanceToParent() == Node.DISTANCE_NULL ) {
                                    x2 = x_current;
                                }
                                else if ( node.getChild2().getChild1() == null ) {
                                    x2 = ( int ) ( x_distance + MOVE + x_distance
                                    * tree.getMostBranchesPerExtNode() );
                                }
                                else {
                                    x2 = ( int ) ( x_current + x_distance * factor );
                                }

                                g.setColor( branch_color );
                                g.drawLine( x1, y1, x1, y2 );
                                g.drawLine( x1, y2, x2, y2 );

                                if ( ag.colorBranchesAccToLnL()
                                && !node.getChild2().significantlyWorse() ) {
                                    g.drawLine( x1, y2-1, x2, y2-1 );
                                    g.drawLine( x1, y2+1, x2, y2+1 );
                                }

                                if ( ag.drawBlOfSeqOnParent()
                                && node.getChild2().getBlOfSeqOnParentBranch() >= 0.0 ) {

                                    drawBranchOfSeqOnParent( node.getChild2(), g );

                                }

                                if ( ag.writeBlOfSeqOnParent()
                                && node.getChild2().getBlOfSeqOnParentBranch() >= 0.0 ) {

                                    writeBranchLengthOfSeqOnParent( node.getChild2(), g );

                                }

                                if ( node.getChild2().getDistanceToParent() != Node.DISTANCE_NULL ) {
                                    if ( !node.getChild2().collapse() ) {
                                        if ( !ag.writeDupSpec() 
                                        && node.getChild2().isDuplicationOrSpecAssigned()
                                        && node.getChild2().isDuplication() ) {
                                            drawDuplicationNode( x2, y2, g );
                                        }
                                        else {
                                            g.setColor( box_color );
                                            g.fillRect( x2 - half_box_size, y2 - half_box_size, BOX_SIZE, BOX_SIZE );
                                        }
                                    }
                                    else {
                                        drawCollapsedNode( x2, y2, g, node.getChild2() );
                                    }

                                }
                                x_current = x2;
                                y_current = y2;
                                node.setIndicator( 2 );
                                node = node.getChild2();
                            }
                            else {
                                node.setIndicator( 2 );
                            }
                        }

                        // In case of collapsed root or collapsed root.child2:
                        if ( node.getParent() == null ) {
                            done = true;
                        }

                        // Moving towards the root:
                        else if ( node.getIndicator() == 2 && node.getChild1() != null ) {
                            node = node.getParent();
                            x_current = node.getXcoord();
                            y_current = node.getYcoord();
                        }

                        // Labelling ext node:
                        if ( node.getChild1() == null ) {

                            if ( ag.speciesExtNodes() && !node.getSpecies().equals( "" ) ) {
                                g.setFont( large_italic_font );
                                g.setColor( species_color );
                                g.drawString( node.getSpecies() + " ", x_current + 3 + half_box_size,
                                y_current + ( int ) fm_large.getAscent() / 2 );
                            }
                            if ( ag.seqNameExtNodes() && !node.getSeqName().equals( "" ) ) {
                                g.setFont( large_font );
                                g.setColor( ext_node_seq_name_color );
                                if ( ag.speciesExtNodes() && !node.getSpecies().equals( "" ) ) {
                                    x = fm_large_italic.stringWidth( node.getSpecies() + " " );
                                }
                                else {
                                    x = 0;
                                }
                                g.drawString( node.getSeqName(), x_current + x + 3 + half_box_size,
                                y_current + ( int ) fm_large.getAscent() / 2 );
                            }
                            if ( ag.ECExtNodes() && !node.getECnumber().equals( "" ) ) {
                                g.setFont( large_font );
                                g.setColor( ec_color );
                                if ( ag.speciesExtNodes() && !node.getSpecies().equals( "" ) ) {
                                    x = fm_large_italic.stringWidth( node.getSpecies() + " " );
                                }
                                else {
                                    x = 0;
                                }
                                if ( ag.seqNameExtNodes() && !node.getSeqName().equals( "" ) ) {
                                    x += fm_large.stringWidth( node.getSeqName() + " " );
                                }

                                g.drawString( node.getECnumber(), x_current + x + 3  + half_box_size,
                                y_current + ( int ) fm_large.getAscent() / 2 );
                            }


                            
                            if ( ag.writeBranchLengthValues() 
                            && node.getParent() != null && node.getDistanceToParent() >= 0.0 ) {
                                g.setFont( small_font );
                                g.setColor( branch_length_color );
                                g.drawString( Double.toString( node.getDistanceToParent() ),
                                node.getParent().getXcoord() + 3,
                                y_current - small_maxDescent );
                            }
                            if ( ag.writeLnLValues() 
                            && node.getParent() != null && node.isLnLonParentBranchAssigned() ) {
                                g.setFont( small_font );
                                g.setColor( lnL_color );

                                g.drawString( lnL_nf.format( tree.getHighestLnL()
                                - node.getLnLonParentBranch() ) +
                                " (" + lnL_nf.format( node.getLnLonParentBranch() ) + ")",
                                node.getParent().getXcoord() + 3,
                                y_current + small_maxAscent );
                            }
                            

                            node.setXcoord( x_current );
                            node.setYcoord( y_current );

                            if ( node.getNextExtNode() == null ) {
                                done = true;
                            }

                            // Need to check whether parent is null in case the tree is only
                            // one node:
                            if ( node.getParent() != null ) {
                                node = node.getParent();
                            }

                            x_current = node.getXcoord();
                            y_current = node.getYcoord();

                        }

                    } while ( !done );

                    tree.setIndicatorsToZero();

                    g.dispose();

                }

                pj.end();

            }
        }

    } // end of paint()


    private void drawBranchOfSeqOnParent( Node node, Graphics g ) {
        g.setColor( branch_p_color );
        xx = ( int ) ( ( x1 + x2 ) / 2 );
        xxd = ( int ) ( ( x1 + x2 ) / 2 + y_distance * 0.7 );
        yyd = ( int ) ( y2 - y_distance * 0.7 );
        xxdd = xxd + ( int ) ( node.getBlOfSeqOnParentBranch() * x_correction_factor );

        g.drawLine( xx, y2, xxd, yyd );
        g.drawLine( xxd, yyd, xxdd, yyd );
        g.drawLine( xxdd + 1 , yyd - 1 , xxdd - 1, yyd + 1 );

    }


    private void writeBranchLengthOfSeqOnParent( Node node, Graphics g ) {
        g.setFont( small_font );
        g.setColor( bl_p_color );
        g.drawString( Double.toString( node.getBlOfSeqOnParentBranch() ),
        ( int ) ( ( x1 + x2 ) / 2 ), y2 + small_maxAscent );
    }
    
    private void drawDuplicationNode( int x, int y, Graphics g ) {
        
        g.setColor( branch_color );
        g.fillOval( x - 3, y - 3, 6, 6 );  

    }


    
    private void drawCollapsedNode( int x, int y, Graphics g, Node node ) {
        int offset = 0;
        g.setColor( collapsed_node_color );
        g.fillRect( x - half_box_size, y - half_box_size, BOX_SIZE, BOX_SIZE );
        
        if ( ag.speciesExtNodes() && !node.getSpecies().equals( "" ) ) {
            g.setFont( large_italic_font );
            g.setColor( species_color );
            g.drawString( node.getSpecies() + ": ", x + 3 + half_box_size,
            y + ( int ) fm_large.getAscent() / 2 );
            offset = fm_large_italic.stringWidth( node.getSpecies() + ": " );
        }

        if ( ag.seqNameExtNodes() && !node.getSeqName().equals( "" ) ) {
            g.setFont( large_font );
            g.setColor( ext_node_seq_name_color );
            g.drawString( node.getSeqName() + ": ", x + 3 + half_box_size + offset,
            y + ( int ) fm_large.getAscent() / 2 );
            offset += fm_large.stringWidth( node.getSeqName() + ": " );
        }

        g.setColor( collapsed_node_color );
        g.setFont( large_font );
        g.drawString( "collapsed", x + offset + 3 + half_box_size,
        y + ( int ) fm_large.getAscent() / 2 );
    }
   


} // End of class ATVprinter.
