// PreorderTreeIterator.java
// Copyright (C) 1999-2000 Washington University School of Medicine
// and Howard Hughes Medical Institute
// All rights reserved

// Implements the Iterator pattern as described in Gamma et al.
// "Design Patterns Elements of Reusable Object-Oriented Software".


package forester.tree;

import java.util.Stack;


/**

@author Christian M. Zmasek

@version 1.01 -- last modified: 06/14/00

*/
public class PreorderTreeIterator implements Iterator {


    /**
 
    Constructor which creates a preorder Iterator for Tree t.
    This implements the Iterator pattern as described in Gamma et al.
    "Design Patterns Elements of Reusable Object-Oriented Software".

    @param t Tree for which a Iterator is to be constructed.

    */
    public PreorderTreeIterator( Tree t ) throws Exception {

        if ( t.isEmpty() ) {
            String message = "PreorderTreeIterator: Tree is empty.";
            throw new Exception( message );
        }

        tree = t;
        first();

    }



    /**

    Resets the Iterator.

    */
    public void first() {

        is_done = false;
        stack = new Stack();
        stack.push( new StackItem( tree.getRoot(), 1 ) );
        next();

    }


    /**
    
    Advances the Iterator by one.
    
    */
    public void next() {

        if ( node !=null && node.isExternal()
        && node.getNextExtNode() == null ) {
            is_done = true;
            return;
        }

        while ( true ) {
            StackItem si = ( StackItem ) stack.pop();
            if ( si.getNode() != null ) {
                switch( si.getPhase() ) {
                    case 1:
                        stack.push( new StackItem( si.getNode(), 2 ) );
                        node = si.getNode();
                        return;
                    case 2:
                        stack.push( new StackItem( si.getNode(), 3 ) );
                        stack.push( new StackItem( si.getNode().getChild1(), 1 ) );
                        break;
                    case 3:
                        stack.push( new StackItem( si.getNode().getChild2(), 1 ) );
                        break;
                }
            }
        }
    }



    /**
    
    Returns true if all Nodes have been visited.    
    
    */
    public boolean isDone() {
        return is_done;
    }

 
 
    /**
    
    Returns the current Node, unless all Nodes have
    been visited (isDone() returns true), in which case it returns 
    null.
    
    */
    public Node currentNode() {
        if ( is_done ) {
            return null;
        }
        else {
            return node;
        }
    }


    private Tree tree;
    private Node node;
    private StackItem si;
    private boolean is_done;
    private Stack stack;


}
