// Tree.java
// Copyright (C) 1999-2001 Washington University School of Medicine
// and Howard Hughes Medical Institute
// All rights reserved

// Internally, the trees are -- after they are completely constructed --
// always rooted binary trees.


package forester.tree;


import java.io.*;
import java.util.Stack;
import java.util.Vector;
import java.util.StringTokenizer;
import java.net.*;
//import java.util.HashMap;    // Requires JDK 1.2 or greater.
import java.util.Hashtable;  // For JDK 1.1.



/**

@author Christian M. Zmasek

@version 1.442 -- last modified: 10/31/00

*/
public class Tree implements Serializable {

    static final int MAX_LENGTH = 10;
    // Max length for seq names, when printing out "clean" NH.
    
    static final long serialVersionUID = -6847390332113L;
    // See Horstmann and Cornell: Core Java 1.2 Vol I, p.686.

    private Node    ext_node0                  = null,
                    root                       = null;
    private double  highest_lnL                = 0.0,
                    lowest_lnL                 = 0.0,
                    longest_distance           = 0.0; 
                    // Longest distance from root to ext. Node.
                    
    private int     number_of_branches         = 0,
                    external_nodes             = 0,
                    number_of_duplications     = -1,
                    most_branches_per_ext_node = 0;
                    // For drawing a tree w/o real branch lenghts,
                    // to align all the ext nodes.
                    
    private boolean rooted                     = false,
                    more_than_bin_nodes_in_NH  = true;


    private String  name = "";

    private Hashtable idhash         = null;

    private int j = 0; // Used in recursive method "preorderReID".

    
    /**

    Default Tree constructor.
    Constructs empty Tree.
    
    */
    public Tree() {
        setExtNode0( null );
        setRoot( null );
    }



    /**

    Tree constructor.

    @param nh_string String in New Hampshire (NH) or New Hampshire X (NHX) format

    */
    public Tree( String nh_string ) throws Exception {
        idhash = null;
        setRooted( false );
        boolean first = true;
        int i;
        String internal_node_info, A, B, next;
        StringBuffer sb;
        Stack st;
        // To simulate >binary trees.
        String null_distance = new String( ":" + Node.DISTANCE_NULL ); 

        sb = new StringBuffer( nh_string );

        sb = TreeHelper.removeWhiteSpace( sb );

        sb = TreeHelper.removeCstyleComments( sb );

        // Remove anything before first "(", unless tree is just one node.
        while ( !TreeHelper.isEmpty( sb ) && sb.charAt( 0 ) != '('
        && sb.toString().indexOf( "(" ) != -1 ) {
            sb = new StringBuffer( sb.toString().substring( 1 ) );
        }

        // If ';' at end, remove it.
        if ( !TreeHelper.isEmpty( sb )
        && sb.charAt( sb.length() - 1 ) == ';' ) {
            sb.setLength( sb.length() - 1 );
        }

        // If info is given for root: tree is rooted; unrooted otherwise.
        if ( !TreeHelper.isEmpty( sb )
        && sb.charAt( sb.length() - 1 ) != ')' ) {
            setRooted( true );
        }

        nh_string = sb.toString();

        if ( TreeHelper.countAndCheckParantheses( nh_string ) <= -1 ) {
            String message = "Tree: Tree( String ): Error in NHX format: ";
            message += "open parantheses != close parantheses.";
            throw new Exception( message );
        }
        if ( !TreeHelper.checkCommas( nh_string ) ) {
            String message = "Tree: Tree( String ): Error in NHX format: ";
            message += "Commas not properly set.";
            throw new Exception( message );
        }

        // Conversion from nh string to tree object.

        // Empty Tree.
        if ( nh_string.length() < 1 ) {
            setExtNode0( null );
            setRoot( null );
        }

        // Check whether nh string represents a tree with more than
        // node or just a single node.
        // More than one node.
        else if ( nh_string.indexOf( "(" ) != -1 ) {

            A = B = next = "";

            st = new Stack();

            i = 0;

            while ( i <= nh_string.length() - 1 ) {
                if ( nh_string.charAt( i ) == ',' ) {
                    st.push( "," );
                }
                if ( nh_string.charAt( i ) == '(' ) {
                    st.push( "(" );
                }
                if ( nh_string.charAt( i ) != ','
                && nh_string.charAt( i ) != '('
                && nh_string.charAt( i ) != ')' ) {
                    sb = new StringBuffer( "" );
                    while ( i <= nh_string.length() - 1
                    && nh_string.charAt( i ) != ')'
                    && nh_string.charAt( i ) != ','   ) {
                        sb.append( nh_string.charAt( i ) );
                        i++;
                    }
                    i--;
                    st.push( sb.toString() );
                }

                // A ")" calls for connection of one kind or another.
                if ( nh_string.charAt( i ) == ')' ) {
                    internal_node_info = "";
                    // If present, read information for internal node.
                    if ( i <= nh_string.length() - 2
                    && nh_string.charAt( i + 1 ) != ')'
                    && nh_string.charAt( i + 1 ) != ',' ) {
                        i++;
                        sb = new StringBuffer( "" );
                        while ( i <= nh_string.length() - 1
                        && nh_string.charAt( i ) != ')'
                        && nh_string.charAt( i ) != ',' ) {
                            sb.append( nh_string.charAt( i ) );
                            i++;
                        }
                        i--;
                        internal_node_info = sb.toString();
                    }

                    first = true;

                    // Parsing between two parantheses.
                    do {

                        A = st.pop().toString();

                        if ( st.empty() ) {
                            connectInternal( internal_node_info );
                            break;
                        }

                        B = st.pop().toString();

                        if ( st.empty() ) {
                            if ( A.equals( "(" ) ) {
                                connectInternal( internal_node_info );
                                st.push( "(" );
                            }
                            else if ( A.equals( "," ) ) {
                                connectInternal( internal_node_info );
                                break;
                            }
                            else {
                                addNodeAndConnect( A, internal_node_info );
                                break;
                            }
                        }
                        else {
                            next = st.peek().toString();

                            if ( !next.equals( "(" ) && B.equals( "," )
                            && !A.equals( "," ) ) {
                                if ( first && !next.equals( "," ) ) {
                                    addNode( A );
                                    st.push( "," );
                                }
                                else {
                                    addNodeAndConnect( A, null_distance );
                                }
                                first = false;
                            }

                            else {
                                first = false;

                                if ( next.equals( "," ) && !B.equals( "," )
                                && !B.equals( "(" )
                                && A.equals( "," ) ) {
                                    addNodeAndConnect( B, null_distance );
                                }

                                else if ( !next.equals( "(" )
                                && B.equals( "," ) && A.equals( "," ) ) {
                                    connectInternal( null_distance );
                                    st.push( "," );
                                }

                                else if ( next.equals( "(" ) && B.equals( "," )
                                && !A.equals( "(" ) && !A.equals( "," ) ) {
                                    addNodeAndConnect( A, internal_node_info );
                                    st.pop();
                                    break;
                                }

                                else if ( next.equals( "(" ) && !B.equals( "(" )
                                && !B.equals( "," ) && A.equals( "," ) ) {
                                    addNodeAndConnect( B, internal_node_info );
                                    st.pop();
                                    break;
                                }

                                else if ( next.equals( "(" )
                                && B.equals( "," ) && A.equals( "," ) ) {
                                    connectInternal( null_distance );
                                    connectInternal( internal_node_info );
                                    st.pop();
                                    break;
                                }

                                else if ( next.equals( "," ) && B.equals( "(" )
                                && !A.equals( "(" ) && !A.equals( "," ) ) {
                                    addNodeAndConnect( A, internal_node_info );
                                    break;
                                }

                                else if ( next.equals( "," )
                                && B.equals( "(" ) && A.equals( "," ) ) {
                                    connectInternal( internal_node_info );
                                    break;
                                }

                                else if ( next.equals( "(" )
                                && B.equals( "(" ) && !A.equals( "(" ) ) {
                                    if ( A.equals( "," ) ) {
                                        connectInternal( internal_node_info );
                                    }
                                    else {
                                        addNodeAndConnect( A,internal_node_info) ;
                                    }
                                    break;
                                }

                                else if ( A.equals("(")
                                && ( ( next.equals("(") && B.equals(",") )
                                || ( next.equals(",") && B.equals("(") )
                                || ( next.equals("(") && B.equals("(") ) ) ) {
                                    connectInternal( internal_node_info );
                                    st.push( "(" );
                                    break;
                                }

                            }

                        } // end of else (st is not empty).

                    } while ( true );


                }

                i++;

            } // End of while loop going through nh_string.


        }

        // Just one node.
        // Conversion from nh string to tree object with one node.
        else {
            addNode( nh_string );
            setRooted( true );
        }


        if ( !isEmpty() ) {
            setRoot( getExtNode0().getRoot() );
            if ( !isRooted() ) {
                getRoot().deleteData();
            }
            calculateMostBranchesPerExtNode();
            findHighestLnL();
            findLowestLnL();
            calculateLongestDistance();
            calculateNumberOfBranches();

            

            // Checks whether branch lenghts are actually bootstraps.
            // (In case of trees coming form NH format.)
            if ( areBranchLenghtsBootstraps() ) {
                moveBranchLenghtsToBootstrap();
            }
        }
    } // End of Tree constructor.



    /**

    Checks whether a Tree object is deleted (or empty).

    @return true if the tree is deleted (or empty), false otherwise

    */
    public boolean isEmpty() {
        return ( getExtNode0() == null );
    }



    /**

    Deletes this Tree.

    */
    public void delete() {
        setExtNode0( null );
        setRoot( null );
        highest_lnL                = 0.0;
        lowest_lnL                 = 0.0;
        longest_distance           = 0.0;
        number_of_branches         = 0;
        external_nodes             = 0;
        most_branches_per_ext_node = 0;
        number_of_duplications     = -1;
        name                       = "";
    }
    

    /**

    Removes external Node from this Tree.
    Remark: getLongestDistance() and getMostBranchesPerExtNode()
    return incorrect values after this method has been called.
    If this tree has only one node, a empty tree will be returned.

    @param n Node to remove

    */
    public void removeExtNode( Node n ) {

        if ( isEmpty() ) {
            return;
        }
     
        // Returns an empty tree.   
        if ( getNumberOfExtNodes() == 1 && n == getExtNode0() ) {
            setExtNode0( null );
            setRoot( null );
            highest_lnL                = 0.0;
            lowest_lnL                 = 0.0;
            longest_distance           = 0.0;
            number_of_branches         = 0;
            external_nodes             = 0;
            most_branches_per_ext_node = 0;
            number_of_duplications     = -1;
            return;  
        }

        Node removed_node = null;

        removed_node = n;

        if ( removed_node == getExtNode0() ) {
            setExtNode0( removed_node.getNextExtNode() );
            removed_node.getNextExtNode().setPrevExtNode( null );
        }
        else if ( removed_node.getNextExtNode() == null ) {
            removed_node.getPrevExtNode().setNextExtNode( null );
        }
        else {
            removed_node.getNextExtNode().setPrevExtNode( removed_node.getPrevExtNode() );
            removed_node.getPrevExtNode().setNextExtNode( removed_node.getNextExtNode() );
        }


        Node p = removed_node.getParent();

        if ( p.isRoot() ) {
            if ( removed_node.isChild1() ) {
                setRoot( getRoot().getChild2() );
                getRoot().setParent( null );
            }
            else {
                setRoot( getRoot().getChild1() );
                getRoot().setParent( null );
            }
        }
        else {
            Node pp = removed_node.getParent().getParent();
            if ( p.isChild1() ) {
                if ( removed_node.isChild1() ) {
                    p.getChild2().setDistanceToParent( addDist( p.getDistanceToParent(), p.getChild2().getDistanceToParent() ) );
                    pp.setChild1( p.getChild2() );
                    p.getChild2().setParent( pp );
                }
                else {
                    p.getChild1().setDistanceToParent( addDist( p.getDistanceToParent(), p.getChild1().getDistanceToParent() ) );
                    pp.setChild1( p.getChild1() );
                    p.getChild1().setParent( pp );
                }

            }
            else {
                if ( removed_node.isChild1() ) {
                    p.getChild2().setDistanceToParent( addDist( p.getDistanceToParent(), p.getChild2().getDistanceToParent() ) );
                    pp.setChild2( p.getChild2() );
                    p.getChild2().setParent( pp );
                }
                else {
                    p.getChild1().setDistanceToParent( addDist( p.getDistanceToParent(), p.getChild1().getDistanceToParent() ) );
                    pp.setChild2( p.getChild1() );
                    p.getChild1().setParent( pp );
                }
            }

            while ( pp != getRoot() ) {
                pp.setSumExtNodes( pp.getSumExtNodes() - 1 );
                pp = pp.getParent();
            }

            pp.setSumExtNodes( pp.getSumExtNodes() - 1 );




        }
        setNumberOfBranches( getNumberOfBranches() - 2 );
        setExtNodes( getNumberOfExtNodes() - 1 );

        return;
    }


    /**

    Private helper method for removeExtNode( Node ).

    */
    private double addDist( double a, double b ) {
        if ( a >= 0.0 && b >= 0.0 ) {
            return a + b;
        }
        else if ( a >= 0.0 ) {
            return a;
        }
        else if ( b >= 0.0 ) {
            return b;
        }
        else if ( a == Node.DISTANCE_NULL && b == Node.DISTANCE_NULL ) {
            return Node.DISTANCE_NULL;
        }
        return Node.DISTANCE_DEFAULT;
    }



    /**

    Returns a Vector with references to all Nodes of this Tree which
    have a matching sequence name.
    
    @param seqname Sequence name (String) of Nodes to find

    @return Vector of references to Nodes of this Tree with matching
    sequence names

    @see #getNodesWithMatchingSpecies(String)

    */
    public Vector getNodes( String seqname ) throws Exception {
        if ( isEmpty() ) {
            String message = "Tree: getNodes( String ): ";
            message += "Tree must not be empty.";
            throw new Exception( message );
        }
        seqname = seqname.trim();
        Vector nodes = new Vector();

        PreorderTreeIterator i = new PreorderTreeIterator( this );
        while ( !i.isDone() ) {
            if ( i.currentNode().getSeqName().trim().equals( seqname ) ) {
                nodes.addElement( i.currentNode() );
            }
            i.next();
        }

        nodes.trimToSize();
        return nodes;
    }



    /**

    Returns a Vector with references to all Nodes of this Tree which
    have a matching species name.

    @param specname species name (String) of Nodes to find

    @return Vector of references to Nodes of this Tree with matching
    species names.

    @see #getNodes(String)

    */
    public Vector getNodesWithMatchingSpecies( String specname ) throws Exception {
        if ( isEmpty() ) {
            String message = "Tree: getNodes( String ): ";
            message += "Tree must not be empty.";
            throw new Exception( message );
        }
        specname = specname.trim();
        Vector nodes = new Vector();

        PreorderTreeIterator i = new PreorderTreeIterator( this );
        while ( !i.isDone() ) {
            if ( i.currentNode().getSpecies().trim().equals( specname ) ) {
                nodes.addElement( i.currentNode() );
            }
            i.next();
        }

        nodes.trimToSize();
        return nodes;
    }



    /**

    Finds the Node of this Tree which has a matching ID number.
    Takes O(n) time. After method hashIDs() has been called it runs in
    constant time.

    @param id ID number (int) of the Node to find

    @return Node with matching ID, null if Node not found

    */
    public Node getNode( int id ) {

        if ( isEmpty() ) {
            return null;
        }

        if ( idhash != null ) {
            return ( Node ) idhash.get( new Integer( id ) );
        }
        else {
            Node node2, node1 = getExtNode0();
            setIndicatorsToZero();
	        while ( node1 != null ) {
                node2 = node1;
                while ( node2 != null ) {
	                if ( node2.getIndicator() == 0 ) {
	                    if ( node2.getID() == id ) {
		                    return node2;
		                }
		                node2.setIndicator( 1 );
		            }
	                node2 = node2.getParent();
	            }
    	        node1 = node1.getNextExtNode();
            }
        }

        // Not found.
        System.err.println( "getNode: Node not found." );
        return null;
    }


    
    /**

    Swaps the the two childern of a Node with ID id of this Tree.
    
    @param id ID (int) of Node
    
    */
    public void swapChildren( int id ) {
        swapChildren( getNode( id ) );
    }



    /**

    Swaps the the two childern of a Node node of this Tree.
    
    @param node a Node of this Tree
    
    */
    public void swapChildren( Node node ) {
        if ( isEmpty() ) {
            return;
        }
        if ( node.isExternal() ) {
            return;
        }
        Node temp = node.getChild1();
        node.setChild1( node.getChild2() );
        node.setChild2( temp );
        adjustExtNode0();
        adjustReferencesInExtNodes();
        adjustNodeCount();
        recalculateAndReset();
    }



    /**

    Returns the subtree of this Tree which has the Node with ID id
    as its root Node.
    
    @param id ID (int) of Node

    */
    public Tree subTree( int id ) throws Exception {
        if ( isEmpty() ) {
            return null;
        }
        Tree tree = copyTree();
        Node node = tree.getNode( id );
        if ( node.isExternal() ) {
            String message = "Tree: subTree( int ): ";
            message += "Can not get a sub tree from a external node.";
            throw new Exception( message );
        }
        node.setParent( null );
        node.setDistanceToParent( Node.DISTANCE_DEFAULT );
        tree.setRooted( true );
        tree.setRoot( node );
        tree.adjustExtNode0();
        tree.adjustReferencesInExtNodes();
        tree.adjustNodeCount();
        tree.recalculateAndReset();
        tree.setExtNodes( tree.getRoot().getSumExtNodes() );
        return tree;
    }



    /**

    Places the root of this Tree on the parent branch of the
    Node with a corresponding ID. The new root is always placed
    on the middle of the branch.
    If the resulting reRooted Tree is to be used any further,
    in most cases the following three methods have to be
    called on the resulting Tree:
    adjustNodeCount()
    recalculateAndReset()
    setExtNodes( getRoot().getSumExtNodes() ) 

    @param id ID (int) of Node of this Tree
    
    */
    public void reRoot( int id ) throws Exception {
        reRoot( getNode( id ), "" );
    }

    
    
    /**

    Places the root of this Tree on the parent branch Node n.
    The new root is always placed on the middle of the branch.
    If the resulting reRooted Tree is to be used any further,
    in most cases the following three methods have to be
    called on the resulting Tree:
    adjustNodeCount()
    recalculateAndReset()
    setExtNodes( getRoot().getSumExtNodes() ) 

    @param n Node of this Tree 
    
    */ 
    public void reRoot( Node n ) throws Exception {
        reRoot( n, "" );
    }



    /**

    Places the root of this Tree on the parent branch of the
    Node with a corresponding ID. Information for the new root Node
    is provided in a String using NHX tags. The new root is always placed
    on the middle of the branch.
    If the resulting reRooted Tree is to be used any further,
    in most cases the following three methods have to be
    called on the resulting Tree:
    adjustNodeCount()
    recalculateAndReset()
    setExtNodes( getRoot().getSumExtNodes() ) 

    @param id ID (int) of Node of this Tree
    @param s  String describing new Node
    
    */ 
    public void reRoot( int id, String s ) throws Exception {
        reRoot( getNode( id ), s );
        
    }
    


    /**

    Places the root of this Tree on the parent branch Node n.
    Information for the new root Node is provided in a String
    using NHX tags. The new root is always placed on the middle
    of the branch.
    If the resulting reRooted Tree is to be used any further,
    in most cases the following three methods have to be
    called on the resulting Tree:
    adjustNodeCount()
    recalculateAndReset()
    setExtNodes( getRoot().getSumExtNodes() ) 

    @param n Node of this Tree 
    @param s String describing new Node
    
    */ 
    public void reRoot( Node n, String s ) throws Exception {

        if ( isEmpty() ) {
            return;
        }

        Node nodeA = n,
             node  = null;

        double  distance1,
                distance2,
                lnL1,
                lnL2,
                bl_of_seq1,
                bl_of_seq2;

        boolean sign_worse1,
                sign_worse2,
                lnL_assign1,
                lnL_assign2;

        if ( nodeA.isPseudoNode() ) {
            String message = "Tree: reRoot: ";
            message += "Can not place root in >bin node.";
            throw new Exception( message );
        }

        setRooted( true );

        if ( nodeA.isRoot() ) {
            return;
        }


        if ( nodeA.getParent().isRoot() ) {
            if ( getRoot().getChild1().isPseudoNode()
            || getRoot().getChild2().isPseudoNode() ) {
                if ( getRoot().getChild1() == nodeA
                &&   getRoot().getChild2().isPseudoNode() ) {
                    node = getRoot().getChild2();
                }
                if ( getRoot().getChild2() == nodeA
                &&   getRoot().getChild1().isPseudoNode() ) {
                    node = getRoot().getChild1();
                }
                if ( nodeA.getDistanceToParent() == Node.DISTANCE_DEFAULT ) {
                    node.setDistanceToParent( Node.DISTANCE_DEFAULT );
                }
                else {
                    node.setDistanceToParent( 
                     nodeA.getDistanceToParent() / 2 );
                    nodeA.setDistanceToParent(
                     nodeA.getDistanceToParent() / 2 );
                }

                node.setLnLonParentBranch( nodeA.getLnLonParentBranch() );
                node.setSignificantlyWorse( nodeA.significantlyWorse() );
                node.setLnLonParentBranchAssigned(
                nodeA.isLnLonParentBranchAssigned() );
                node.setBlOfSeqOnParentBranch(
                 nodeA.getBlOfSeqOnParentBranch() );
            }
            return;
        }

        Node nodeB = nodeA.getParent();
        Node nodeC = nodeB.getParent();

        // New Root.
        Node new_root = new Node( s );
        new_root.setParent( null );
        if ( nodeB.getChild2() == nodeA ) {
            new_root.setChild1( nodeA );
            new_root.setChild2( nodeB );
        }
        else {
            new_root.setChild1( nodeB );
            new_root.setChild2( nodeA );
        }

        distance1   = nodeC.getDistanceToParent();
        lnL1        = nodeC.getLnLonParentBranch();
        sign_worse1 = nodeC.significantlyWorse();
        lnL_assign1 = nodeC.isLnLonParentBranchAssigned();
        bl_of_seq1  = nodeC.getBlOfSeqOnParentBranch();

        nodeC.setDistanceToParent( nodeB.getDistanceToParent() );

        nodeC.setLnLonParentBranch( nodeB.getLnLonParentBranch() );
        nodeC.setLnLonParentBranchAssigned( nodeB.isLnLonParentBranchAssigned() );
        nodeC.setSignificantlyWorse( nodeB.significantlyWorse() );
        nodeC.setBlOfSeqOnParentBranch( nodeB.getBlOfSeqOnParentBranch() );

        nodeB.setLnLonParentBranch( nodeA.getLnLonParentBranch() );
        nodeB.setLnLonParentBranchAssigned( nodeA.isLnLonParentBranchAssigned() );
        nodeB.setSignificantlyWorse( nodeA.significantlyWorse() );
        nodeB.setBlOfSeqOnParentBranch( nodeA.getBlOfSeqOnParentBranch() );


        // New root is always placed in the middle of the branch:
        if ( nodeA.getDistanceToParent() == Node.DISTANCE_DEFAULT ) {
            nodeB.setDistanceToParent( Node.DISTANCE_DEFAULT );
        }
        else {
            nodeB.setDistanceToParent( nodeA.getDistanceToParent() / 2 );
            nodeA.setDistanceToParent( nodeA.getDistanceToParent() / 2 );
        }

        // nodeA:
        nodeA.setParent( new_root );


        // nodeB:
        if ( nodeB.getChild1() == nodeA ) {
            nodeB.setChild1( nodeC );
        }
        else {
            nodeB.setChild2( nodeC );
        }
        nodeB.setParent( new_root );

        // moving to the old root, swapping references:
        while ( !nodeC.isRoot() ) {

            nodeA = nodeB;
            nodeB = nodeC;
            nodeC = nodeC.getParent();

            if ( nodeB.getChild1() == nodeA ) {
                nodeB.setChild1( nodeC );
            }
            else {
                nodeB.setChild2( nodeC );
            }
            nodeB.setParent( nodeA );

            distance2   = nodeC.getDistanceToParent();
            lnL2        = nodeC.getLnLonParentBranch();
            sign_worse2 = nodeC.significantlyWorse();
            lnL_assign2 = nodeC.isLnLonParentBranchAssigned();
            bl_of_seq2  = nodeC.getBlOfSeqOnParentBranch();

            nodeC.setDistanceToParent( distance1 );
            nodeC.setLnLonParentBranch( lnL1 );
            nodeC.setSignificantlyWorse( sign_worse1 );
            nodeC.setLnLonParentBranchAssigned( lnL_assign1 );
            nodeC.setBlOfSeqOnParentBranch( bl_of_seq1 );

            distance1   = distance2;
            lnL1        = lnL2;
            sign_worse1 = sign_worse2;
            lnL_assign1 = lnL_assign2;
            bl_of_seq1  = bl_of_seq2;
        }


        // removing the old root:
        if ( nodeC.getChild1() == nodeB ) {
            node = nodeC.getChild2();
        }
        else {
            node = nodeC.getChild1();
        }
        node.setParent( nodeB );

        if ( nodeC.getDistanceToParent() == Node.DISTANCE_NULL
        && node.getDistanceToParent() == Node.DISTANCE_NULL ) {
            node.setDistanceToParent( Node.DISTANCE_NULL );
        }
        else if ( ( nodeC.getDistanceToParent() == Node.DISTANCE_DEFAULT
        ||  nodeC.getDistanceToParent() == Node.DISTANCE_NULL )
        && ( node.getDistanceToParent() == Node.DISTANCE_DEFAULT
        ||  node.getDistanceToParent() == Node.DISTANCE_NULL ) ) {
            node.setDistanceToParent( Node.DISTANCE_DEFAULT );
        }
        else {
            node.setDistanceToParent( ( nodeC.getDistanceToParent() >= 0.0 ?
            nodeC.getDistanceToParent() : 0.0 )
            + ( node.getDistanceToParent() >= 0.0 ?
            node.getDistanceToParent() : 0.0 ) );
        }


        if ( nodeC.getDistanceToParent() != Node.DISTANCE_NULL ) {
            node.setLnLonParentBranch( nodeC.getLnLonParentBranch() );
            node.setSignificantlyWorse( nodeC.significantlyWorse() );
            node.setLnLonParentBranchAssigned( nodeC.isLnLonParentBranchAssigned() );
            node.setBlOfSeqOnParentBranch( nodeC.getBlOfSeqOnParentBranch() );
        }

        if ( nodeB.getChild1() != nodeC ) {
            nodeB.setChild2( node );
        }
        else {
            nodeB.setChild1( node );
        }


        setRoot( new_root );

        adjustExtNode0();
        adjustReferencesInExtNodes();
        return;

    }

    /**

    Returns a Tree which is a copy of this Tree, except it has the branch
    lenghts of tree2. Important (but obvious): The topology of both trees
    needs to be the same. (The method is not robust in this point, and will
    produce wrong results if the internal topology differs.)
    Furthermore, the combination of sequence name, species, and EC number
    needs to be unique for each external node.

    @param tree2 the Tree to copy the branch lenghts from

    */
    public Tree copyBranchLengthValuesFrom( Tree tree2 ) throws Exception {
        if ( isEmpty() || tree2.isEmpty() ) {
            String message = "Tree: copyBranchLengthValuesFrom( Tree ): ";
            message += "Tree(s) must not be empty.";
            throw new Exception( message );
        }
        Node node0, node2_, node1_;
        Node[] nodes1, nodes2;
        int i = 0;
        Tree tree1 = copyTree();
        tree1.setIndicatorsToZero();
        tree2.setIndicatorsToZero();

        // Visit each node of tree1 and set distance to parent to DISTANCE_NULL:
        Node node2, node1 = tree1.getExtNode0();
        while ( node1 != null ) {
            node2 = node1;
            while ( node2 != null ) {
                if ( !node2.isExternal() ) {
                    node2.setDistanceToParent( Node.DISTANCE_NULL );
                }
                node2 = node2.getParent();
            }
            node1 = node1.getNextExtNode();
        }

        // Visit each internal node of tree2:
        node1 = tree2.getExtNode0();
        while ( node1 != null ) {
            node2 = node1;
            while ( node2 != null ) {
                if ( node2.getIndicator() == 0
                && !node2.isPseudoNode() ) {
                    node2.setIndicator( 1 );
                    nodes2 = node2.copyAllExtChildren();
                    node0 = nodes2[ 0 ];
                    i++;
                    node1_ = tree1.getExtNode0();
                    // Visit each internal node of tree1:
                    while ( node1_ != null ) {
                        node2_ = node1_;
                        if ( node0.equals( node1_ ) ) {
                            while ( node2_ != null ) {
                                if ( node2_.getIndicator()
                                != i && node2_.getIndicator() != -1 ) {
                                    node2_.setIndicator( i );
                                    if ( nodes2.length
                                    == node2_.getSumExtNodes() ) {
                                        nodes1 = node2_.copyAllExtChildren();
                                        if ( Node.compareArraysOfNodes(
                                         nodes2, nodes1 ) ) {
                                            node2_.setIndicator( -1 );
                                            node2_.setDistanceToParent(
                                             node2.getDistanceToParent() );
                                        }
                                        else {
                                            String message = "Trees were ";
                                            message += "not identical.";
                                            throw new Exception( message );
                                        }
                                    }
                                }
                                node2_ = node2_.getParent();
                            }
                        }
                        node1_ = node1_.getNextExtNode();
                    }
                    //end of visiting each internal node of tree1.
                }
                node2 = node2.getParent();
            }
            node1 = node1.getNextExtNode();
        }
        if ( !tree2.isRooted() ) {
            tree1.setRooted( false );
        }
        else {
            tree1.setRooted( true );
        }
        tree1.recalculateAndReset();
        tree1.reassignIDs();

        return tree1;
    }



    /**

    Recalculates and resets parameters of this Tree:
    most-branches-per-external-Node, highest and lowest lnL,
    longest distance, number of branches, sum of ext Nodes, 
    deletes data of root Node if this Tree is unrooted.
    To be used after Tree has been modified.

    */
    public void recalculateAndReset() {
    if ( isEmpty() ) { return; }
        setIndicatorsToZero();
        if ( !isRooted() ) {
            getRoot().deleteData();
        }
        calculateMostBranchesPerExtNode();
        findHighestLnL();
        findLowestLnL();
        calculateLongestDistance();
        calculateNumberOfBranches();
        setExtNodes( getRoot().getSumExtNodes() );
    }
   
   
    
    /**

    Returns a deep copy of this Tree. 

    */
    public Tree copyTree() {

        Tree tree                       = new Tree();

        if ( isEmpty() ) {
            tree.delete();
            return tree;
        }

        tree.rooted                     = rooted;
        tree.highest_lnL                = highest_lnL;
        tree.lowest_lnL                 = lowest_lnL;
        tree.longest_distance           = longest_distance;
        tree.number_of_branches         = number_of_branches;
        tree.external_nodes             = external_nodes;
        tree.most_branches_per_ext_node = most_branches_per_ext_node;
        tree.number_of_duplications     = number_of_duplications;
        tree.more_than_bin_nodes_in_NH  = more_than_bin_nodes_in_NH;
        tree.name                       = new String( name );

        tree.root = Node.copyTree( root );

        tree.getRoot().setParents();

        tree.adjustExtNode0();
        tree.adjustReferencesInExtNodes();
        tree.setIndicatorsToZero();
        return tree;
    }


   
    /**
    
    Removes the root Node of this Tree 
    and makes at least a trifurcation at its basal node.

    */
    public void unRootAndTrifurcate() {
        if ( isEmpty() ) {
            return;
        }
        unRoot();
        double sum;
        if ( getNumberOfExtNodes() <= 2 ) {
            return;
        }

        sum = getRoot().getChild2().getDistanceToParent()
        + getRoot().getChild1().getDistanceToParent();
        if ( getRoot().getChild2().isExternal() ) {
            if ( sum >= 0.0 ) {
                getRoot().getChild2().setDistanceToParent( sum );
            }
            else {
                getRoot().getChild2().setDistanceToParent( Node.DISTANCE_DEFAULT );
            }
            getRoot().getChild1().setDistanceToParent( Node.DISTANCE_NULL );
        }
        else {
            if ( sum >= 0.0 ) {
                getRoot().getChild1().setDistanceToParent( sum );
            }
            else {
                getRoot().getChild1().setDistanceToParent( Node.DISTANCE_DEFAULT );
            }
            getRoot().getChild2().setDistanceToParent( Node.DISTANCE_NULL );
        }
        recalculateAndReset();
        return;
    }

    
    /**
    
    Removes the root Node this Tree.

    */
    public void unRoot() {
        if ( isEmpty() ) {
            return;
        }

        setIndicatorsToZero();
        if ( !isRooted() || getNumberOfExtNodes() <= 1 ) {
            return;
        }
        setRooted( false );
        recalculateAndReset();
        return;
    }



    //      d (2nd argument, double)
    //     |--|
    //             r (3rd argument, double)
    //        |----------|
    //    
    //                   |
    //        |==========O  <--- Tree to be fused (4th argument, Tree)
    //        |          |
    //        |       |   
    //     |==O=======O <--ID (1st argument, int)
    //     |  ^       |
    //     |  |________ new Node (5th argument, Node)       
    //     |
    // ----O
    //     |
    //     |
    //     |          | 
    //     |==========O
    //                |

    /**

    Returns a Tree which is the result of fusing a rooted Tree to this Tree.

    @param id ID (int) of the Node of this Tree of whose parent branch
              to fuse to
    @param d  distance to parent (double) of newly introduced Node
    @param r  distance to parent (double) of Tree to fuse to this Tree 
    @param t  Tree to fuse to this Tree
    @param n  the newly introduced Node

    */
    public Tree fuseTrees( int id, double d, double r, Tree t, Node n ) throws Exception {
        if ( isEmpty() || t.isEmpty() ) {
            String message = "Tree: fuseTrees( int , double , double , Tree";
            message += " , Node ): Tree(s) must not be empty.";
            throw new Exception( message );
        }
        double n2d;

        if ( !t.isRooted() ) {
            String message = "fuseTrees( int, double, double, Tree, Node";
            message += " ): Cannot fuse an unrooted Tree to another Tree.";
            throw new Exception( message );
        }
        if ( r < 0.0 && r != Node.DISTANCE_DEFAULT ) {
            r = 0.0;
            System.err.print( "fuseTrees( int, double, double, Tree, Node ): WARNING: " );
            System.err.println( "Negative r has been replaced by 0.0." );
        }
        if ( d < 0.0 && d != Node.DISTANCE_DEFAULT && d != Node.DISTANCE_NULL ) {
            d = 0.0;
            System.err.print( "fuseTrees( int, double, double, Tree, Node ): WARNING: " );
            System.err.println( "Negative d has been replaced by 0.0." );
        }


        Tree this_tree = copyTree();
        t = t.copyTree();

        Node new_node = n.copyNodeData();
        Node node2 = this_tree.getNode( id );

        if ( node2.isPseudoNode() ) {
            System.err.println( "fuseTrees(int, double, double, Tree, Node ): WARNING: Attempted to fuse to a pseudo node." );
            d = Node.DISTANCE_NULL;
            // Always add to the deepest node with distance to parent is NULL:
            while ( !node2.isRoot()
            && node2.getParent().isPseudoNode() ) {
                node2 = node2.getParent();
            }
            // If node2.parent's both children have NULL distance to parent, always
            // attach to child1:
            if ( node2.getParent().getChild1().isPseudoNode()
            && node2.getParent().getChild2().isPseudoNode() ) {
                node2 = node2.getParent().getChild1();
            }
        }

        t.getRoot().setDistanceToParent( r );

        // Fuse tree t to branch "behind" the root of this_tree (which must be rooted):
        if ( node2.isRoot() && this_tree.isRooted() ) {
            //System.err.println( "fuseTrees: behind root");
            new_node.setChild2( t.getRoot() );
            new_node.setChild1( node2 );
            this_tree.setRoot( new_node );
        }
        else {
            // this_tree is not rooted:
            if ( node2.isRoot() && !this_tree.isRooted() ) {
                if ( node2.getChild1().isPseudoNode() ) {
                    d = Node.DISTANCE_NULL;
                    node2 = node2.getChild1();
                }
                else if ( node2.getChild2().isPseudoNode() ) {
                    d = Node.DISTANCE_NULL;
                    node2 = node2.getChild2();
                }
                else {
                    String message = "Tree: fuseTrees( int, double, double, Tree, Node";
                    message += " ): Meaningless placement of tree to be fused.";
                    throw new Exception( message );
                }
            }

            // Fuse onto all other branches:
            Node node1 = node2.getParent();
            new_node.setParent( node1 );
            if ( node1.getChild1() == node2 ) {
                node1.setChild1( new_node );
                new_node.setChild1( t.getRoot() );
                new_node.setChild2( node2 );
            }
            else {
                node1.setChild2( new_node );
                new_node.setChild2( t.getRoot() );
                new_node.setChild1( node2 );
            }
        }

        t.setIndicatorsToZero();
        this_tree.setIndicatorsToZero();

        t.getRoot().setParent( new_node );
        node2.setParent( new_node );

        n2d = node2.getDistanceToParent();

        if ( d > n2d && n2d != Node.DISTANCE_DEFAULT && n2d != Node.DISTANCE_NULL ) {
            new_node.setDistanceToParent( n2d );
            node2.setDistanceToParent( 0.0 );
            System.err.print( "fuseTrees(int, double, double, Tree, Node ): WARNING: " );
            System.err.println( "Distance d was larger than distance between Nodes 1 and 2." );
        }
        else if ( ( d >= n2d && n2d == Node.DISTANCE_DEFAULT ) || d == Node.DISTANCE_DEFAULT
        || d == Node.DISTANCE_NULL ) {
            new_node.setDistanceToParent( d );
        }
        else {
            new_node.setDistanceToParent( d );
            node2.setDistanceToParent( n2d - d );
        }

        this_tree.setExtNodes( this_tree.getNumberOfExtNodes() + t.getNumberOfExtNodes() );
        this_tree.adjustExtNode0();
        this_tree.adjustReferencesInExtNodes();
        this_tree.adjustNodeCount();
        this_tree.recalculateAndReset();
        this_tree.reassignIDs();
        return this_tree;
    }



    /**

    Returns a Tree which is the result of fusing a rooted Tree to this Tree.

    @param id ID (int) of the Node of this Tree of whose parent branch
              to fuse to
    @param d  distance to parent (double) of newly introduced Node 
    @param t  Tree to fuse to this Tree
    @param n  the newly introduced Node

    */
    public Tree fuseTrees( int id, double d, Tree t, Node n )  throws Exception {
        return this.fuseTrees( id, d, t.getRoot().getDistanceToParent(), t, n );
    }



    /**

    Returns a Tree which is the result of fusing a rooted Tree to this Tree.

    @param id ID (int) of the Node of this Tree of whose parent branch
              to fuse to
    @param t  Tree to fuse to this Tree
    @param n  the newly introduced Node

    */
    public Tree fuseTrees( int id, Tree t, Node n )  throws Exception {
        return this.fuseTrees( id, Node.DISTANCE_DEFAULT,
        t.getRoot().getDistanceToParent(), t, n );
    }



    /**

    Returns a Tree which is the result of fusing a rooted Tree to this Tree.

    @param id ID (int) of the Node of this Tree of whose parent branch
              to fuse to
    @param d  distance to parent (double) of newly introduced Node
    @param r  distance to parent (double) of Tree to fuse to this Tree 
    @param t  Tree to fuse to this Tree

    */
    public Tree fuseTrees( int id, double d, double r, Tree t ) throws Exception {
        return this.fuseTrees( id, d, r, t, new Node( "" ) );
    }
    
    
    
    /**

    Returns a Tree which is the result of fusing a rooted Tree to this Tree.

    @param id ID (int) of the Node of this Tree of whose parent branch
              to fuse to
    @param d  distance to parent (double) of newly introduced Node
    @param t  Tree to fuse to this Tree

    */
    public Tree fuseTrees( int id, double d, Tree t ) throws Exception {
        return this.fuseTrees( id, d, t.getRoot().getDistanceToParent(), t, new Node( "" ) );
    }
    
    
    
    /**

    Returns a Tree which is the result of fusing a rooted Tree to this Tree.

    @param id ID (int) of the Node of this Tree of whose parent branch
              to fuse to
    @param t  Tree to fuse to this Tree

    */
    public Tree fuseTrees( int id, Tree t ) throws Exception {
        return this.fuseTrees( id, Node.DISTANCE_DEFAULT,
        t.getRoot().getDistanceToParent(), t, new Node( "" ) );
    }
    


    /**

    Returns an array of references to Trees which are the result of fusing
    Tree t2 to all possible branches of this Tree.

    @param t2 the Tree to fuse to all branches of this Tree

    */
    public Tree[] fuseToAllBranches( Tree t2 ) throws Exception {

        Tree tree1 = copyTree();
        Tree tree2 = t2.copyTree();
        double d = 0.0;
        int i = 0;
        Tree[] trees = new Tree[ tree1.getNumberOfBranches() ];
        Node node2, node1 = tree1.getExtNode0();

        tree1.setIndicatorsToZero();
        tree2.setIndicatorsToZero();

        if ( !tree1.isRooted() ) {
            tree1.getRoot().setIndicator( 1 );
            if ( !tree1.getRoot().isExternal()
            && !tree1.getRoot().getChild1().isPseudoNode()
            && !tree1.getRoot().getChild2().isPseudoNode() ) {
                tree1.getRoot().getChild1().setIndicator( 1 );
            }
        }

        while ( node1 != null ) {
            node2 = node1;
            while ( node2 != null ) {
                if ( node2.getIndicator() == 0 ) {
                    while ( !node2.isRoot()
                    && node2.isPseudoNode() ) {
                        node2.setIndicator( 1 );
                        node2 = node2.getParent();
                    }
                    if ( node2.getIndicator() == 0 ) {

                        d = node2.getDistanceToParent();
                        if ( d <= 0.0 && d != Node.DISTANCE_DEFAULT ) {
                            d = 0.0;
                        }
                        else if ( d != Node.DISTANCE_DEFAULT ) {
                            d /= 2.0;
                        }
                        node2.setIndicator( 1 );
                        trees[ i ] = tree1.fuseTrees( node2.getID(), d, tree2 );
                        i++;
                    }
                }
                node2 = node2.getParent();
            }
            node1 = node1.getNextExtNode();
        }

        return trees;
    }



    /**

    Returns the root Node of this Tree.

    */
    public Node getRoot() {
        return root;
    }
    
    
    
    void setRoot( Node n ) {
        root = n;
    }

    
    
    /**

    Returns true is this Tree is rooted.

    */
    public boolean isRooted() {
        return rooted;
    }
    
    
    
    /**

    Sets whether this Tree is rooted or not.

    */
    public void setRooted( boolean b ) {
        rooted = b;
    }
    
    
    
    private void setHighestLnL( double d ) {
        highest_lnL = d;
    }
    
    
    
    /**

    Returns the highest log likelihood value associated with branches
    of this Tree (double).
    
    @see #findHighestLnL()

    */
    public double getHighestLnL() {
        return highest_lnL;
    }
    
    
    
    private void setLowestLnL( double d ) {
        lowest_lnL = d;
    }
    
    
    
    /**
    
    Returns the lowest log likelihood value associated with branches
    of this Tree (double).

    @see #findLowestLnL()

    */
    public double getLowestLnL() {
        return lowest_lnL;
    }
    
    
    
    /**

    Returns the longest distance of this Tree (double).
    Used for the drawing of Trees.

    */
    public double getLongestDistance() {
        return longest_distance;
    }
    
    
    
    private void setLongestDistance( double d ) {
        longest_distance = d;
    }
    
    
    
    /**

    Returns the total number of branches of this Tree (int).
    
    @see #calculateNumberOfBranches()

    */
    public int getNumberOfBranches() {
        return number_of_branches;
    }
    
    
   
    private void setNumberOfBranches( int i ) {
        number_of_branches = i;
    }
    


    /**

    Returns the number of duplications of this Tree (int).
    A return value of -1 indicates that the number of duplications
    is unknown.

    */
    public int getNumberOfDuplications() {
        return number_of_duplications;
    }
    
    
    
    /**
    
    Sets the number of duplications of this Tree (int).
    A value of -1 indicates that the number of duplications
    is unknown.
    
    @param clean_nh set to true for clean NH format
    
    */
    public void setNumberOfDuplications( int i ) {
        if ( i < 0 ) {
            number_of_duplications = -1;    
        }
        else {
            number_of_duplications = i;
        }
    }
    
    
    
    /**

    Returns the first external Node.

    */
    public Node getExtNode0() {
        return ext_node0;
    }
    
    
    
    private void setExtNode0( Node n ) {
        ext_node0 = n;
    }
    
    
    
    /**

    Returns the maximum number of branches per external Node (int).
    Used for the drawing of Trees.
   
    */
    public int getMostBranchesPerExtNode() {
        return most_branches_per_ext_node;
    }
    
    
    
    private void setMostBranchesPerExtNode( int i ) {
        most_branches_per_ext_node = i;
    }
    
    
    
    /**

    Returns the sum of external Nodes of this Tree (int).
  
    */
    public int getNumberOfExtNodes() {
        return external_nodes;
    }
    
    
    
    private void setExtNodes( int i ) {
        external_nodes = i;
    }
    
    
    
    /**
    
    Returns whether to allow more than binary Nodes in New Hampshire (NH)
    output. The basal node can always appear trifurcated in the NH output.

    */
    private boolean allowMoreThanBinaryNodesInNHoutput() {
        return more_than_bin_nodes_in_NH;
    }


    
    
    /**

    Sets whether to allow more than binary Nodes in New Hampshire (NH)
    output. The basal node can always appear trifurcated in the NH output.

    */
    public void allowMoreThanBinaryNodesInNHoutput( boolean b ) {
        more_than_bin_nodes_in_NH = b;
    }



    /**
    
    Returns the name of this Tree.

    */
    public String getName() {
        return name;
    }


    
    
    /**

    Sets the name of this Tree to s.

    */
    public void setName( String s ) {
        name = s;
    }



    /**

    Prints descriptions of all external Nodes of this Tree to the console.

    */
    public void printExtNodes() {
        if ( isEmpty() ) { return; }
        Node node = getExtNode0();
        while ( node != null ) {
            System.out.println( node + "\n" );
            node = node.getNextExtNode();
        }
    }



    /**

    Prints descriptions of all Nodes of this Tree to the console.

    */
    public void printAllNodes() {
        if ( isEmpty() ) { return; }
        getRoot().preorderPrint();
    }



    /**

    Sets the indicators of all Nodes of this Tree to 0.

    */
    public void setIndicatorsToZero() {
        if ( isEmpty() ) {
            return;
        }
        getRoot().setIndicatorsToZero();
        
    }



    /**

    Calculatas the longest Distance of this Tree.
    The use of this method is for the drawing of Trees.
    The result can be obtained with "getLongestDistance()".

    */
    void calculateLongestDistance() {
        if ( isEmpty() ) {
            return;
        }
        double longest = 0.0;
        double sum     = 0.0;
        Node node2, node1 = getExtNode0();

        while ( node1 != null ) {
            node2 = node1;
            while ( node2 != null ) {
                if ( !node2.isRoot() && node2.getDistanceToParent() > 0 ) {
                    if ( node2.collapse() ) {
                        sum = 0.0;
                    }
                    sum += node2.getDistanceToParent();
                }
                node2 = node2.getParent();
            }
            node1 = node1.getNextExtNode();
            if ( sum > longest ) {
                longest = sum;
            }
            sum = 0;
        }
        if ( root.getDistanceToParent() > 0 ) {
            longest += root.getDistanceToParent();
        }
        setLongestDistance( longest );
    }



    /**

    Determines the most-branches-per-external-Node.
    The use of this method is for the drawing of Trees.
    The result can be obtained with "getMostBranchesPerExtNode()".

    */
    void calculateMostBranchesPerExtNode() {
        if ( isEmpty() ) { return; }
        int sum = 0;
        int most= 0;
        Node node2, node1 = getExtNode0();

        while ( node1 != null ) {
            node2 = node1;
            while ( node2 != null ) {
                if ( !node2.isRoot()
                && !node2.getParent().isPseudoNode()
                && !node2.isPseudoNode() ) {
                    sum = sum - node2.getSumExtNodes()
                    + node2.getParent().getSumExtNodes();
                }

                node2 = node2.getParent();
            }
            node1 = node1.getNextExtNode();
            if ( sum > most ) {
                most = sum;
            }
            sum = 0;
        }

        setMostBranchesPerExtNode( most );
    }



    /**

    Finds the lowest of all log likelihood value associated with
    branches of this Tree.
    The result can be obtained with "getLowestLnL()". 
    
    @see #getLowestLnL()

    */
    public void findLowestLnL() {
        if ( isEmpty() ) { return; }
        double lowest = getExtNode0().getLnLonParentBranch();
        PreorderTreeIterator i = null;
        try {
            i = new PreorderTreeIterator( this );
        }
        catch ( Exception e ) {
            e.printStackTrace();
            System.err.println( "Tree: Unexpected failure." );
            System.exit( -1 );
        }
        while ( !i.isDone() ) {
            if ( i.currentNode().isLnLonParentBranchAssigned()
            && !i.currentNode().isPseudoNode()
            && i.currentNode().getLnLonParentBranch() < lowest ) {
                lowest = i.currentNode().getLnLonParentBranch();
            }
            i.next();
        }
        setLowestLnL( lowest );
    }



    /**

    Finds the highest of all log likelihood value associated with
    branches of this Tree.
    The result can be obtained with "getHighestLnL()". 
    
    @see #getHighestLnL()

    */
    public void findHighestLnL() {
        if ( isEmpty() ) { return; }
        double highest = getExtNode0().getLnLonParentBranch();
        PreorderTreeIterator i = null;
        try {
            i = new PreorderTreeIterator( this );
        }
        catch ( Exception e ) {
            e.printStackTrace();
            System.err.println( "Tree: Unexpected failure." );
            System.exit( -1 );
        }
        while ( !i.isDone() ) {
            if ( i.currentNode().isLnLonParentBranchAssigned()
            && !i.currentNode().isPseudoNode()
            && i.currentNode().getLnLonParentBranch() > highest ) {
                highest = i.currentNode().getLnLonParentBranch();
            }
            i.next();
        }
        setHighestLnL( highest );
    }



    /**

    Calculates to number of branches of this Tree.
    The result can be obtained with "getNumberOfBranches()".
    E.g. (a,b): 1 branch (unrooted). (a,b)x: 3 branches (rooted).

    @see #getNumberOfBranches()

    */
    public void calculateNumberOfBranches() {
        if ( isEmpty() ) { return; }
        if ( isRooted() ) {
            setNumberOfBranches( 2 * getNumberOfExtNodes() - 1
        - countBranchesWithDISTANCE_NULL() );         }
        else {
            if ( !getRoot().isExternal()
            && !getRoot().getChild1().isPseudoNode()
            && !getRoot().getChild2().isPseudoNode() ) {
                setNumberOfBranches( 2 * getNumberOfExtNodes() - 3
                - countBranchesWithDISTANCE_NULL() );
            }
            else {
                setNumberOfBranches( 2 * getNumberOfExtNodes() - 2
                - countBranchesWithDISTANCE_NULL() );
            }
        }
    }



    /**
    
    Moves the values in the branch length field to the bootstrap field, for
    each Node of this Tree.
    Converts a Tree originating from a phylip treefile after bootstrapping
    and which therefore has its bootstrap values where the branch lenghts
    would be.
    
    */
    public void moveBranchLenghtsToBootstrap() throws Exception {
        if ( isEmpty() ) {
            String message = "Tree: moveBranchLenghtsToBootstrap(): ";
            message += "Tree must not be empty.";
            throw new Exception( message );
        }
        setIndicatorsToZero();
        Node node2, node1 = getExtNode0();
        while ( node1 != null ) {
            node2 = node1;
            while ( node2 != null ) {
                if ( node2.getIndicator() == 0 ) {
                    if ( node2.getDistanceToParent() > 0.0 ) {
                        if ( !node2.isExternal() ) {
                            node2.setBootstrap( ( int ) node2.getDistanceToParent() );
                        }
                        node2.setDistanceToParent( Node.DISTANCE_DEFAULT );
                    }
                    else {
                        node2.setBootstrap( Node.BOOTSTRAP_DEFAULT );
                    }
                    node2.setIndicator( 1 );
                }
                node2 = node2.getParent();
            }
            node1 = node1.getNextExtNode();
        }
        recalculateAndReset();
    }



    /**

    Checks whether the branch length values actually are bootstrap values
    All external Nodes must have the same, >0, divisible by 10 branch length.
    Boostrap of root must be assigned.
    
    */
    public boolean areBranchLenghtsBootstraps() {
        if ( isEmpty() ) {
            return false;
        }
        if ( getNumberOfExtNodes() <= 1 ) {
            return false;
        }
        // Has bootstrap already been assigned?
        if ( getRoot().getBootstrap() != Node.BOOTSTRAP_DEFAULT ) {
            return false;
        }
        Node node = getExtNode0();
        double d1 = 0.0, d2 = 0.0;
        while ( node != null ) {
            d1 = node.getDistanceToParent();
        if ( d1 <= 0.0 || ( d1 %  10 ) != 0 ) { return false; }
            node = node.getNextExtNode();
            if ( node != null ) {
                d2 = node.getDistanceToParent();
            if ( d2 <= 0.0 || ( d2 %  10 ) != 0 ) { return false; }
            if ( d1 != d2 ) { return false; }
            }
        }
        return true;
    }



    /**
  
    Converts this Tree to a New Hampshire X (String) representation.
    
    @return String
    
    @see #toNewHampshireX()
    
    */
    public String toString() {
        if ( isEmpty() ) { return ""; }
        return toNewHampshireX();
    }



    /**
  
    Converts this Tree to a New Hampshire (String) representation.
    If the boolean clean_nh is true, the length of sequence names
    will be truncated to MAX_LENGTH (usually 10) characters and 
    everything after a "/" will be removed (including the "/").

    @param clean_nh set to true for clean NH format

    @return String
    
    */
    public String toNewHampshire( boolean clean_nh ) {

        if ( isEmpty() ) { return ""; }
        int i = 0;
        String s = "", nh_string = "";
        Stack A_stack = new Stack();
        Stack B_stack = new Stack();
        Node node = root;
        
        setIndicatorsToZero();

        do {
            if ( !node.isExternal() ) {
                if ( node.getIndicator() == 0 ) {
                    node.setIndicator( 1 );
                    node = node.getChild1();
                }
                if ( node.getIndicator() == 1  ) {
                    node.setIndicator( 2 );
                    node = node.getChild2();
                }
                // Add internal node:
                if ( node.getIndicator() == 2 ) {

                    if ( !node.isPseudoNode()
                    || ( !allowMoreThanBinaryNodesInNHoutput()
                     && !node.getParent().isRoot()
                     && ( !(node.getParent().getChild1().isPseudoNode()  
                      && node.getParent().getChild2().isPseudoNode() )
                     || node.getParent().getChild1() == node ) ) ) {

                        i = 0;
                        while ( i < 2 * node.getSumExtNodes() - 2 ) {
                            B_stack.push( A_stack.pop() );
                            if ( !B_stack.peek().toString().equals( "(" )
                            && !B_stack.peek().toString().equals( ")" ) ) {
                                i++;
                            }
                        }
                        A_stack.push( "(" );
                        while ( !B_stack.empty() ) {
                            A_stack.push( B_stack.pop() );
                        }
                        A_stack.push( ")" );
                    }

                    if ( node.getDistanceToParent() >= 0.0
                    && !node.isPseudoNode() ) {
                        s = ":" + node.getDistanceToParent() + ",";
                    }
                    else if ( node.isPseudoNode()
                    && !allowMoreThanBinaryNodesInNHoutput()
                    && !node.getParent().isRoot()
                    && ( !(node.getParent().getChild1().isPseudoNode()
                     && node.getParent().getChild2().isPseudoNode() )
                     || node.getParent().getChild1() == node ) ) {
                        s = ":0.0,";
                    }
                    else {
                        s = ",";
                    }

                    A_stack.push( s );

                    node = node.getParent();
                }
            }
            // Add external node. Always preceeded by a "," which will need
            // to be removed in cases of "(,X"
            else {
                // If no seq_name is given, use species as name;
                // If no species is given, use EC as name
                // Otherwise, no information will be given.
                
                if ( clean_nh && !node.getSeqName().equals( "" ) ) {

                    String tmp = node.getSeqName();

                    try {
                        if ( tmp.length() > MAX_LENGTH ) {
                            tmp = tmp.substring( 0, MAX_LENGTH + 1 );
                        }
                        if ( tmp.indexOf( '/' ) > 0 ) {
                            tmp = tmp.substring( 0, tmp.indexOf( '/' ) );
                        }
                    }
                    catch ( Exception e ) {
                        System.err.println( "\ntoNewHampshire()" +
                        "Unexpected Exception.\n" );
                    }

                    s = "," + tmp;
                }
                else if ( !clean_nh && !node.getSeqName().equals( "" ) ) {
                    s = "," + node.getSeqName();
                }
                else if ( !clean_nh && !node.getSpecies().equals( "" ) ) {
                    s = "," + node.getSpecies();
                }
                else if ( !clean_nh && !node.getECnumber().equals( "" ) ) {
                    s = "," + node.getECnumber();
                }
                else {
                    s = ",\t";
                    // Introduces a tab to prevent comma from being removed.
                    // Tab is removed later.
                }
                if ( node.getDistanceToParent() >= 0.0 ) {
                    s += ( ":" + node.getDistanceToParent() );
                }
                A_stack.push( s );
                // Need to check whether parent is null in case the tree is only
                // one node:
                if ( !node.isRoot() ) {
                    node = node.getParent();
                }
                else {
                    node.setIndicator( 2 );
                }
            }
        } while ( !node.isRoot() || node.getIndicator() != 2 );

        //inverse:
        while ( !A_stack.empty() ) {
            B_stack.push( A_stack.pop() );
        }

        //make a nh string out of B_stack:
        nh_string = stackToString( B_stack );

        //In case tree is just one node -- return empty string:
        if ( nh_string.indexOf( "(" ) == -1 ) {
            return "";
        }

        return nh_string;
    }



    /**
  
    Converts this Tree to a New Hampshire X (String) representation.
    
    */
    public String toNewHampshireX() {
        if ( isEmpty() ) { return ""; }

        int i = 0;
        String s = "", nh_string = "";
        Stack A_stack = new Stack();
        Stack B_stack = new Stack();
        Node node = root;

        setIndicatorsToZero();

        do {
            if ( !node.isExternal() ) {
                if ( node.getIndicator() == 0 ) {
                    node.setIndicator( 1 );
                    node = node.getChild1();
                }
                if ( node.getIndicator() == 1 ) {
                    node.setIndicator( 2 );
                    node = node.getChild2();
                }
                // Add internal node:
                if ( node.getIndicator() == 2 ) {

                    if ( !node.isPseudoNode()
                    || ( !allowMoreThanBinaryNodesInNHoutput()
                    && !node.getParent().isRoot() )
                    && ( !(node.getParent().getChild1().isPseudoNode()  &&
                    node.getParent().getChild2().isPseudoNode() ) ||
                    node.getParent().getChild1() == node ) ) {
                        i = 0;
                        while ( i < 2 * node.getSumExtNodes() - 2 ) {

                            B_stack.push( A_stack.pop() );
                            if (    !B_stack.peek().toString().equals( "(" )
                            &&  !B_stack.peek().toString().equals( ")" ) ) {
                                i++;
                            }
                        }
                        A_stack.push( "(" );
                        while ( !B_stack.empty() ) {
                            A_stack.push( B_stack.pop() );
                        }
                        A_stack.push( ")" );
                    }

                    // If there is info about an internal node, a "," needs to be added.
                    // In cases of ",)" the comma will be removed later.

                    s = "";
                    if ( !node.isPseudoNode() ) {

                        s = node.toNewHampshireX();
                        
                    }
                    else if ( node.isPseudoNode()
                    && !allowMoreThanBinaryNodesInNHoutput()
                    && !node.getParent().isRoot()
                    && ( !(node.getParent().getChild1().isPseudoNode()  &&
                    node.getParent().getChild2().isPseudoNode() ) ||
                    node.getParent().getChild1() == node ) ) {
                        s = ":0.0";
                    }
                    // If s contains something, add a comma:
                    if ( s.length() >= 1 ) {
                        s += ",";
                    }
                    A_stack.push( s );
                    node = node.getParent();
                }
            }
            // Add external node. Always preceeded by a "," which will need
            // to be removed in cases of "(,X"
            else {
                s = ",";
                s += node.toNewHampshireX();
               
                A_stack.push( s );

                // Need to check whether parent is null in case the tree is only
                // one node:
                if ( !node.isRoot() ) {
                    node = node.getParent();
                }
                else {
                    node.setIndicator( 2 );
                }
            }


        } while ( !node.isRoot() || node.getIndicator() != 2 );

        //inverse:
        while ( !A_stack.empty() ) {
            B_stack.push( A_stack.pop() );
        }
        //make a nh string out of B_stack:
        nh_string = stackToString( B_stack );

        // If information is present about the root, add it after the last ")":
        if ( getNumberOfExtNodes() >= 2
        && ( !root.getSeqName().equals( "" ) || root.getDistanceToParent() != 0.0
        || !root.getSpecies().equals( "" ) || !root.getECnumber().equals( "" )
        || root.isLnLonParentBranchAssigned() || root.isDuplicationOrSpecAssigned()
        || root.getBootstrap() != 0 ) ) {

            //remove ; at the end
            nh_string = nh_string.substring( 0, nh_string.length() - 1 );

            nh_string += node.toNewHampshireX();
            
            nh_string += ";";
        }

        return nh_string;

    }



    /**
  
    Private helper method used by the "toNewHampshire..." methods.

    */
    private String stackToString( Stack stack ) {
        int i;
        String s, string;
        StringBuffer sb;
        if ( getNumberOfExtNodes() >= 2 ) {
            string = "(";
        }
        else {
            // No need to add a "(" if tree has just one node
            string = "";
        }
        while ( !stack.empty() ) {
            s =  stack.pop().toString();
            //add "," in between ")(":
            if (  s.startsWith( "(" ) && string.endsWith( ")" )  ) {
                string += ( "," + s );
            }
            //add "," in between "'not(,'(":
            else if (  s.startsWith( "(" ) && !string.endsWith( "(" )
            && !string.endsWith( ","  )  ) {
                string += ( "," + s );
            }
            //get rid of the "," in "(,X":
            else if (  s.startsWith( "," ) && string.endsWith( "(" )  ) {
                string += s.substring( 1 );
            }
            //get rid of the "," in "X,)":
            else if (  s.startsWith( ")" ) && string.endsWith( "," )  ) {
                string = string.substring( 0, string.length() - 1 );
                string += s;
            }
            //get rid of one comma in ",,":
            else if (  s.startsWith( "," ) && string.endsWith( "," )  ) {
                string += s.substring( 1 );
            }
            else {
                string += s;
            }
        }

        // Remove "," at end of nh_string:
        if ( string.endsWith( "," )  ) {
            string = string.substring( 0, string.length() - 1 );
        }
        if ( getNumberOfExtNodes() >=2 ) {
            string += ");";
        }
        else {
            // No need to add a "(" if tree has just one node
            string += ";";
        }
        // Remove "," at begining of nh_string (can only happen
        // if tree contains just one node):
        if ( string.startsWith( "," )  ) {
            string = string.substring( 1 );
        }


        // Remove tabs which were introduced to protect commas originating from
        // unnamed ext nodes from being removed:
        sb = new StringBuffer( string );

        for ( i = 0; i <= sb.length() - 1; i++ ) {
            if ( sb.charAt( i ) == '\t' ) {
                sb = new StringBuffer( sb.toString().substring( 0, i ) +
                sb.toString().substring( i + 1 ) );
                i--;
            }
        }

        string = sb.toString();
        return string;
    }



    /**

    Returns a Vector containing the Node IDs of all Nodes which are between
    two external Nodes (node2 and node1).
    The order is so that element 0 is node2, and the last element is
    node1.
    Even if the Tree is rooted, this method does not return the root
    as part of the pathway (not ideal).
    The boolean ret_pseudo_nodes determines whether or not to return
    pseudo Nodes.
    (true: return pseudo nodes, false: only return real nodes.)
    
    */
    public Vector getPath( Node node1, Node node2, boolean ret_pseudo_nodes )
    throws Exception {

        if ( isEmpty() ) {
            return null;
        }

        if ( !node1.isExternal() || !node2.isExternal() ) {
            String message = "Tree: getPath( Node, Node ): ";
            message += "Nodes must be external.";
            throw new Exception( message );
        }

        Vector nodes = new Vector( 50, 50 );
        Tree tree = copyTree();

        if ( getNumberOfExtNodes() < 2 || node1 == node2 ) {
            nodes.addElement( new Integer( node1.getID() ) );
            return nodes;
        }

        try {
            tree.reRoot( node1.getID() );
            tree.adjustNodeCount();
            tree.recalculateAndReset();
        }
        catch ( Exception e ) {
            throw new Exception( "Tree: getPath( Node, Node ): " + e );
        }

        node1 = tree.getNode( node1.getID() );
        node2 = tree.getNode( node2.getID() );

        while ( !node2.isRoot() ) {
            if ( ret_pseudo_nodes || !getNode( node2.getID() ).isPseudoNode() ) {
                nodes.addElement( new Integer( node2.getID() ) );
            }
            node2 = node2.getParent();
        }

        nodes.addElement( new Integer( node1.getID() ) );

        nodes.trimToSize();

        return nodes;

    }



    /**

    Returns a Vector of references to all external siblings
    and nieces of a external Node n.
    The order is the same as in the Tree.

    @return Vector of references to Nodes

    */
    public Vector getSiblings( Node n ) throws Exception {

        if ( isEmpty() ) {
            String message = "Tree: getSiblings( Node ): ";
            message += "Tree must not be empty.";
            throw new Exception( message );
        }

        if ( !n.isExternal() ) {
            String message = "Tree: getSiblings( Node ): ";
            message += "This method only works for external nodes.";
            throw new Exception( message );
        }

        if ( n.isRoot() ) {
            String message = "Tree: getSiblings( Node ): ";
            message += "Root has no sibling.";
            throw new Exception( message );
        }

        if ( n.isPseudoNode() ) {
            String message = "Tree: getSiblings( Node ): ";
            message += "Attempt to get sibling of a pseudonode.";
            throw new Exception( message );
        }


        Node node = n;
        Vector nodes;


        do {
            node = node.getParent();
        } while ( node.isPseudoNode() );

        nodes = node.getAllExternalChildren();
        // Since this methods guarantees that the order is not changed,
        // getAllExternalChildren() needs to return the Nodes in the same order
        // as in the Tree.

        // remove the original node n out of the vector:
        if ( !nodes.removeElement( n ) ) {
            String message = "Tree: getSiblings( Node ): ";
            message += "Unexpected failure.";
            throw new Exception( message );
        }

        nodes.trimToSize();

        return nodes;

    }



    /**

    Finds the last common ancestor Node of two Nodes specified by
    their sequence names seqname1 and seqname2. This method obviously
    can produce a meaningful result only, if this Tree is correctly
    rooted.

    */
    public Node getLastCommonAncestor( String seqname1, String seqname2 )
    throws Exception {

        if ( isEmpty() ) {
            String message = "Tree: getLastCommonAncestor( String, String ): ";
            message += "Tree must not be empty.";
            throw new Exception( message );
        }

        seqname1 = seqname1.trim();
        seqname2 = seqname2.trim();

        Vector nodes1 = getNodes( seqname1 ),
        nodes2 = getNodes( seqname2 );

        // Check whether names are unique and present:
        if ( nodes1 == null || nodes2 == null ) {
            String message = "Tree: getLastCommonAncestor( String, String ): ";
            message += "Node(s) which matching seq name(s) not found";
            throw new Exception( message );
        }
        if ( nodes1.size() != 1 || nodes2.size() != 1 ) {
            String message = "Tree: getLastCommonAncestor( String, String ): ";
            message += "Seq name(s) is (are) not unique.";
            throw new Exception( message );
        }


        int id1 = ( ( Node ) nodes1.firstElement() ).getID();
        int id2 = ( ( Node ) nodes2.firstElement() ).getID();

        return getLastCommonAncestor( id1, id2 );

    }



    /**

    Finds the last common ancestor Node of two Nodes specified by
    their IDs id1 and id2. This method obviously
    can produce a meaningful result only, if this Tree is correctly
    rooted.

    */
    public Node getLastCommonAncestor( int id1, int id2 )
    throws Exception {

        if ( isEmpty() ) {
            String message = "Tree: getLastCommonAncestor( int, int ): ";
            message += "Tree must not be empty.";
            throw new Exception( message );
        }
       
        Node node1 ,
        node2_,
        node2 = null;

        try {
            node1  = getNode( id1 );
            node2_ = getNode( id2 );
        }
        catch ( Exception e ) {
            String message = "Tree: getLastCommonAncestor( int, int ): ";
            message += "Node(s) not found.";
            throw new Exception( message );
        }


        // Find the "first common parent node":
        while ( !node1.isRoot() ) {

            node2 = node2_;

            while ( !node2.isRoot() ) {
                if ( node2 == node1 && !node2.isPseudoNode() ) {
                    return node2;
                }
                node2 = node2.getParent();
            }
            node1 = node1.getParent();

        }
        // In case the "first common parent node" is the root:
        if ( node2 == node1 && !node2.isPseudoNode() ) {
            return node2;
        }

        // If for some reason not found:
        throw new Exception ( "getLastCommonAncestor: Unexpected error." );
    }




    /**

    Adjusts the references (previous and next) in the
    external Nodes of this Tree.

    */
    private void adjustReferencesInExtNodes() {
        if ( isEmpty() ) { return; }

        // getIndicator() must be 0 for all nodes:
        getRoot().setIndicatorsToZero();

        Node node      = getRoot();
        Node prev_node = null;

        if ( getNumberOfExtNodes() <= 1 ) {
            return;
        }

        do {

            if ( node.getIndicator() == 0 && !node.isExternal() ) {
                node.setIndicator( 1 );
                node = node.getChild1();
            }

            if ( node.getIndicator() == 1 && !node.isExternal() ) {
                node.setIndicator( 2 );
                node = node.getChild2();
            }

            if ( node.isExternal() ) {

                node.setNextExtNode( null );
                node.setPrevExtNode( prev_node );
                if ( prev_node != null ) {
                    prev_node.setNextExtNode( node );
                }

                prev_node = node;
                node.setIndicator( 2 );

            }

            // Moving towards the root:
            if ( node.getIndicator() == 2 ) {
                node = node.getParent();
            }

        } while ( !node.isRoot() || node.getIndicator() != 2 );
    }



    /**
  
    Adjusts the "first" external Node.
   
    */
    private void adjustExtNode0() {
        Node node = getRoot();
        if ( node == null ) { return; }
        while ( !node.isExternal() ) {
            node = node.getChild1();
        }
        setExtNode0( node );
    }



    /**
  
    (Re)counts the number of children for each Node of this Tree.
    As an example, this method needs to be called after a Tree has
    been reRooted.

    */
    public void adjustNodeCount() {
        if ( isEmpty() ) { return; }

        Node node1, node2 = getExtNode0();

        //Set all SumExtNodes to 0:
        do {
            node1 = node2;
            do {
                node1.setSumExtNodes( 0 );
                node1 = node1.getParent();
            } while ( node1 != null );
            node2 = node2.getNextExtNode();
        } while( node2 != null );

        //Set all to the new, correct values:
        node2 = ext_node0;
        O: do {
            node1 = node2;
            do {
                if ( node1.collapse() ) {
                    if ( node1.getIndicator() != -1234 ) {
                        node1.setIndicator( -1234 );  
                    }
                    else {
                        node2 = node2.getNextExtNode();
                        continue O;
                    }   
                }
                node1.setSumExtNodes( node1.getSumExtNodes() + 1 );
                node1 = node1.getParent();
            } while ( node1 != null );
            node2 = node2.getNextExtNode();
        } while( node2 != null );
    }



    /**
  
    Reassigns the IDs of the Nodes of this Tree.
   
    */
    private void reassignIDs() throws Exception {
        if ( isEmpty() ) { return; }

        PreorderTreeIterator i = null;
        try {
            i = new PreorderTreeIterator( this );
        }
        catch ( Exception e ) {
            e.printStackTrace();
            System.err.println( "Tree: Unexpected failure." );
            System.exit( -1 );
        }
        while ( !i.isDone() ) {
            Node.setNodeCount( Node.getNodeCount() + 1 );
            i.currentNode().setID( Node.getNodeCount() );
            i.next();
        }
    }



    /**

    Used for building Trees.

    */
    private void addNode( String s ) throws Exception {
        if ( getExtNode0() == null ) {
            setExtNode0( new Node( s ) );
        }
        else {
            getExtNode0().addExtNode( s );
        }
        external_nodes++;
    }



    /**

    Used for building Trees.

    */
    void addNodeAndConnect( String s1, String s2 ) throws Exception {
        if ( getExtNode0() == null ) {
            throw new Exception( "addNodeAndConnect( String, String ): Error: Cannot add and connect one node to empty tree." );
        }
        getExtNode0().addExtNode( s1 );
        getExtNode0().connect( s2 );
        external_nodes++;
    }



    /**

    Used for building Trees.

    */
    private void connectInternal( String s ) throws Exception {
        if ( isEmpty() ) { return; }
        getExtNode0().connect( s );
    }



    /**

    Counts all branches which have DISTANCE_NULL,
    and which therefore lead to pseudo Nodes.

    */
    private int countBranchesWithDISTANCE_NULL() {
        if ( isEmpty() ) { return -1; }
        int count = 0;
        PreorderTreeIterator i = null;
        try {
            i = new PreorderTreeIterator( this );
        }
        catch ( Exception e ) {
            e.printStackTrace();
            System.err.println( "Tree: Unexpected failure." );
            System.exit( -1 );
        }
        while ( !i.isDone() ) {
            if ( i.currentNode().isPseudoNode() ) {
                count++;
            }
            i.next();
        }
        return count;
    }



    /**

    Resets the ID values of the Nodes of this Tree in
    preorder, starting with i.
    
    @param i the starting value (int)

    @return i plus the total number of Nodes of this Tree (int)
    
    */
    public int preorderReID( int i ) {

        if ( isEmpty() ) { 
            return i;
        }

        Stack stack = new Stack();
        Node n = getRoot();
        stack.push( n );
        while ( !stack.empty() ) {
            n = ( Node ) stack.pop();
            n.setID( i++ ); 
            if ( n.getChild2() != null ) {
                stack.push( n.getChild2() );
            }
            if ( n.getChild1() != null ) {
                stack.push( n.getChild1() );
            }
            
        }
        return i;
    }



    /**

    Hashes the ID number of each Node of this Tree to its corresonding
    Node, in order to make method getNode( id ) run in constant time.
    Important: The user is responsible for calling this method (again) after
    this Tree has been changed/created/renumbered. 

    */
    public void hashIDs() {
 
        idhash = null;

        if ( isEmpty() ) {
            return;
        }
             
        //idhash = new HashMap(); // Requires JDK 1.2 or greater.
        idhash = new Hashtable();  // For JDK 1.1.
        // Setting the initial capacity does not result in improved time efficiency.

        Stack stack = new Stack();
        Node n = getRoot();
        stack.push( n );
        while ( !stack.empty() ) {
            n = ( Node ) stack.pop(); 
            // ID is the key, reference to Node is the value.
            idhash.put( new Integer( n.getID() ), n );
            if ( n.getChild1() != null ) {
                stack.push( n.getChild1() );
            }
            if ( n.getChild2() != null ) {
                stack.push( n.getChild2() );
            }
        }
    }

} // End of class Tree.

